//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GL_MASK_EFFECT_H)
    #define CANDERA_GL_MASK_EFFECT_H

#include <Candera/Engine2D/Effects/InPlaceEffect2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/Engine2D/Property/Property.h>
#include <Candera/Engine2D/Property/ColorProperty.h>
#include <Candera/Engine2D/Property/Image2DProperty.h>
#include <Candera/Engine2D/Property/Node2DProperty.h>
#include <CanderaPlatform/Device/Common/Effects/EffectDataTypes.h>
#include <CanderaPlatform/Device/Common/Effects/MaskProperties.h>

namespace Candera {

    /** 
     *  @brief Apply an alpha mask to an image.
     *
     *  There are no other applied transformations.
     *  The position of the mask is controlled through a node, that can be either
     *  the current node, or some other specified node. The mask can be independently rotated or scaled.
     *  Only the alpha channel of the mask is used and multiplied with the alpha
     *  channel of the source image.
     */
    class GlMaskEffect : public InPlaceEffect2D {

        FEATSTD_TYPEDEF_BASE(InPlaceEffect2D);

    public:
        typedef MemoryManagement::SharedPointer<GlMaskEffect> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class
         *  @return A pointer to the created object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~GlMaskEffect();

        /**
         *  Activate the effect on a 2D context.
         *  @param node   Node 2d
         *  @param output The effect is activated on this 2D context.
         */
        void ActivateMask(const Node2D& node, ContextHandle2D output);

        /**
         *  Deactivate the effect.
         *  @param output The effect is deactivated for this 2D context.
         */
        void DeactivateMask(ContextHandle2D output) const;

        /**
         *  Uploads all associated resources.
         *  @return true if successful.
         */
        virtual bool Upload();

        /**
         *  Unloads all associated resources.
         *  @return true if successful.
         */
        virtual bool Unload();

        /**
        *   Check whether all the resources are uploaded.
        *   @return true if all the resources are uploaded, false otherwise.
        */
        virtual bool IsUploaded() const { return m_uploaded; }

        /**
         *  Uploads all resources that changed since last Upload/Update.
         *  Unloads resources removed from the effect, that were previously uploaded
         *  by an Upload/Update.
         *  This should only be called on Uploaded effects. (otherwise, call Upload.)
         *  @return true if successful.
         */
        virtual bool Update();


        /**
         *  Retrieves the current mask node. This returned property
         *  can be changed. The performed changes will affect this class.
         *  Property used to control the transformation of the mask. 
         *  If set to 0 the transformation of the current node is used.
         *  @return     The mask node property.
         */
        Node2DProperty& MaskNode() { return m_maskNode; }

        /**
         *  Retrieves the current mask. This returned property
         *  can be changed. The performed changes will affect this class.
         *  Property used as mask image. Only the alpha channel of the mask is used.
         *  @return     The currently set mask image.
         */
        Image2DProperty& Mask() { return m_mask; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /// @cond excluded from doxygen
        CdaEffect2DDef(Candera::GlMaskEffect, GlMaskEffect, EFFECT2D_TYPE_INPLACE)
            CdaEffect2DProperties()

                CdaEffect2DProperty(MaskNode, Node2D*, m_maskNode)
                    CdaDescription("Node used to control the transformation of the mask. "
                        "If set to 0 the transformation of the current node is used.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(Mask, MemoryManagement::SharedPointer<Image2D>, m_mask)
                    CdaDescription("The image used as mask. Only the alpha channel of the mask is used.")
                    CdaUpdateType(OnApplyUpdateType)
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        GlMaskEffect();
        GlMaskEffect(const GlMaskEffect& rhs);
        GlMaskEffect& operator = (const GlMaskEffect& rhs);

        /**
         *  Triggers the rendering for the effect.
         *  @param  input       Defines the input for the effect. The first effect inside the effect list NO input. The input for the second
         *                      effect in the list must be set to the output of the first effect and so on.
         *  @param  inputArea   Area of the input that should be used as input. This should by supplied through the ouputArea parameter of the last effect.
         *  @param  transform   Defines the transformation matrix for the effect.
         *  @param  node        Node this effect is referenced to, containing transformation matrix for the effect.
         *  @param  output      Defines the output for the effect. The last effect inside the effect list must be the context of the render target.
         *  @param  outputArea  Output parameter through which the effect supplies the area of the output that was modified.
         */
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);



    private:
        bool m_uploaded;

        Node2DProperty m_maskNode;
        Image2DProperty m_mask;

        MemoryManagement::SharedPointer<Image2D> m_uploadedMask;
    };



}   // namespace Candera

#endif  // CANDERA_GL_MASK_EFFECT_H
