//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(CANDERA_BLUR_EFFECT_H)
    #define CANDERA_BLUR_EFFECT_H

#include <Candera/Engine2D/Effects/InPlaceEffect2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/Engine2D/Property/Vector2Property.h>
#include <Candera/Engine2D/Property/Property.h>
#include <Candera/System/Mathematics/MathDataTypes.h>
#include <CanderaPlatform/Device/Common/Effects/EffectDataTypes.h>

namespace Candera {

    /**
     *  @brief Blurs the output of the predecessor effect by applying a gaussean filter convolution.
     *         When blurring the image is blitted n + m times, where n and m are the dimensions
     *         of the gaussean filter kernel. On each of these rendered image a fixed alpha value,
     *         defined by the precalculated gaussean filter kernel, is applied. The
     *         filter kernel is calculated whenever the dimensions of the kernel change.
     *         The bigger the gaussean kernel is, the more the image is blurred. Of course,
     *         a bigger filter kernel results in decreased performance.
     */
    class BlurEffect : public InPlaceEffect2D {

        FEATSTD_TYPEDEF_BASE(InPlaceEffect2D);

    public:
        typedef MemoryManagement::SharedPointer<BlurEffect> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class.
         *  @return Pointer to the created BlurEffect object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~BlurEffect();

        /**
         *  Retrieves the current filter size.
         *  @return The current filter size.
         */
        UInt8Property& FilterSize() { return m_filterSize; }

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /// @cond excluded from doxygen
        CdaEffect2DDef(Candera::BlurEffect, BlurEffect, EFFECT2D_TYPE_INPLACE)
            CdaEffect2DProperties()

                CdaEffect2DProperty(FilterSize, UInt8, m_filterSize)
                    CdaDescription("Quadratic Size of filter kernel.")
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        BlurEffect();
        BlurEffect(const BlurEffect& rhs);
        BlurEffect& operator = (const BlurEffect& rhs);

    private:
        UInt8Property m_filterSize;
        Int32 m_center;
        Float* m_filterKernel;
        UInt8 m_lastFilterSize;

        bool IsFilterKernelValid();
        void UpdateFilterKernel();
    };

}   // namespace Candera

#endif  // CANDERA_BLUR_EFFECT_H
