//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BlurEffect.h"
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>

namespace Candera {

    FEATSTD_RTTI_DEFINITION(BlurEffect, InPlaceEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    BlurEffect::BlurEffect() :
        m_filterSize(1),
        m_center(0),
        m_filterKernel(0),
        m_lastFilterSize(0)
    {
    }

    BlurEffect::BlurEffect(const BlurEffect& rhs) :
        Base(rhs),
        m_filterSize(rhs.m_filterSize.Get()),
        m_center(rhs.m_center),
        m_filterKernel(0),
        m_lastFilterSize(0)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    BlurEffect::~BlurEffect()
    {
        FEATSTD_DELETE_ARRAY(m_filterKernel);
            m_filterKernel = 0;
        }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    BlurEffect::SharedPointer BlurEffect::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        BlurEffect* brush = FEATSTD_NEW(BlurEffect);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        BlurEffect::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void BlurEffect::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node,
                            ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::RenderEffect2D, "BlurEffect"));

        if (!IsFilterKernelValid()) {
            UpdateFilterKernel();
        }

        static_cast<void>(RenderDevice2D::SetSurface(output, RenderDevice2D::SourceSurface, input));

        Camera2D* cam = Renderer2D::GetActiveCamera();
        Matrix3x2 worldTransform = node.GetWorldTransform();
        Matrix3x2 camTransform;
        if (cam != 0) {
            camTransform = cam->GetViewMatrix();
            const Vector2& cameraTranslationOffset = Renderer2D::GetCameraTranslationOffset();
            camTransform.Translate(cameraTranslationOffset.GetX(), cameraTranslationOffset.GetY());
        }
        outputArea = inputArea;

        Matrix3x2 blurTransform;
        blurTransform.SetIdentity();

        static_cast<void>(RenderDevice2D::SetActiveArea(output, RenderDevice2D::SourceSurface, 0.0F, 0.0F, -1.0F, -1.0F));

        static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, transform));
        static_cast<void>(RenderDevice2D::SetActiveArea(output,
            RenderDevice2D::SourceSurface,
            inputArea.GetLeft(),
            inputArea.GetTop(),
            inputArea.GetWidth(),
            inputArea.GetHeight()));
        static_cast<void>(Renderer2D::Blit(output));//First draw fully opaque.

        static_cast<void>(RenderDevice2D::SetAlphaBlendOperation(output, RenderDevice2D::SourceSurface, RenderDevice2D::Add, RenderDevice2D::SourceAlpha, RenderDevice2D::One));
        if (m_filterSize > 0) {
            Float invScaleX = node.GetScale().GetX();
            Float invScaleY = node.GetScale().GetY();
            if (invScaleX != 0.0F) {
                invScaleX = 1.0F / invScaleX;
            }
            if (invScaleY != 0.0F) {
                invScaleY = 1.0F / invScaleY;
            }

            //Gauss filter for 2d is seperable in two 1D filters for x and y, resulting in m + n iterations instead of m * n.
            for (Int32 i = 0; i < m_filterSize; i++) {
                if( i != m_center) {
                    Float offset = static_cast<Float>(i) - static_cast<Float>(m_center);
                    Float absOffset = Math::Absolute(offset);

                    static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.0F, 1.0F, 1.0F, m_filterKernel[i]));
                    blurTransform.SetTranslation(offset, 0.0F);
                    blurTransform = worldTransform * blurTransform;
                    static_cast<void>(RenderDevice2D::SetActiveArea(output,
                        RenderDevice2D::SourceSurface,
                        inputArea.GetLeft() - (offset * invScaleX),
                        inputArea.GetTop(),
                        inputArea.GetWidth() - (absOffset * invScaleX),
                        inputArea.GetHeight()));
                    static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, blurTransform * camTransform));
                    static_cast<void>(Renderer2D::Blit(output));
                    static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, transform));

                    static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.0F, 1.0F, 1.0F, m_filterKernel[i]));
                    blurTransform.SetTranslation(0.0F, offset);
                    blurTransform = worldTransform * blurTransform;
                    static_cast<void>(RenderDevice2D::SetActiveArea(output,
                        RenderDevice2D::SourceSurface,
                        inputArea.GetLeft(),
                        inputArea.GetTop() - (offset * invScaleY),
                        inputArea.GetWidth(),
                        inputArea.GetHeight() - (absOffset * invScaleY)));
                    static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, blurTransform * camTransform));
                    static_cast<void>(Renderer2D::Blit(output));
                    static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, transform));

                    static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.0F, 1.0F, 1.0F, m_filterKernel[i]));
                    blurTransform.SetTranslation(offset, offset);
                    blurTransform = worldTransform * blurTransform;
                    static_cast<void>(RenderDevice2D::SetActiveArea(output,
                        RenderDevice2D::SourceSurface,
                        inputArea.GetLeft() - (offset * invScaleX),
                        inputArea.GetTop() - (offset * invScaleY),
                        inputArea.GetWidth() - (absOffset * invScaleX),
                        inputArea.GetHeight() - (absOffset * invScaleY)));
                    static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, blurTransform * camTransform));
                    static_cast<void>(Renderer2D::Blit(output));
                    static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, transform));

                    static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.0F, 1.0F, 1.0F, m_filterKernel[i]));
                    blurTransform.SetTranslation(- offset, offset);
                    blurTransform = worldTransform * blurTransform;
                    static_cast<void>(RenderDevice2D::SetActiveArea(output,
                        RenderDevice2D::SourceSurface,
                        inputArea.GetLeft() + (offset * invScaleX),
                        inputArea.GetTop() - (offset * invScaleY),
                        inputArea.GetWidth() - (absOffset * invScaleX),
                        inputArea.GetHeight() - (absOffset * invScaleY)));
                    static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, blurTransform * camTransform));
                    static_cast<void>(Renderer2D::Blit(output));
                    static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, transform));
                }
            }

            static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.0F, 1.0F, 1.0F, 1.0F));
        }

        //Reset Active area to input area.
        static_cast<void>(RenderDevice2D::SetActiveArea(output,
            RenderDevice2D::SourceSurface,
            inputArea.GetLeft(),
            inputArea.GetTop(),
            inputArea.GetWidth(),
            inputArea.GetHeight()));

        static_cast<void>(RenderDevice2D::SetAlphaBlendOperation(output, 
            RenderDevice2D::SourceSurface, RenderDevice2D::Add, RenderDevice2D::One, RenderDevice2D::Zero));
    }

    /******************************************************************************
     *  IsFilterKernelValid
     ******************************************************************************/
    bool BlurEffect::IsFilterKernelValid()
    {
        bool valid = true;

        if ((m_lastFilterSize != m_filterSize)) {
            valid = false;
        }

        return valid;
    }

     /******************************************************************************
     *  UpdateFilterKernel
     ******************************************************************************/
    void BlurEffect::UpdateFilterKernel()
    {
        if(m_filterKernel != 0) {
            FEATSTD_DELETE_ARRAY(m_filterKernel);
        }

        UInt8 filterSize = m_filterSize;

        m_lastFilterSize = filterSize;
        m_filterKernel = FEATSTD_NEW_ARRAY(Float, filterSize);

        if (0 != m_filterKernel) {
            UInt dimension = filterSize;

            m_center = static_cast<Int32>(dimension >> 1);
            if ((dimension & 1U) == 0) {
                m_center--;
            }

            Float sigma = (static_cast<Float>(dimension)) / 3.0F;

            if(sigma != 0.0F) {
                Float gaussPrefix = 1.0F / (sigma * Math::SquareRoot(2.0F * Math::Pi()));
                Float sigma2 = 2.0F * (sigma * sigma);

                for (Int32 i = 0; i < filterSize; i++) {
                    Float x = Math::Absolute(static_cast<Float>(m_center - i));
                    m_filterKernel[i] = gaussPrefix * Math::Power(Math::EulerConstant(), - ((x * x)/sigma2));
                }
            }
        }
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer BlurEffect::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(BlurEffect)(*this));
    }

}   // namespace Candera

