//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BlurBitmapBrushBlend.h"
#include <CanderaPlatform/Device/Common/Effects/BlurEffect.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include "Candera/Engine2D/Core/Mesh2D.h"

namespace Candera {
    FEATSTD_RTTI_DEFINITION(BlurBitmapBrushBlend, CombinedEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    BlurBitmapBrushBlend::BlurBitmapBrushBlend()
    {
    }

    BlurBitmapBrushBlend::BlurBitmapBrushBlend(const BlurBitmapBrushBlend& rhs) :
        Base(rhs),
        m_bitmapBrush(rhs.m_bitmapBrush),
        m_blurEffect(rhs.m_blurEffect),
        m_blendEffect(rhs.m_blendEffect)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    BlurBitmapBrushBlend::~BlurBitmapBrushBlend()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    BlurBitmapBrushBlend::SharedPointer BlurBitmapBrushBlend::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        BlurBitmapBrushBlend* brush = FEATSTD_NEW(BlurBitmapBrushBlend);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        BlurBitmapBrushBlend::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void BlurBitmapBrushBlend::Render(SurfaceHandle /*input*/, const Rectangle& /*inputArea*/, const Matrix3x2& transform,
                                           const Node2D& node, ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::RenderEffect2D, "BlurBitmapBrushBlend"));

        Image2D* image = m_bitmapBrush.Image();

        if (image != 0) {
            m_blendEffect.ActivateBlend(output);
            Rectangle rect;
            m_bitmapBrush.GetBoundingRectangle(rect);

            if (node.IsTypeOf(Mesh2D::GetTypeId())) {
                const Mesh2D& mesh = static_cast<const Mesh2D&>(node);
                if (!(mesh.GetVertexBuffer2D().PointsToNull())) {
                    rect = mesh.GetVertexBuffer2D()->GetBoundingRectangle();
                }
            }

            static_cast<void>(image->Activate(output));
            m_blurEffect.Render(image->Get2DSurfaceHandle(), rect, transform, node, output, outputArea);
            m_blendEffect.DeactivateBlend(output);
        }
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool BlurBitmapBrushBlend::Upload()
    {
        bool success = m_bitmapBrush.Upload();
        success = m_blurEffect.Upload() && success;
        success = m_blendEffect.Upload() && success;
        return success;
    }
    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool BlurBitmapBrushBlend::Unload()
    {
        bool success = m_bitmapBrush.Unload();
        success = m_blurEffect.Unload() && success;
        success = m_blendEffect.Unload() && success;
        return success;
    }
    /******************************************************************************
     *  Update
     ******************************************************************************/
    bool BlurBitmapBrushBlend::Update()
    {
        bool success = m_bitmapBrush.Update();
        success = m_blurEffect.Update() && success;
        success = m_blendEffect.Update() && success;
        return success;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer BlurBitmapBrushBlend::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(BlurBitmapBrushBlend)(*this));
    }

}   // namespace Candera
