//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BlendEffect.h"
#include <Candera/Engine2D/Core/Image2D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>

namespace Candera {

    FEATSTD_RTTI_DEFINITION(BlendEffect, BlendEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    BlendEffect::BlendEffect() :
        Base(),
        m_colorBlendFactorSrc(RenderDevice2D::SourceAlpha),
        m_colorBlendFactorDst(RenderDevice2D::InverseSourceAlpha),
        m_colorBlendOperation(RenderDevice2D::Add),
        m_alphaBlendFactorSrc(RenderDevice2D::One),
        m_alphaBlendFactorDst(RenderDevice2D::Zero),
        m_alphaBlendOperation(RenderDevice2D::Add)
    {
    }

    BlendEffect::BlendEffect(const BlendEffect& rhs) :
        Base(rhs),
        m_colorBlendFactorSrc(rhs.m_colorBlendFactorSrc.Get()),
        m_colorBlendFactorDst(rhs.m_colorBlendFactorDst.Get()),
        m_colorBlendOperation(rhs.m_colorBlendOperation.Get()),
        m_alphaBlendFactorSrc(rhs.m_alphaBlendFactorSrc.Get()),
        m_alphaBlendFactorDst(rhs.m_alphaBlendFactorDst.Get()),
        m_alphaBlendOperation(rhs.m_alphaBlendOperation.Get())
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    BlendEffect::~BlendEffect()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    BlendEffect::SharedPointer BlendEffect::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        BlendEffect* brush = FEATSTD_NEW(BlendEffect);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        BlendEffect::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void BlendEffect::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& /*node*/, ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "BlendEffect"));

        bool result = Renderer2D::SetTransformationMatrix(
            output,
            RenderDevice2D::SourceSurface,
            transform);

        result = result && RenderDevice2D::SetSurface(
            output,
            RenderDevice2D::SourceSurface,
            input);

        result = result && RenderDevice2D::SetActiveArea(
            output,
            RenderDevice2D::SourceSurface,
            inputArea.GetLeft(),
            inputArea.GetTop(),
            inputArea.GetWidth(),
            inputArea.GetHeight());

        if (result) {
            ActivateBlend(output);
            result = Renderer2D::Blit(output);
            DeactivateBlend(output);
        }

        if (result) {
            Float left;
            Float top;
            Float width;
            Float height;
            static_cast<void>(RenderDevice2D::GetUpdatedArea(output, &left, &top, &width, &height));
            outputArea = Rectangle(left, top, width, height);
        }
    }

    /******************************************************************************
     *  SetBlendOperation
     ******************************************************************************/
    void BlendEffect::ActivateBlend(ContextHandle2D output) const
    {
        static_cast<void>(RenderDevice2D::SetColorBlendOperation(output, RenderDevice2D::SourceSurface, m_colorBlendOperation, m_colorBlendFactorSrc, m_colorBlendFactorDst));
        static_cast<void>(RenderDevice2D::SetAlphaBlendOperation(output, RenderDevice2D::SourceSurface, m_alphaBlendOperation, m_alphaBlendFactorSrc, m_alphaBlendFactorDst));
    }

    /******************************************************************************
     *  ResetBlendOperation
     ******************************************************************************/
    void BlendEffect::DeactivateBlend(ContextHandle2D output) const
    {
        static_cast<void>(RenderDevice2D::SetBlendOperation(output, RenderDevice2D::SourceSurface, RenderDevice2D::Add, RenderDevice2D::One, RenderDevice2D::Zero));
    }

    /******************************************************************************
    *  SetBlendMode
    ******************************************************************************/
    void BlendEffect::SetBlendMode(RenderDevice2D::BlendFactor blendFactorSrc, RenderDevice2D::BlendFactor blendFactorDst, RenderDevice2D::BlendOperation blendOperation)
    {
        m_colorBlendFactorSrc = blendFactorSrc;
        m_alphaBlendFactorSrc = blendFactorSrc;

        m_colorBlendFactorDst = blendFactorDst;
        m_alphaBlendFactorDst = blendFactorDst;

        m_colorBlendOperation = blendOperation;
        m_alphaBlendOperation = blendOperation;
    }

    /******************************************************************************
    *  SetBlendMode
    ******************************************************************************/
    void BlendEffect::SetBlendModeSeparate(RenderDevice2D::BlendFactor colorBlendFactorSrc, RenderDevice2D::BlendFactor colorBlendFactorDst,
                                           RenderDevice2D::BlendOperation colorBlendOperation,
                                           RenderDevice2D::BlendFactor alphaBlendFactorSrc, RenderDevice2D::BlendFactor alphaBlendFactorDst,
                                           RenderDevice2D::BlendOperation alphaBlendOperation)
    {
        m_colorBlendFactorSrc = colorBlendFactorSrc;
        m_alphaBlendFactorSrc = alphaBlendFactorSrc;

        m_colorBlendFactorDst = colorBlendFactorDst;
        m_alphaBlendFactorDst = alphaBlendFactorDst;

        m_colorBlendOperation = colorBlendOperation;
        m_alphaBlendOperation = alphaBlendOperation;
    }

    /******************************************************************************
     *  SetBlendMode
     ******************************************************************************/
    void BlendEffect::SetBlendMode(const RenderDevice2D::BlendMode& blendMode)
    {
        m_colorBlendFactorSrc = blendMode.colorBlendFactorSrc;
        m_colorBlendFactorDst = blendMode.colorBlendFactorDst;
        m_colorBlendOperation = blendMode.colorBlendOperation;
        m_alphaBlendFactorSrc = blendMode.alphaBlendFactorSrc;
        m_alphaBlendFactorDst = blendMode.alphaBlendFactorDst;
        m_alphaBlendOperation = blendMode.alphaBlendOperation;
    }

    /******************************************************************************
     *  GetBlendMode
     ******************************************************************************/
    RenderDevice2D::BlendMode BlendEffect::GetBlendMode() const
    {
        RenderDevice2D::BlendMode blendMode;
        blendMode.colorBlendFactorSrc = m_colorBlendFactorSrc;
        blendMode.colorBlendFactorDst = m_colorBlendFactorDst;
        blendMode.colorBlendOperation = m_colorBlendOperation;
        blendMode.alphaBlendFactorSrc = m_alphaBlendFactorSrc;
        blendMode.alphaBlendFactorDst = m_alphaBlendFactorDst;
        blendMode.alphaBlendOperation = m_alphaBlendOperation;

        return blendMode;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer BlendEffect::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(BlendEffect)(*this));
    }

}   // namespace Candera
