//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_BITMAP_BRUSH_MASK_BLEND_H)
    #define CANDERA_BITMAP_BRUSH_MASK_BLEND_H

#include <Candera/Engine2D/Effects/CombinedEffect2D.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrush.h>
#include <CanderaPlatform/Device/Common/Effects/MaskEffect.h>
#include <CanderaPlatform/Device/Common/Effects/BlendEffect.h>

namespace Candera {

    class Image2D;

    /**
     *  @brief  Output a bitmap image, apply an alpha mask, and blend it with the store buffer.
     *          Same as chaining (BitmapBrush + MaskEffect + BlendEffect)
     *          @see Candera::BitmapBrush
     *          @see Candera::MaskEffect
     *          @see Candera::BlendEffect
     */
    class BitmapBrushMaskBlend : public CombinedEffect2D {

        FEATSTD_TYPEDEF_BASE(CombinedEffect2D);

    public:

        typedef MemoryManagement::SharedPointer<BitmapBrushMaskBlend> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class
         *  @return A pointer to the created object.
         */
        static SharedPointer Create();

        /**
         *  Destructor
         */
        virtual ~BitmapBrushMaskBlend();

        // overrides Effect2D::GetBrushEffect2D
        virtual BrushEffect2D* GetBrushEffect2D() { return &m_bitmapBrush; }

        // overrides Effect2D::GetInplaceEffect2DCount
        virtual UInt8 GetInPlaceEffect2DCount() const { return 1; }

        // overrides Effect2D::GetInPlaceEffect2D        
        virtual InPlaceEffect2D* GetInPlaceEffect2D(UInt8 index) { return (index == 0)? &m_maskEffect : 0; }

        // overrides Effect2D::GetBlendEffect2D
        virtual BlendEffect2D* GetBlendEffect2D() { return &m_blendEffect; }

        // overrides Effect2D::GetBoundingRectangle
        virtual void GetBoundingRectangle(Rectangle& boundingRectangle) const { m_bitmapBrush.GetBoundingRectangle(boundingRectangle); };

        // overrides Effect2D::Upload
        virtual bool Upload();

        // overrides Effect2D::Unload
        virtual bool Unload();

        // overrides Effect2D::Update
        virtual bool Update();

        /**
         *  Retrieves the bitmap brush.
         *  @return     The bitmap brush.
         */
        const BitmapBrush& GetBitmapBrush() const { return m_bitmapBrush; }
        BitmapBrush& GetBitmapBrush() { return m_bitmapBrush; }

        /**
         *  Retrieves the mask effect.
         *  @return     The mask effect.
         */
        const MaskEffect& GetMaskEffect() const { return m_maskEffect; }
        MaskEffect& GetMaskEffect() { return m_maskEffect; }

        /**
         *  Retrieves the blend effect.
         *  @return     The blend effect.
         */
        const BlendEffect& GetBlendEffect() const { return m_blendEffect; }
        BlendEffect& GetBlendEffect() { return m_blendEffect; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /// @cond excluded from doxygen
        CdaEffect2DCombinedDef(Candera::BitmapBrushMaskBlend, BitmapBrushMaskBlend,  EFFECT2D_TYPE_BRUSH, EFFECT2D_TYPE_BLEND)
            CdaEffect2DProperties()

                CdaEffect2DImportProperties(BitmapBrush, Candera::BitmapBrush, m_bitmapBrush)
                CdaEffect2DImportProperties(MaskEffect, Candera::MaskEffect, m_maskEffect)
                CdaEffect2DImportProperties(BlendEffect, Candera::BlendEffect, m_blendEffect)

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        BitmapBrushMaskBlend();
        BitmapBrushMaskBlend(const BitmapBrushMaskBlend& rhs);
        BitmapBrushMaskBlend& operator = (const BitmapBrushMaskBlend& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

    private:
        // Use class wrapper to forward protected members.
        class BitmapBrushWrapper : public BitmapBrush {
            public:
                using BitmapBrush::Render;
        } m_bitmapBrush;
        class MaskEffectWrapper : public MaskEffect {} m_maskEffect;
        class BlendEffectWrapper : public BlendEffect {} m_blendEffect;
    };

}   // namespace Candera

#endif  // CANDERA_BITMAP_BRUSH_MASK_BLEND_H
