//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_BITMAP_BRUSH_H)
    #define CANDERA_BITMAP_BRUSH_H

//this class has been customized by Bosch
#define CANDERA_CUSTOMIZATION_BITMAP_BRUSH

#include <Candera/Engine2D/Effects/BrushEffect2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/Engine2D/Property/Image2DProperty.h>
#include <Candera/Engine2D/Property/FilterProperty.h>
#include <FeatStd/Util/Optional.h>

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3DDevicePool.h>
#endif

namespace Candera {

    class Image2D;

    /**
     *  @brief  Output an image. The image is transformed to the space of the current node,
     *  but no pixel transformation is applied (e.g. blend), for that additional
     *  effects need to be specified.
     */
    class BitmapBrush : public BrushEffect2D {

        FEATSTD_TYPEDEF_BASE(BrushEffect2D);

    public:

#ifdef CANDERA_CUSTOMIZATION_BITMAP_BRUSH
       class UploadListener
       {
       public:
          virtual ~UploadListener() {}
          virtual void OnPreUnload(const MemoryManagement::SharedPointer<Image2D>& image) = 0;
          virtual void OnPostUpload(const MemoryManagement::SharedPointer<Image2D>& image) = 0;
       };

       static void SetUploadListener(UploadListener* listener)
       {
          uploadListener() = listener;
       }
       static UploadListener* GetUploadListener()
       {
          return uploadListener();
       }
#endif

        typedef MemoryManagement::SharedPointer<BitmapBrush> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class
         *  @return MemoryManagement::SharedPointer to the created Bitmap object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        // overrides Effect2D::GetBoundingRectangle
        virtual void GetBoundingRectangle(Rectangle& boundingRectangle) const override;
        virtual void GetLayoutingRectangle(Rectangle& rectangle) const override;

        // overrides Effect2D::Upload
        virtual bool Upload();

        // overrides Effect2D::Unload
        virtual bool Unload();

        // overrides Effect2D::Update
        virtual bool Update();

        // overrides Effect2D::IsUploaded
        virtual bool IsUploaded() const { return m_uploaded; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /**
         *  Access the image attached to the effect. If the effect is already uploaded
         *  Update must be called before the changes to the image become apparent.
         *  The image may be any Image2D object.
         *  @return A property that can be used to modify the attached image.
         */
        Image2DProperty& Image() { return m_image; }

        /**
        *  Checks whether the effect contains a Bitmap that is configured as a NinePatch image.
        *  @return true if effect contains a Bitmap that is configured as a NinePatch image, false otherwise.
        */
        bool IsNinePatch() const;

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
        void ActivateNinePatch(const Matrix3x2& transform, ContextHandle2D output, const Node2D& node);
        void DeactivateNinePatch() const;
#endif

        /**
         *  Access the mipmap filter property of the effect.
         *  @return A property that can be used to modify the mipmap filter.
         */
        MipMapFilterProperty& MipMapFilter() { return m_mipMapFilter; }

#ifdef CANDERA_CUSTOMIZATION_BITMAP_BRUSH
        static FeatStd::Optional<RenderDevice2D::Filter> DefaultFilter;
#endif

        /**
        * Mesh2D cannot be supported by all effects, as certain effects would create 2D vertex buffers, which would be overriden by the Mesh2D.
        * This flag helps to distinguish between effects that can be used with Mesh2D and effects that are not available.
        * @return Whether Mesh2D can be used with this effect (true) or not (false).
        */
        virtual bool IsMesh2DSupported() const override { return true; }

        /// @cond excluded from doxygen
        //Definition of dynamic properties.
        CdaEffect2DDef(Candera::BitmapBrush, BitmapBrush, EFFECT2D_TYPE_BRUSH)
            CdaEffect2DProperties()
                CdaEffect2DImportBaseClassProperties(BrushEffect2D, Candera::BrushEffect2D)

                CdaEffect2DProperty(Image, MemoryManagement::SharedPointer<Image2D>, m_image)
                    CdaDescription("The image.")
                    CdaUpdateType(OnApplyUpdateType)
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(MipMapFilter, RenderDevice2D::MipMapFilter, m_mipMapFilter)
                    CdaDescription("Control filtering when image has mipmap chain.")
                    CdaUpdateType(ContinuousUpdateType)
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        BitmapBrush();
        explicit BitmapBrush(const BitmapBrush& rhs);
        BitmapBrush& operator = (const BitmapBrush& rhs);

        /**
         * Destructor
         */
        virtual ~BitmapBrush();

        /**
         * Render the bitmap without setting an active area. The active area is taken
         * from the RenderDevice2D.
         */
        CANDERA_DEPRECATED_3_5_0("Use RenderToArea with the additional node parameter instead.", void RenderToArea(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea));
        void RenderToArea(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea, const Node2D& node);

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
        void PrepareNinePatchVertexGeometry(Float width, Float height, const Bitmap::NinePatchProperties& ninePatchProperties, Vector2 toScale, bool scaleX, bool scaleY);

        void PrepareNinePatchIndices(UInt16* indexBuffer) const;

        inline void SetVertex(SizeType offset, Float* vertex, Float x, Float y, Float u, Float v) const
        {
            vertex[offset] = x;
            vertex[offset + 1] = y;
            vertex[offset + 2] = u;
            vertex[offset + 3] = v;
        }

        VertexBuffer::SharedPointer CreateNinePatchVertexBuffer() const;
#endif

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

    private:
#ifdef CANDERA_CUSTOMIZATION_BITMAP_BRUSH
       static UploadListener*& uploadListener()
       {
          static UploadListener* listener = NULL;
          return listener;
       }
#endif

        bool m_uploaded;
        MemoryManagement::SharedPointer<Image2D> m_uploadedImage;
        Image2DProperty m_image;
        MipMapFilterProperty m_mipMapFilter;
        BitmapImage2D* m_bitmapImage2D;
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
        VertexBuffer::SharedPointer m_vertexBuffer;
#endif
        Vector2 m_boundingRectangle;
    };

}   // namespace Candera

#endif  // CANDERA_BITMAP_BRUSH_H
