//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BitmapBrush.h"
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Image2D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/Engine2D/Core/Mesh2D.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <Candera/Engine2D/Core/Group2D.h>
#include <Candera/EngineBase/Common/AbstractNodePointer.h>
#include <Candera/EngineBase/Layout/Layouter.h>

#if defined(CANDERA_3D_ENABLED)
#include <Candera/Engine3D/Core/VertexGeometry.h>
#endif

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaPlatformDevice);
    FEATSTD_RTTI_DEFINITION(BitmapBrush, BrushEffect2D)

#ifdef CANDERA_CUSTOMIZATION_BITMAP_BRUSH
    FeatStd::Optional<RenderDevice2D::Filter> BitmapBrush::DefaultFilter;
#endif

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    BitmapBrush::BitmapBrush() :
        Base(),
        m_uploaded(false),
        m_mipMapFilter(RenderDevice2D::LinearMipMapFilter),
        m_bitmapImage2D(0),
        m_boundingRectangle(0.0F, 0.0F)
    {
#ifdef CANDERA_CUSTOMIZATION_BITMAP_BRUSH
       if (DefaultFilter == true)
       {
          m_filter = (*DefaultFilter);
       }
#endif
    }

    BitmapBrush::BitmapBrush(const BitmapBrush& rhs) :
        Base(rhs),
        m_uploaded(false),
        m_image(const_cast<Image2D*>(rhs.m_image.Get())),
        m_bitmapImage2D(0),
        m_boundingRectangle(rhs.m_boundingRectangle)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    BitmapBrush::~BitmapBrush()
    {
        m_bitmapImage2D = 0;
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    BitmapBrush::SharedPointer BitmapBrush::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        BitmapBrush* bmpBrush = FEATSTD_NEW(BitmapBrush);
        FEATSTD_DEBUG_ASSERT(bmpBrush != 0);

        BitmapBrush::SharedPointer sharedPtr(bmpBrush);
        return sharedPtr;
    }

    /******************************************************************************
     *  RenderToArea
     ******************************************************************************/
    void BitmapBrush::RenderToArea(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea)
    {
        class DummyGroup : public Group2D { public: DummyGroup() {} } group;
        RenderToArea(transform, output, outputArea, group);
    }

    void BitmapBrush::RenderToArea(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea, const Node2D& node)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "BitmapBrush"));

        if (!m_uploadedImage.PointsToNull()) {

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
            ActivateNinePatch(transform, output, node);
#else
            FEATSTD_UNUSED(node);
#endif

#ifdef CANDERA_CUSTOMIZATION_BITMAP_BRUSH
           if (Filter() == RenderDevice2D::AutomaticFilter)
           {
              if ((transform.Get(0, 1) == 0.0F) && (transform.Get(1, 0) == 0.0F) //no rotation
                 && ((transform.Get(0, 0) == 1.0F) || (m_uploadedImage->GetWidth() == 1.0F)) //no horizontal scaling or image width 1
                 && ((transform.Get(1, 1) == 1.0F) || (m_uploadedImage->GetHeight() == 1.0F))) //no vertical scaling or image height 1
              {
                 RenderDevice2D::SetFilter(output, RenderDevice2D::NearestFilter);
              }
              else
              {
                 RenderDevice2D::SetFilter(output, RenderDevice2D::BilinearFilter);
              }
           }
#endif

            bool result = Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, transform);
            result = result && m_uploadedImage->Activate(output);
            if (m_uploadedImage->IsMipMappingEnabled()) {
                static_cast<void>(RenderDevice2D::SetMipMapFilter(output, m_mipMapFilter()));
            }

            result = result && Renderer2D::Blit(output);

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
            if (!node.IsTypeOf(Mesh2D::GetTypeId())) {
                //If geometry is set Mesh2D takes care of activating and deactivating custom vertex buffer. NinePatch is not possible with Mesh2D.
                Internal::Context2DOver3DDevicePool& pool = Internal::Context2DOver3DDevicePool::GetInstance();
                pool.SetCustomVertexBuffer(VertexBuffer::SharedPointer(0));
            }
#endif

            if (result) {
                Float left;
                Float top;
                Float width;
                Float height;
                static_cast<void>(RenderDevice2D::GetUpdatedArea(output, &left, &top, &width, &height));
                outputArea = Rectangle(left, top, width, height);
            }
        }
    }

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
    void BitmapBrush::PrepareNinePatchVertexGeometry(Float width, Float height, const Bitmap::NinePatchProperties& ninePatchProperties, Vector2 toScale, bool scaleX, bool scaleY)
    {
        if (m_vertexBuffer != 0)
        {
            VertexGeometry::VertexArrayResource vertexResource(m_vertexBuffer->GetVertexGeometry()->GetVertexArrayResourceHandle());
            Float* vertices = const_cast<Float*>(FeatStd::Internal::PointerToPointer<const Float*>(vertexResource.GetData()));

            Float originalWidth = static_cast<Float>(m_image()->GetWidth());
            Float originalHeight = static_cast<Float>(m_image()->GetHeight());

            Float top = static_cast<Float>(ninePatchProperties.top);
            Float bottom = static_cast<Float>(ninePatchProperties.bottom);
            Float left = static_cast<Float>(ninePatchProperties.left);
            Float right = static_cast<Float>(ninePatchProperties.right);

            Float leftRight = originalWidth - (right + left);
            Float topBottom = originalHeight - (bottom + top);

            SizeType idx = 0;

            Float h1 = 0.0F;
            Float h2 = 0.0F;
            Float h3 = 0.0F;

            Float v1 = 0.0F;
            Float v2 = 0.0F;
            Float v3 = 0.0F;

            if (width != 0.0F) {
                if (!scaleX) {
                    h1 = left / width;
                    h2 = h1 + (leftRight * toScale.GetX()) / width;
                    h3 = 1.0F;
                }
                else {
                    h1 = (left * toScale.GetX()) / width;
                    h2 = h1;
                    h3 = 1.0F;
                }
            }

            if (height != 0.0F) {
                if (!scaleY) {
                    v1 = top / height;
                    v2 = v1 + (topBottom * toScale.GetY()) / height;
                    v3 = 1.0F;
                }
                else {
                    v1 = (top * toScale.GetY()) / height;
                    v2 = v1;
                    v3 = 1.0F;
                }
            }

            SetVertex(idx, vertices, 0.0F, 0.0F, 0.0F, 0.0F);
            idx += 4;
            SetVertex(idx, vertices, h1, 0.0F, left / originalWidth, 0.0F);
            idx += 4;
            SetVertex(idx, vertices, h2, 0.0F, 1.0F - (right / originalWidth), 0.0F);
            idx += 4;
            SetVertex(idx, vertices, h3, 0.0F, 1.0F, 0.0F);
            idx += 4;

            SetVertex(idx, vertices, 0.0F, v1, 0.0F, top / originalHeight);
            idx += 4;
            SetVertex(idx, vertices, h1, v1, left / originalWidth, top / originalHeight);
            idx += 4;
            SetVertex(idx, vertices, h2, v1, 1.0F - (right / originalWidth), top / originalHeight);
            idx += 4;
            SetVertex(idx, vertices, h3, v1, 1.0F, top / originalHeight);
            idx += 4;

            SetVertex(idx, vertices, 0.0F, v2, 0.0F, (1.0F - (bottom / originalHeight)));
            idx += 4;
            SetVertex(idx, vertices, h1, v2, left / originalWidth, 1.0F - (bottom / originalHeight));
            idx += 4;
            SetVertex(idx, vertices, h2, v2, 1.0F - (right / originalWidth), 1.0F - (bottom / originalHeight));
            idx += 4;
            SetVertex(idx, vertices, h3, v2, 1.0F, 1.0F - (bottom / originalHeight));
            idx += 4;

            SetVertex(idx, vertices, 0.0F, v3, 0.0F, 1.0F);
            idx += 4;
            SetVertex(idx, vertices, h1, v3, left / originalWidth, 1.0F);
            idx += 4;
            SetVertex(idx, vertices, h2, v3, 1.0F - (right / originalWidth), 1.0F);
            idx += 4;
            SetVertex(idx, vertices, h3, v3, 1.0F, 1.0F);

            if (!m_vertexBuffer->IsUploaded()) {
                if (!m_vertexBuffer->Upload(DeviceObject::NoHint)) {
                    return;
                }
            }
            else {
                if (m_vertexBuffer->Update(0, 16)) {
                    m_vertexBuffer->SetElementCount(54);
                }
                else {
                    return;
                }
            }
        }
    }

    void BitmapBrush::PrepareNinePatchIndices(UInt16* indexBuffer) const
    {
        SizeType idx = 0;
        // Upper row
        indexBuffer[idx++] = 0; indexBuffer[idx++] = 4; indexBuffer[idx++] = 1;
        indexBuffer[idx++] = 4; indexBuffer[idx++] = 5; indexBuffer[idx++] = 1;
        indexBuffer[idx++] = 1; indexBuffer[idx++] = 5; indexBuffer[idx++] = 2;
        indexBuffer[idx++] = 5; indexBuffer[idx++] = 6; indexBuffer[idx++] = 2;
        indexBuffer[idx++] = 2; indexBuffer[idx++] = 6; indexBuffer[idx++] = 3;
        indexBuffer[idx++] = 6; indexBuffer[idx++] = 7; indexBuffer[idx++] = 3;

        // Middle row
        indexBuffer[idx++] = 4; indexBuffer[idx++] = 8; indexBuffer[idx++] = 5;
        indexBuffer[idx++] = 8; indexBuffer[idx++] = 9; indexBuffer[idx++] = 5;
        indexBuffer[idx++] = 5; indexBuffer[idx++] = 9; indexBuffer[idx++] = 6;
        indexBuffer[idx++] = 9; indexBuffer[idx++] = 10; indexBuffer[idx++] = 6;
        indexBuffer[idx++] = 6; indexBuffer[idx++] = 10; indexBuffer[idx++] = 7;
        indexBuffer[idx++] = 10; indexBuffer[idx++] = 11; indexBuffer[idx++] = 7;

        // Lower row
        indexBuffer[idx++] = 8; indexBuffer[idx++] = 12; indexBuffer[idx++] = 9;
        indexBuffer[idx++] = 12; indexBuffer[idx++] = 13; indexBuffer[idx++] = 9;
        indexBuffer[idx++] = 9; indexBuffer[idx++] = 13; indexBuffer[idx++] = 10;
        indexBuffer[idx++] = 13; indexBuffer[idx++] = 14; indexBuffer[idx++] = 10;
        indexBuffer[idx++] = 10; indexBuffer[idx++] = 14; indexBuffer[idx++] = 11;
        indexBuffer[idx++] = 14; indexBuffer[idx++] = 15; indexBuffer[idx++] = 11;
    }

    VertexBuffer::SharedPointer BitmapBrush::CreateNinePatchVertexBuffer() const
    {
        using namespace Candera::Internal;

        VertexBuffer::SharedPointer vertexBuffer = VertexBuffer::Create();
        if (vertexBuffer == 0) {
            return VertexBuffer::SharedPointer(0);
        }

        SizeType vertexCount = 16;
        Context2DOver3DDevicePool::Vertex* vertices = CANDERA_NEW_ARRAY(Context2DOver3DDevicePool::Vertex, vertexCount);
        if (0 == vertices) {
            return VertexBuffer::SharedPointer(0);
        }

        SizeType indexCount = 18 * 3;
        UInt16* indexBuffer = CANDERA_NEW_ARRAY(UInt16, indexCount);
        if (0 == indexBuffer) {
            CANDERA_DELETE_ARRAY(vertices);
            return VertexBuffer::SharedPointer(0);
        }
        PrepareNinePatchIndices(indexBuffer);

        VertexGeometry* vertexGeometry = CANDERA_NEW(VertexGeometry)(
            vertices, MemoryManagement::AdaptedArrayDisposer<const void*, const Context2DOver3DDevicePool::Vertex*>::Dispose,
            Context2DOver3DDevicePool::VertexFormat, 0,
            indexBuffer, MemoryManagement::AdaptedArrayDisposer<const void*, const UInt16*>::Dispose,
            static_cast<UInt32>(vertexCount), sizeof(Context2DOver3DDevicePool::Vertex), 2, static_cast<UInt32>(indexCount),
            VertexGeometry::VideoMemory,
            VertexGeometry::UInt16IndexedArrayBuffer,
            VertexGeometry::DynamicWrite);

        static_cast<void>(vertexBuffer->SetVertexGeometry(vertexGeometry, VertexBuffer::VertexGeometryDisposer::Dispose));
        vertexBuffer->SetPrimitiveType(VertexBuffer::Triangles);

        return vertexBuffer;
    }
#endif

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void BitmapBrush::Render(SurfaceHandle /*input*/, const Rectangle& /*inputArea*/, const Matrix3x2& transform,
                             const Node2D& node, ContextHandle2D output, Rectangle& outputArea)
    {
        Rectangle rect(0.0F, 0.0F, -1.0F, -1.0F);
        if (node.IsTypeOf(Mesh2D::GetTypeId())) {
            const Mesh2D& mesh = static_cast<const Mesh2D&>(node);
            if (!(mesh.GetVertexBuffer2D().PointsToNull())) {
                rect = mesh.GetVertexBuffer2D()->GetBoundingRectangle();
            }
        }
        if (RenderDevice2D::SetActiveArea(output, RenderDevice2D::SourceSurface, rect.GetPosition().GetX(), rect.GetPosition().GetY(), rect.GetSize().GetX(), rect.GetSize().GetY())) {
            RenderToArea(transform, output, outputArea, node);
        }
    }


    /******************************************************************************
     *  GetBoundingRectangle
     ******************************************************************************/
    void BitmapBrush::GetBoundingRectangle(Rectangle& boundingRectangle) const
    {
        boundingRectangle.SetPosition(0.0F, 0.0F);

        Vector2 size(0.0F, 0.0F);

        if (m_image() != 0) {
            Vector2 toValue(static_cast<Float>(m_image()->GetWidth()),
                static_cast<Float>(m_image()->GetHeight()));

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
            if ((m_bitmapImage2D != 0) && (m_bitmapImage2D->GetBitmap() != 0) && m_bitmapImage2D->GetBitmap()->IsNinePatch()) {
                toValue.SetX(m_boundingRectangle.GetX());
                toValue.SetY(m_boundingRectangle.GetY());
            }
#endif
            size.SetX(toValue.GetX());
            size.SetY(toValue.GetY());
        }

        boundingRectangle.SetSize(size);
    }

    /******************************************************************************
    *  GetBoundingRectangle
    ******************************************************************************/
    void BitmapBrush::GetLayoutingRectangle(Rectangle& rectangle) const
    {
        rectangle.SetPosition(0.0F, 0.0F);
        rectangle.SetSize((0 != m_image()) ? Vector2(static_cast<Float>(m_image()->GetWidth()), static_cast<Float>(m_image()->GetHeight())): Vector2());
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool BitmapBrush::Upload()
    {
        if (m_uploaded) {
            return false;
        }

        if (m_uploadedImage.PointsToNull() && (m_image() != 0)) {
            if (!m_image()->Upload()){
                return false;
            }
            m_uploadedImage = MemoryManagement::SharedPointer<Image2D>(m_image());
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
            m_bitmapImage2D = Dynamic_Cast<BitmapImage2D*>(m_uploadedImage.GetPointerToSharedInstance());
#endif

#ifdef CANDERA_CUSTOMIZATION_BITMAP_BRUSH
           if (GetUploadListener() != 0) {
              GetUploadListener()->OnPostUpload(m_uploadedImage);
           }
#endif
        }

        m_uploaded = true;
        return true;
    }

    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool BitmapBrush::Unload()
    {
        if (!m_uploaded) {
            return false;
        }

        if (!m_uploadedImage.PointsToNull()) {
#ifdef CANDERA_CUSTOMIZATION_BITMAP_BRUSH
            if (GetUploadListener() != 0) {
                GetUploadListener()->OnPreUnload(m_uploadedImage);
            }
#endif

            if (!m_uploadedImage->Unload()) {
                return false;
            }
            m_uploadedImage.Release();
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
            m_bitmapImage2D = 0;
            if ((m_vertexBuffer != 0) && m_vertexBuffer->IsUploaded()) {
                if (!m_vertexBuffer->Unload()) {
                    return false;
                }
            }
#endif
        }

        m_uploaded = false;

        return true;
    }

    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool BitmapBrush::Update()
    {
        if (!m_uploaded) {
            return false;
        }

        bool success = true;
        if (m_image() != m_uploadedImage.GetPointerToSharedInstance()) {
            success = ((m_image() == 0) || m_image()->Upload());
            if (m_uploadedImage != 0) {
               success = success &&  m_uploadedImage->Unload();
                m_uploadedImage.Release();
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
                m_bitmapImage2D = 0;
#endif
            }
            if (success) {
                m_uploadedImage = MemoryManagement::SharedPointer<Image2D>(m_image());
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
                m_bitmapImage2D = Dynamic_Cast<BitmapImage2D*>(m_uploadedImage.GetPointerToSharedInstance());
#endif
            }
        }

        return success;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer BitmapBrush::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(BitmapBrush)(*this));
    }

    /******************************************************************************
    *  IsNinePatch
    ******************************************************************************/
    bool BitmapBrush::IsNinePatch() const
    {
        if ((0 != m_bitmapImage2D) && (m_bitmapImage2D->GetBitmap() != 0)) {
            return m_bitmapImage2D->GetBitmap()->IsNinePatch();
        }
        return false;
    }

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
    void BitmapBrush::ActivateNinePatch(const Matrix3x2& transform, ContextHandle2D output, const Node2D& node)
    {
        if (!m_uploadedImage.PointsToNull()) {

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
            Vector2 actualSize;
            AbstractNodePointer abstractNodePointer(const_cast<Node2D*>(&node));
#ifdef CANDERA_LAYOUT_ENABLED
            if ((0 != m_bitmapImage2D) && (m_bitmapImage2D->GetBitmap() != 0)) {
                if (m_bitmapImage2D->GetBitmap()->IsNinePatch() && (!node.IsTypeOf(Mesh2D::GetTypeId()))) {
                    actualSize = Layouter::GetActualSize(abstractNodePointer);
                }
            }
#endif
#else
            FEATSTD_UNUSED(node);
#endif

            static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, transform));
            static_cast<void>(m_uploadedImage->Activate(output));
            if (m_uploadedImage->IsMipMappingEnabled()) {
                static_cast<void>(RenderDevice2D::SetMipMapFilter(output, m_mipMapFilter()));
            }

            Internal::Context2DOver3DDevicePool& pool = Internal::Context2DOver3DDevicePool::GetInstance();
            VertexBuffer::SharedPointer vertexBuffer;

            if ((0 != m_bitmapImage2D) && (m_bitmapImage2D->GetBitmap() != 0)) {
                if (m_bitmapImage2D->GetBitmap()->IsNinePatch() &&
                    (!node.IsTypeOf(Mesh2D::GetTypeId()))) {  //Only allow nine patch creation if no geometry is set.
                    Vector2 scale(node.GetScale());

                    if (m_vertexBuffer == 0) {
                        m_vertexBuffer = CreateNinePatchVertexBuffer();
                    }

                    if (m_vertexBuffer != 0) {
#ifdef CANDERA_LAYOUT_ENABLED
                        if (actualSize == Vector2(-1.F, -1.F)) {
                            actualSize.SetX(Layouter::GetSize(*(abstractNodePointer.ToCanderaObject())).GetX());
                            if (Math::FloatAlmostEqual(actualSize.GetX(), -1.F)) {
                                actualSize.SetX(static_cast<Float>(m_uploadedImage->GetWidth()));
                            }
                            actualSize.SetY(Layouter::GetSize(*(abstractNodePointer.ToCanderaObject())).GetY());
                            if (Math::FloatAlmostEqual(actualSize.GetY(), -1.F)) {
                                actualSize.SetY(static_cast<Float>(m_uploadedImage->GetHeight()));
                            }
                        }
#else
                        actualSize.SetX(static_cast<Float>(m_uploadedImage->GetWidth()));
                        actualSize.SetY(static_cast<Float>(m_uploadedImage->GetHeight()));
#endif

                        Float scaledWidth = actualSize.GetX();
                        Float scaledHeight = actualSize.GetY();

                        Bitmap::NinePatchProperties npp = *(m_bitmapImage2D->GetBitmap()->GetNinePatchProperties());

                        Float horizontalBordersSum = static_cast<Float>(npp.left + npp.right);
                        Float verticalBordersSum = static_cast<Float>(npp.top + npp.bottom);
                        Float leftRight = static_cast<Float>(m_uploadedImage->GetWidth()) - horizontalBordersSum;
                        Float topBottom = static_cast<Float>(m_uploadedImage->GetHeight()) - verticalBordersSum;

                        Float scaledLeftRight = scaledWidth - horizontalBordersSum;
                        Float scaledTopBottom = scaledHeight - verticalBordersSum;

                        Vector2 toScale;
                        bool xScale = false;
                        bool yScale = false;

                        if (scaledLeftRight <= .0F) {
                            xScale = true;
                            toScale.SetX(((scaledWidth != 0.0F) && (horizontalBordersSum != 0.0F)) ? (scaledWidth / horizontalBordersSum) : 0.0F);
                            FEATSTD_DEBUG_ASSERT(toScale.GetX() <= 1.F);
                            scaledWidth = horizontalBordersSum * toScale.GetX();
                        }
                        else {
                            toScale.SetX((leftRight != 0.0F) ? (scaledLeftRight / leftRight) : 0.0F);
                        }

                        if (scaledTopBottom <= .0F) {
                            yScale = true;
                            toScale.SetY(((scaledHeight != 0.0F) && (verticalBordersSum != 0.0F)) ? (scaledHeight / verticalBordersSum) : 0.0F);
                            FEATSTD_DEBUG_ASSERT(toScale.GetY() <= 1.F);
                            scaledHeight = verticalBordersSum * toScale.GetY();
                        }
                        else {
                            toScale.SetY((topBottom != 0.0F) ? (scaledTopBottom / topBottom) : 0.0F);
                        }

                        static_cast<void>(RenderDevice2D::SetActiveArea(output, RenderDevice2D::SourceSurface, 0.F, 0.F, scaledWidth, scaledHeight));
                        Candera::Rectangle rect = Candera::Rectangle(0.0F, 0.0F, scaledWidth, scaledHeight);
                        pool.SetCustomVertexBuffer(m_vertexBuffer, rect);

                        m_boundingRectangle.SetX(scaledWidth);
                        m_boundingRectangle.SetY(scaledHeight);

                        PrepareNinePatchVertexGeometry(scaledWidth, scaledHeight, npp, toScale, xScale, yScale);
                    }
                }
                else {
                    if (m_bitmapImage2D->GetBitmap()->IsNinePatch() && (node.IsTypeOf(Mesh2D::GetTypeId()))) {
                        FEATSTD_LOG_INFO("NinePatch is ignored when rendered with Mesh2D.");
                    }
                }
            }
        }
    }

    void BitmapBrush::DeactivateNinePatch() const
    {
        Internal::Context2DOver3DDevicePool& pool = Internal::Context2DOver3DDevicePool::GetInstance();
        pool.SetCustomVertexBuffer(VertexBuffer::SharedPointer(0));
    }

#endif

}   // namespace Candera
