//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "EglTypeMapper.h"

namespace Candera {


    EGLint EglTypeMapper::MapEglInteger(Int32 value)
    {
        return (value == static_cast<Int32>(EglConfiguration::EglDontCare)) ? EGL_DONT_CARE : static_cast<EGLint>(value);
    }

    EGLint EglTypeMapper::MapEglBoolean(Int32 value)
    {
        EGLint converted = EGL_DONT_CARE;
        switch (value) {
            case EglConfiguration::EglTrue:
                converted = EGL_TRUE;
                break;
            case EglConfiguration::EglFalse:
                converted = EGL_FALSE;
                break;
            case EglConfiguration::EglDontCare:
                // converted is already set to EGL_DONT_CARE
                break;
            default:
                break;
        }
        return converted;
    }

    EGLint EglTypeMapper::MapEglConformantBitmask(Int32 value)
    {
        UInt32 convertedValue = 0;
        UInt32 unsignedValue = static_cast<UInt32>(value);
        if ((unsignedValue & static_cast<UInt32>(EglConfiguration::EglConformantOpenGlBit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_OPENGL_BIT);
        }
        if ((unsignedValue & static_cast<UInt32>(EglConfiguration::EglConformantOpenGlEsBit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_OPENGL_ES_BIT);
        }
        if ((unsignedValue & static_cast<UInt32>(EglConfiguration::EglConformantOpenGlEs2Bit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_OPENGL_ES2_BIT);
        }
        if ((unsignedValue & static_cast<UInt32>(EglConfiguration::EglConformantOpenVgBit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_OPENVG_BIT);
        }
        return static_cast<EGLint>(convertedValue);
    }

    EGLint EglTypeMapper::MapEglRenderableTypeBitmask(Int32 value)
    {
        UInt32 convertedValue = 0;
        UInt32 unsignedValue = static_cast<UInt32>(value);
        if ((unsignedValue & static_cast<UInt32>(EglConfiguration::EglRenderableTypeOpenGlBit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_OPENGL_BIT);
        }
        if ((unsignedValue & static_cast<UInt32>(EglConfiguration::EglRenderableTypeOpenGlEsBit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_OPENGL_ES_BIT);
        }
        if ((unsignedValue & static_cast<UInt32>(EglConfiguration::EglRenderableTypeOpenGlEs2Bit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_OPENGL_ES2_BIT);
        }
        if ((unsignedValue & static_cast<UInt32>(EglConfiguration::EglRenderableTypeOpenVgBit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_OPENVG_BIT);
        }
        return static_cast<EGLint>(convertedValue);
    }

    EGLint EglTypeMapper::MapEglSurfaceTypeBitmask(Int32 value)
    {
        UInt32 convertedValue = 0;
        UInt32 unsignedValue = static_cast<UInt32>(value);
        if ((unsignedValue &  static_cast<UInt32>(EglConfiguration::EglWindowBit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_WINDOW_BIT);
        }
        if ((unsignedValue &  static_cast<UInt32>(EglConfiguration::EglPixmapBit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_PIXMAP_BIT);
        }
        if ((unsignedValue &  static_cast<UInt32>(EglConfiguration::EglPBufferBit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_PBUFFER_BIT);
        }
        if ((unsignedValue &  static_cast<UInt32>(EglConfiguration::EglMultisampleResolveBoxBit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_MULTISAMPLE_RESOLVE_BOX_BIT);
        }
        if ((unsignedValue &  static_cast<UInt32>(EglConfiguration::EglSwapBehaviourPreservedBit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_SWAP_BEHAVIOR_PRESERVED_BIT);
        }
        if ((unsignedValue &  static_cast<UInt32>(EglConfiguration::EglVgColorspaceLinearBit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_VG_COLORSPACE_LINEAR_BIT);
        }
        if ((unsignedValue &  static_cast<UInt32>(EglConfiguration::EglVgAlphaFormatPreBit)) != 0) {
            convertedValue |= static_cast<UInt32>(EGL_VG_ALPHA_FORMAT_PRE_BIT);
        }
        return static_cast<EGLint>(convertedValue);
    }

    EGLint EglTypeMapper::MapEglDepthEncodingEnum(Int32 value)
    {
        EGLint eglValue = EGL_DONT_CARE;
        switch (value) {
            case EglConfiguration::EglDepthEncodingNone:
                eglValue = EGL_DEPTH_ENCODING_NONE_NV;
                break;
            case EglConfiguration::EglDepthEncodingNonlinear:
                eglValue = EGL_DEPTH_ENCODING_NONLINEAR_NV;
                break;
            default:
                break;
        }
        return eglValue;
    }

    void EglTypeMapper::MapEglConfiguration(EGLint* eglConfig, const Int32* const config)
    {
        Int32 attributeCount = 0; // Number of attributes that shall be propagated to eglConfig.

        for (Int32 i = 0; i < static_cast<Int32>(EglConfiguration::EglNumberOfConfigAttributes); ++i) {
            EGLint eglAttribute = static_cast<EGLint>(eglConfigAttributeMapping[i]);

            /**
             * Check if eglAttribute is supported. If an EglConfigAttribute of class EGLConfiguration cannot be mapped to a native EGL value,
             * then this is an extension that is not supported and thus must not be propagated to EGL via eglConfig.
             */
            if (eglAttribute != c_eglInvalidParameter) {
                // Add EGL native attribute name to eglConfig.
                eglConfig[attributeCount * 2] = eglAttribute;
                // Map and add related EGL value to eglConfig.
                switch (i) {
                    // integer
                    case EglConfiguration::EglBufferSize:
                    case EglConfiguration::EglRedSize:
                    case EglConfiguration::EglGreenSize:
                    case EglConfiguration::EglBlueSize:
                    case EglConfiguration::EglLuminanceSize:
                    case EglConfiguration::EglAlphaSize:
                    case EglConfiguration::EglAlphaMaskSize:
                    case EglConfiguration::EglConfigId:
                    case EglConfiguration::EglDepthSize:
                    case EglConfiguration::EglLevel:
                    case EglConfiguration::EglMaxSwapInterval:
                    case EglConfiguration::EglMinSwapInterval:
                    case EglConfiguration::EglNativeVisualType:
                    case EglConfiguration::EglSampleBuffers:
                    case EglConfiguration::EglSamples:
                    case EglConfiguration::EglStencilSize:
                    case EglConfiguration::EglTransparentRedValue:
                    case EglConfiguration::EglTransparentGreenValue:
                    case EglConfiguration::EglTransparentBlueValue:
                    case EglConfiguration::EglCoverageSampleBuffers:
                    case EglConfiguration::EglCoverageSamples:
                        eglConfig[attributeCount * 2 + 1] = MapEglInteger(config[i]);
                        break;

                    // boolean
                    case EglConfiguration::EglNativeRenderable:
                        eglConfig[attributeCount * 2 + 1] = MapEglBoolean(config[i]);
                        break;

                    // bitmask
                    case EglConfiguration::EglConformant:
                        eglConfig[attributeCount * 2 + 1] = MapEglConformantBitmask(config[i]);
                        break;
                    case EglConfiguration::EglRenderableType:
                        eglConfig[attributeCount * 2 + 1] = MapEglRenderableTypeBitmask(config[i]);
                        break;
                    case EglConfiguration::EglSurfaceType:
                        eglConfig[attributeCount * 2 + 1] = MapEglSurfaceTypeBitmask(config[i]);
                        break;

                    // enum
                    case EglConfiguration::EglConfigCaveat:
                        eglConfig[attributeCount * 2 + 1] = MapEglConfigCaveatEnum(config[i]);
                        break;
                      case EglConfiguration::EglTransparentType:
                          eglConfig[attributeCount * 2 + 1] = MapEglTransparentTypeEnum(config[i]);
                          break;
                    case EglConfiguration::EglColorBufferType:
                        eglConfig[attributeCount * 2 + 1] = MapEglColorBufferTypeEnum(config[i]);
                        break;
                    case EglConfiguration::EglDepthEncoding:
                        eglConfig[attributeCount * 2 + 1] = MapEglDepthEncodingEnum(config[i]);
                        break;

                    default:
                        break;
                } // switch

                attributeCount++;
            } // if
        } // for

        // Terminate EGL config array with EGL_NONE.
        eglConfig[2 * (static_cast<Int32>(attributeCount))] = EGL_NONE;
    }


} // namespace Candera

