//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_EglKhrExternalTextureImage_h
#define Candera_EglKhrExternalTextureImage_h

#include <CanderaPlatform/Device/Common/Base/ExternalTextureImage.h>
#include <CanderaPlatform/Device/Common/EGL/EglInclude.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>


namespace Candera {

    /**
    *  @brief EglKhrExternalTextureImage stores information necessary for creating an external texture.
    *  Due to the design of the EGL / GL extension this can be used either as consumer (texture) or as producer (attached to an FBO / Offscreen Rendertarget) but never as both.
    *  Width and height can be set explicitly or through the attribute list.
    **/
    class EglKhrExternalTextureImage : public ExternalTextureImage {
        friend class RenderDevice;
        FEATSTD_TYPEDEF_BASE(ExternalTextureImage);
    public:

        FEATSTD_RTTI_DECLARATION();
        FEATSTD_TYPEDEF_SHARED_POINTER(EglKhrExternalTextureImage);
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        enum Role{
            Producer,
            Consumer,
            Undefined
        };

        //dtor
        virtual ~EglKhrExternalTextureImage();

        // Overridden function of TextureImage.
        virtual Handle GetVideoMemoryHandle() const override;

        // Overridden function of TextureImage.
        virtual TextureTargetType GetTextureTargetType() const override;

        // Overridden function of Measurable.
        virtual UInt GetSize() const override;


        // Can be either PRODUCER or CONSUMER never both at the same time. Can only be set ONCE on purpose.
        Role GetRole() const { return m_role; }
        void SetRole(Role role) { if (m_role == Undefined) { m_role = role; } }

        // EGL_NONE terminated list of attributes for the external buffer import (type, value)*EGL_NONE
        // @param length Size of the array including terminator.
        void SetAttributeList(const Int* attributeList, const SizeType length);

        // returns EGL_NONE terminated list of attributes
        Int* GetAttributeList() { return m_eglAttributeList; }

        // set type of buffer that is handed over by SetExternalBufferType XOR SetAttributeList.
        void SetExternalBuffer(void* buffer){ m_externalBuffer = buffer; }
        // returns type of buffer
        void* GetExternalBuffer() {return m_externalBuffer;}

        // set external buffer. Some types of external buffer require a NULL pointer here and the actual buffer handle in the attribute list.
        void SetExternalBufferType(EGLint bufferType) { m_externalBufferType = bufferType; }
        // get external buffer.
        EGLint GetExternalBufferType() { return m_externalBufferType; }


        virtual void DisposeInternal() override;


    protected:
        virtual bool UploadInternal(LoadingHint loadingHint) override;


        virtual bool UnloadInternal(LoadingHint loadingHint) override;

    private:
        FEATSTD_MAKE_CLASS_UNCOPYABLE(EglKhrExternalTextureImage);

        Handle m_videoMemoryHandle[CANDERA_MAX_CONTEXT_COUNT];
        void SetVideoMemoryHandle(Handle handle);
        void SetEglImage(void* image){ m_eglImage = image; }
        void* GetEglImage(){ return m_eglImage; }
        void* m_eglImage;
        Int* m_eglAttributeList;
        void* m_externalBuffer;
        Int m_externalBufferType;
        Role m_role;
        EglKhrExternalTextureImage();
    };
}

#endif
