//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################
#include "EglKhrExternalTextureImage.h"

#include <CanderaPlatform/Device/Common/EGL/EglInclude.h>
#include <CanderaPlatform/Device/Common/EGL/EglWrapper.h>

#include <Candera/System/Diagnostics/Log.h>
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/Engine3D/Core/TextureImage.h>

#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine3D);

    EglKhrExternalTextureImage::EglKhrExternalTextureImage()
        :
        m_eglImage(EGL_NO_IMAGE_KHR),
        m_eglAttributeList(0),
        m_externalBuffer(0),
        m_externalBufferType(0),
        m_role(Undefined)
    {
        MemoryPlatform::Set(m_videoMemoryHandle, 0, sizeof(m_videoMemoryHandle));
    }
    EglKhrExternalTextureImage::~EglKhrExternalTextureImage(){
        static_cast<void>(Unload(ForceAll));
        DisposeInternal();
    }

    EglKhrExternalTextureImage::SharedPointer EglKhrExternalTextureImage::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)
        EglKhrExternalTextureImage* ptr = FEATSTD_NEW(EglKhrExternalTextureImage)();
        if (ptr == 0) {
            FEATSTD_LOG_ERROR("Texture image create failed, out of memory.");
        }
        EglKhrExternalTextureImage::SharedPointer sharedPointer(ptr);
        return sharedPointer;
    }

    Handle EglKhrExternalTextureImage::GetVideoMemoryHandle() const
    {
        return m_videoMemoryHandle[ContextResourcePool::GetActive().GetIndex()];
    }


    void EglKhrExternalTextureImage::SetVideoMemoryHandle(Handle handle)
    {
        m_videoMemoryHandle[ContextResourcePool::GetActive().GetIndex()] = handle;
    }

    Candera::TextureImage::TextureTargetType EglKhrExternalTextureImage::GetTextureTargetType() const
    {
        //Workaround for extension design.
        if (m_role == Producer){
            return Candera::TextureImage::TextureExternalOes;
        }
        else {
            return Candera::TextureImage::Texture2D;
        }
    }


    bool EglKhrExternalTextureImage::UploadInternal(LoadingHint loadingHint)
    {
        if (m_role!=Undefined){
            return Renderer::UploadExternalTextureImage(*this, 0, loadingHint);
        }
        else{
            FEATSTD_LOG_ERROR("Upload failed due to undefined role.");
            return false;
        }
    }


    bool EglKhrExternalTextureImage::UnloadInternal(LoadingHint loadingHint)
    {
        return Renderer::UnloadExternalTextureImage(*this, loadingHint);
    }

    UInt EglKhrExternalTextureImage::GetSize() const
    {
        return 0;
    }

    void EglKhrExternalTextureImage::SetAttributeList(const Int* attributeList, const SizeType lenght)
    {
        //cleanup
        if (0 != m_eglAttributeList){
            FEATSTD_DELETE_ARRAY(m_eglAttributeList);
            m_eglAttributeList = 0;
        }

        //sanity check
        if (0 == attributeList || lenght <=0 ){
            FEATSTD_LOG_ERROR("Invalid argument");
            return;
        }

        m_eglAttributeList = FEATSTD_NEW_ARRAY(Int, lenght);

        // copy attributes and set width and height if in attribute list
        SizeType i = 0;
        while (((i < lenght) && !((0 == i % 2) && (attributeList[i] == EGL_NONE) /*check for termination*/))){
            m_eglAttributeList[i] = attributeList[i];
            //set width and height from attribute list
            if (0 == i % 2 && m_eglAttributeList[i] == EGL_WIDTH){
                SetWidth(attributeList[i + 1]);
            }
            if (0 == i % 2 && m_eglAttributeList[i] == EGL_HEIGHT){
                SetHeight(attributeList[i + 1]);
            }
            i++;
        }
        // Terminate the list
        m_eglAttributeList[i] = EGL_NONE;

    }

    void EglKhrExternalTextureImage::DisposeInternal()
    {
        Base::DisposeInternal();
        if (0 != m_eglAttributeList){
            FEATSTD_DELETE_ARRAY(m_eglAttributeList);
            m_eglAttributeList = 0;
        }
    }

    FEATSTD_RTTI_DEFINITION(EglKhrExternalTextureImage, Base)



}
