/* The authors of this work have released all rights to it and placed it
in the public domain under the Creative Commons CC0 1.0 waiver
(http://creativecommons.org/publicdomain/zero/1.0/).

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/
 
//########################################################################
// Source code from http://en.literateprograms.org/Median_cut_algorithm_(C_Plus_Plus)?oldid=19175 (retrieved on 2015-03-25)
// has been modified for integration by Socionext Embedded Software Austria (SESA)
//
// (C)  Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <CanderaPlatform/Device/Common/BitmapConverter/MedianCut.h>
#include <queue>
#include <algorithm>

namespace Candera { namespace ClutBitmapConverter {

    Block::Block(Point* points, OffsetType pointsLength) :
        m_points(points),
        m_pointsLength(pointsLength)
    {
        for (Int32 i = 0; i < NUM_DIMENSIONS; i++) {
            m_minCorner.x[i] = 0x00;
            m_maxCorner.x[i] = 0xff;
        }
    }

    Point* Block::GetPoints()
    {
        return m_points;
    }

    OffsetType Block::GetNumPoints() const
    {
        return m_pointsLength;
    }

    Int32 Block::GetLongestSideIndex() const
    {
        Int32 m = static_cast<Int32>(m_maxCorner.x[0] - m_minCorner.x[0]);
        Int32 maxIndex = 0;
        for (Int32 i = 1; i < NUM_DIMENSIONS; i++) {
            Int32 diff = static_cast<Int32>(m_maxCorner.x[i] - m_minCorner.x[i]);
            if (diff > m) {
                m = diff;
                maxIndex = i;
            }
        }
        return maxIndex;
    }

    Int32 Block::GetLongestSideLength() const
    {
        Int32 i = GetLongestSideIndex();
        return static_cast<Int32>(m_maxCorner.x[i] - m_minCorner.x[i]);
    }

    bool Block::operator<(const Block& rhs) const
    {
        return this->GetLongestSideLength() < rhs.GetLongestSideLength();
    }

    void Block::Shrink()
    {
        OffsetType i;
        Int32 j;
        for (j = 0; j < NUM_DIMENSIONS; j++) {
            m_minCorner.x[j] = m_points[0].x[j];
            m_maxCorner.x[j] = m_points[0].x[j];
        }
        for (i = 1; i < m_pointsLength; i++) {
            for (j = 0; j < NUM_DIMENSIONS; j++) {
                m_minCorner.x[j] = GetMin(m_minCorner.x[j], m_points[i].x[j]);
                m_maxCorner.x[j] = GetMax(m_maxCorner.x[j], m_points[i].x[j]);
            }
        }
    }

    VECTOR(Point) MedianCut(Point* image, OffsetType numPoints, SizeType desiredSize)
    {
        std::priority_queue<Block> blockQueue;
        Block initialBlock(image, numPoints);
        initialBlock.Shrink();
        blockQueue.push(initialBlock);
        while (blockQueue.size() < desiredSize) {
            Block longestBlock = blockQueue.top();
            blockQueue.pop();
            Point* begin = longestBlock.GetPoints();
            Point* median = begin + (longestBlock.GetNumPoints() + 1) / 2;
            Point* end = begin + longestBlock.GetNumPoints();
            switch (longestBlock.GetLongestSideIndex()) {
            case 0: std::nth_element(begin, median, end, CoordinatePointComparator<0>()); break;
            case 1: std::nth_element(begin, median, end, CoordinatePointComparator<1>()); break;
            case 2: std::nth_element(begin, median, end, CoordinatePointComparator<2>()); break;
            case 3: std::nth_element(begin, median, end, CoordinatePointComparator<3>()); break;
            default: break;
            }

            FEATSTD_LINT_CURRENT_SCOPE(946, "Relational or subtract operator applied to pointers [MISRA C++ Rule 5-0-15], [MISRA C++ Rule 5-0-17], [MISRA C++ Rule 5-0-18]: changing this would affect the algorithm ")
            FEATSTD_LINT_CURRENT_SCOPE(947, "Subtract operator applied to pointers [MISRA C++ Rule 5-0-15], [MISRA C++ Rule 5-0-17], [MISRA C++ Rule 5-0-18]: changing this would affect the algorithm ")
            Block block1(begin, median - begin);
            Block block2(median, end - median);
            block1.Shrink();
            block2.Shrink();
            blockQueue.push(block1);
            blockQueue.push(block2);
        }

        VECTOR(Point) result;
        while (!blockQueue.empty()) {
            Block block = blockQueue.top();
            blockQueue.pop();
            Point * points = block.GetPoints();

            Int32 sum[NUM_DIMENSIONS] = { 0 };
            for (Int32 i = 0; i < block.GetNumPoints(); i++) {
                for (Int32 j = 0; j < NUM_DIMENSIONS; j++) {
                    sum[j] += points[i].x[j];
                }
            }

            Point averagePoint;
            for (Int32 j = 0; j < NUM_DIMENSIONS; j++) {
                if (block.GetNumPoints()>0) {
                    averagePoint.x[j] = UInt8(sum[j] / block.GetNumPoints());
                }
                else {
                    averagePoint.x[j] = 0;
                }
            }

                static_cast<void>(VECTOR_PUSH(result, averagePoint));


        }
        return result;
    }
}}
