//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GENERIC_RUN_LENGTH_ENCODED_BITMAP_H)
#define GENERIC_RUN_LENGTH_ENCODED_BITMAP_H

#include <Candera/Environment.h>
#include <CanderaPlatform/Device/Common/BitmapConverter/GenericBitmapConvertor.h>
#include <CanderaPlatform/Device/Common/BitmapConverter/GenericBitmapAccessInternal.h>

namespace Candera
{
    //xxx: gkoefn: spelling mistake, should be GenericBitmapConverter
    namespace GenericBitmapConvertor
    {
        ///////////////////////////////////////////////////////////////////////////////
        //  Encoder
        ///////////////////////////////////////////////////////////////////////////////

        //Class definitions
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        class RunLengthEncodingContainer;

        /**
          * @brief Generic data accessor.
          */
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        class RunLengthEncodingAccessor;

        /**
          * @brief Iterator of a Destination Run Length Encoded Bitmap
          */
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        class RunLengthEncodingIterator
        {
        public:
            typedef RunLengthEncodingAccessor<HostType, WordType, Endianness, PixelEndianness> Accessor;

            /**
              * Constructor.
              * @param host container to integrate through.
              */
            RunLengthEncodingIterator(RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>& host);

            /**
              * Increment the iterator to point to the next item within the container.
              * @return the iterator after incrementation
              */
            RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness> operator++ ();

            /**
              * Increment the iterator to point to the next item within the container.
              * @return the iterator before incrementation
              */
            RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness> operator++ (Internal::PostIncrementType);

            /**
              * Retrieve the item pointed to.
              * @return an item accessor
              * @see RunLengthEncodingAccessor.
              */
            Accessor operator* ();

            /**
              * Get the offset in bytes from the start of the container.
              * @return number of bytes from the start of the container.
              */
            UInt32 GetOffset() const;

            /**
              * Checks whether the iterator has gone passed the end of the container.
              * @return true if the iterator has gone passed the end of the container.
              */
            bool IsAtEnd() const;

            friend class RunLengthEncodingAccessor<HostType, WordType, Endianness, PixelEndianness>;
        private:
            RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>& m_host;
            UInt32 m_currentCount;

            //forbid assignment by making it private
            RunLengthEncodingIterator& operator=(const RunLengthEncodingIterator&);
        };

        /**
          * @brief Data accessor of a Destination Run Length Encoded Bitmap
          */
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        class RunLengthEncodingAccessor
        {
        public:
            /**
              * Write the item.
              * @param val the value of the item packed within an integer.
              */
            void set(HostType val);

            friend class RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness>;
        private:
            RunLengthEncodingAccessor(RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness>& source);
            RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness>& m_source;

            //forbid assignment by making it private
            RunLengthEncodingAccessor& operator=(const RunLengthEncodingAccessor&);
        };

        /**
          * @brief Wrapper for a Destination Run Length Encoded Bitmap
          */
        template <typename THostType, typename WordType, typename Endianness, typename PixelEndianness>
        class RunLengthEncodingContainer
        {
        public:
            typedef THostType HostType;
            typedef RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness> Iterator;

            enum {
                c_bitmapHeaderSize = 2U * sizeof(UInt32), ///< Size of the bitmap header.
                c_blockHeaderSize = 8, ///< Size of a block header.
                c_blockCompressedFlag = static_cast<UInt32>(1) << (c_blockHeaderSize - 1), ///< First bit in the header indicates whether the block is compressed or not.                
                c_maxPixelsPerBlock = c_blockCompressedFlag ///< The other bits in the header indicate the number of pixels in the block.
            };
            /**
              * Constructor.
              * @param buffer bitmap data.
              * @param size describes the number of bits.
              * @param bpp number of bits per pixel.
              */
            RunLengthEncodingContainer(WordType* buffer, UInt32 size, UInt32 bpp, UInt32 countBase);

            /**
              * Retrieve an iterator to the first pixel of the bitmap.
              * @return iterator to the first pixel.
              */
            RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness> Begin();

            /**
              * Finalize processing the bitmap.
              */
            void End();

            friend class RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness>;
            friend class RunLengthEncodingAccessor<HostType, WordType, Endianness, PixelEndianness>;

        private:
            WordType* m_buffer;
            UInt32 m_size;
            UInt32 m_bpp;
            UInt32 m_countBase;

            UInt32 m_pixelCursor;

            UInt32 m_currentCount;
            UInt32 m_headerPos;
            UInt32 m_bufPos;

            UInt32 m_offset;
            UInt32 m_end;

            UInt32 m_criticalCompressCount;

            HostType m_value;

            /**
              * Add a new pixel to the bitmap. If the iterator is ahead or
              * behind the compression cursor, the value is ignored.
              * @param iterator pixel count of the current iterator.
              * @param val pixel to add.
              */
            void set(UInt32 iterator, HostType val);

            /**
              * Write the compressed data to the bitmap.
              */
            void Compress();
            /**
              * Write the uncompressed data to the bitmap.
              */
            void Burn();
            /**
              * Update uncompressed data header to match the burnt data.
              */
            void Commit();

            UInt32 GetMaxPixelsPerBlock() { return c_maxPixelsPerBlock - m_countBase; }

            WordType* GetBufferStart();
        };

        //Template method definitions

        //Iterator
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness>::RunLengthEncodingIterator(RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>& host) :
            m_host(host),
            m_currentCount(0)
        {
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness> RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness>::operator++ ()
        {
            m_currentCount ++;
            return *this;
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness> RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness>::operator++ (Internal::PostIncrementType)
        {
            RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness> prev(*this);
            operator ++();
            return prev;
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        typename RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness>::Accessor RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness>::operator* ()
        {
            return Accessor(*this);
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        UInt32 RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness>::GetOffset() const
        {
            return (m_host.m_bufPos + m_host.m_offset) >> 3;
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        bool RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness>::IsAtEnd() const
        {
            return (m_currentCount > m_host.m_pixelCursor) ||
                ((m_currentCount == m_host.m_pixelCursor) &&
                    (m_host.m_bufPos > m_host.m_end));
        }

        //Byte accessor
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        void RunLengthEncodingAccessor<HostType, WordType, Endianness, PixelEndianness>::set(HostType val)
        {
            m_source.m_host.set(m_source.m_currentCount, val);
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        RunLengthEncodingAccessor<HostType, WordType, Endianness, PixelEndianness>::RunLengthEncodingAccessor(RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness>& source) :
            m_source(source) {}

        //Container
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>::RunLengthEncodingContainer(WordType* buffer, UInt32 size, UInt32 bpp, UInt32 countBase) :
            m_buffer(buffer),
            m_size(size),
            m_bpp(bpp),
            m_countBase(countBase),
            m_pixelCursor(0),
            m_currentCount(0),
            m_headerPos(0),
            m_bufPos(0),
            m_offset(((c_bitmapHeaderSize + sizeof(WordType)) << 3) - 1),
            m_end(((size - c_bitmapHeaderSize) << 3) - (c_blockHeaderSize + bpp)),
            m_criticalCompressCount((c_blockHeaderSize/bpp) + 2),
            m_value()
        {
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness> RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>::Begin()
        {
            return RunLengthEncodingIterator<HostType, WordType, Endianness, PixelEndianness>(*this);
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        void RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>::End()
        {
            if (m_currentCount > m_criticalCompressCount) {
                Compress();
            }
            else{
                Burn();
                Commit();
            }

            UInt32* header = reinterpret_cast<UInt32*>(m_buffer);
            Internal::Access::SetBitmapBytes<UInt32, UInt32, LittleEndian>(
                header, 0, c_bitmapHeaderSize);
            Internal::Access::SetBitmapBytes<UInt32, UInt32, LittleEndian>(
                header, 32, m_headerPos);
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        void RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>::set(UInt32 iterator, HostType val)
        {
            //only allow writing of pixels in order.
            //fail in all other cases.
            if (iterator != m_pixelCursor++){
                return;
            }
            val = Internal::EndiannessConverter<HostType, PixelEndianness>::FromHost(val);

            if (((m_currentCount) != 0) && (val == m_value)) {
                if (m_currentCount == GetMaxPixelsPerBlock()) {
                    Compress();
                }
                m_currentCount ++;
                if (m_currentCount > m_criticalCompressCount) {
                    Commit();
                }
            }
            else {
                if (m_currentCount > m_criticalCompressCount) {
                    Compress();
                }
                else {
                    Burn();
                }
                m_value = val;
                m_currentCount = 1;
            }
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        void RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>::Compress() {
            FEATSTD_DEBUG_ASSERT(m_headerPos == m_bufPos); //there should be no uncommitted Burn

            //write compressed header
            UInt8 header = (static_cast<UInt8>((m_currentCount - 1) + m_countBase) | 
                static_cast<UInt8>(c_blockCompressedFlag));
            Internal::Access::SetBitmapBits<UInt8, WordType, Endianness>(GetBufferStart(), m_bufPos, c_blockHeaderSize, header);
            m_bufPos += c_blockHeaderSize;
            //write compressed data
            Internal::Access::SetBitmapBits<HostType, WordType, Endianness>(GetBufferStart(), m_bufPos, m_bpp, m_value);
            m_bufPos += m_bpp;

            //update cursor position
            m_headerPos = m_bufPos;
            m_currentCount = 0;
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        void RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>::Burn() {
            if (m_currentCount == 0) {
                //no value to Burn
                return;
            }
            UInt32 writeCount;
            if (m_headerPos == m_bufPos) {
                //this is the first Burn. leave room for header.
                m_bufPos += c_blockHeaderSize;

                writeCount = m_currentCount;
                m_currentCount = 0;
            }
            else {
                UInt32 count = m_bufPos - (m_headerPos + c_blockHeaderSize);
                FEATSTD_DEBUG_ASSERT(count % m_bpp == 0); //must hold items
                count /= m_bpp;

                if ((count + m_currentCount) > GetMaxPixelsPerBlock()) {
                    writeCount = GetMaxPixelsPerBlock() - count;
                }
                else {
                    writeCount = m_currentCount;
                }
                m_currentCount -= writeCount;
            }
            while (writeCount > 0) {
                Internal::Access::SetBitmapBits<HostType, WordType, Endianness>(GetBufferStart(), m_bufPos, m_bpp, m_value);
                m_bufPos += m_bpp;
                writeCount --;
            }
            if (m_currentCount != 0) {
                Commit();
                Burn();
            }
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        void RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>::Commit() {
            if (m_headerPos == m_bufPos) {
                //no Burn
                return;
            }
            UInt32 count = m_bufPos - (m_headerPos + c_blockHeaderSize);
            FEATSTD_DEBUG_ASSERT(count % m_bpp == 0); //must hold items
            count /= m_bpp;
            FEATSTD_DEBUG_ASSERT(count > 0 && count <= GetMaxPixelsPerBlock()); //must hold countable items.

            //write uncompressed header
            UInt8 header = static_cast<UInt8>((count - 1) + m_countBase);
            Internal::Access::SetBitmapBits<UInt8, WordType, Endianness>(GetBufferStart(), m_headerPos, c_blockHeaderSize, header);

            m_headerPos = m_bufPos;
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        WordType* RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>::GetBufferStart()
        {
            void* untypedData = reinterpret_cast<UInt8*>(m_buffer) + c_bitmapHeaderSize;
            return reinterpret_cast<WordType*>(untypedData);
        }

        ///////////////////////////////////////////////////////////////////////////////
        //  Decoder
        ///////////////////////////////////////////////////////////////////////////////

        //Class definitions
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        class RunLengthDecodingContainer;

        /**
          * @brief Generic data accessor.
          */
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        class RunLengthDecodingAccessor;

        /**
          * @brief Iterator of a Source Run Length Encoded Bitmap
          */
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        class RunLengthDecodingIterator
        {
        public:
            typedef RunLengthDecodingAccessor<HostType, WordType, Endianness, PixelEndianness> Accessor;

            enum {
                c_blockHeaderSize = RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>::c_blockHeaderSize,
                c_blockCompressedFlag = RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>::c_blockCompressedFlag,
                c_maxPixelsPerBlock = RunLengthEncodingContainer<HostType, WordType, Endianness, PixelEndianness>::c_maxPixelsPerBlock
            };

            /**
              * Constructor.
              * @param host container to integrate through.
              * @param bufPos the position of the buffer
              */
            RunLengthDecodingIterator(const RunLengthDecodingContainer<HostType, WordType, Endianness, PixelEndianness>& host, UInt32 bufPos);

            /**
              * Increment the iterator to point to the next item within the container.
              * @return the iterator after incrementation
              */
            RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness> operator++ ();
            /**
              * Increment the iterator to point to the next item within the container.
              * @return the iterator before incrementation
              */
            RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness> operator++ (Internal::PostIncrementType);

            /**
              * Get the offset in bytes from the start of the container.
              * @return number of bytes from the start of the container.
              */
            UInt32 GetOffset() const;

            /**
              * Checks whether the iterator has gone passed the end of the container.
              * @return true if the iterator has gone passed the end of the container.
              */
            bool IsAtEnd() const;

            /**
              * Retrieve the item pointed to.
              * @return an item accessor.
              * @see RunLengthDecodingAccessor
              */
            const Accessor operator* () const;

            friend class RunLengthDecodingAccessor<HostType, WordType, Endianness, PixelEndianness>;
        private:
            const RunLengthDecodingContainer<HostType, WordType, Endianness, PixelEndianness>& m_host;

            UInt32 m_bufPos;
            UInt32 m_currentCount;
            bool m_walk;

            HostType get() const;

            //forbid assignment by making it private
            RunLengthDecodingIterator& operator=(const RunLengthDecodingIterator&);
        };

        /**
          * @brief Data accessor of a Source Run Length Encoded Bitmap
          */
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        class RunLengthDecodingAccessor
        {
        public:
            /**
              * Retrieves the item.
              * @return the value of the item packed within an integer.
              */
            HostType get() const;

            friend class RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>;
        private:
            RunLengthDecodingAccessor(const RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>& source);
            const RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>& m_source;

            //forbid assignment by making it private
            RunLengthDecodingAccessor& operator=(const RunLengthDecodingAccessor&);
        };

        /**
          * @brief Wrapper for a Source Run Length Encoded Bitmap
          */
        template <typename THostType, typename WordType, typename Endianness, typename PixelEndianness>
        class RunLengthDecodingContainer
        {
        public:
            typedef THostType HostType;
            typedef RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness> Iterator;

            /**
              * Constructor.
              * @param buffer bitmap data.
              * @param size describes the number of bits.
              * @param bpp number of bits per pixel.
              */
            RunLengthDecodingContainer(const WordType* buffer, UInt32 size, UInt32 bpp, UInt32 countBase);

            /**
              * Retrieve an iterator to the first pixel of the bitmap.
              * @return iterator to the first pixel.
              */
            RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness> Begin() const;

            /**
              * Finalize processing the bitmap.
              */
            void End() const;

            friend class RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>;
            friend class RunLengthDecodingAccessor<HostType, WordType, Endianness, PixelEndianness>;
        private:
            const WordType* m_buffer;
            UInt32 m_size;
            UInt32 m_bpp;
            UInt32 m_countBase;

            UInt32 m_offset;
            UInt32 m_end;

            UInt32 GetBufferSize() const;
            const WordType* GetBufferStart() const;
        };

        //Template method definitions

        //Iterator
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>::RunLengthDecodingIterator(const RunLengthDecodingContainer<HostType, WordType, Endianness, PixelEndianness>& host, UInt32 bufPos) :
            m_host(host), m_bufPos(bufPos - host.m_bpp), m_currentCount(0), m_walk(false)
        {
            operator ++ ();
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness> RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>::operator++ ()
        {
            if (m_currentCount == 0) {
                m_bufPos += m_host.m_bpp;
                if (m_bufPos == m_host.GetBufferSize()) {
                    //reached end of buffer
                    return *this;
                }

                //start of new block. read block header
                FEATSTD_DEBUG_ASSERT(m_bufPos + c_blockHeaderSize <= m_host.GetBufferSize());
                UInt8 val = Internal::Access::GetBitmapBits<UInt8, WordType, Endianness>(m_host.GetBufferStart(), m_bufPos, c_blockHeaderSize);
                m_bufPos += c_blockHeaderSize;

                const UInt8 typedBlockCompressedFlag = static_cast<UInt8>(c_blockCompressedFlag);
                if ((val & typedBlockCompressedFlag) != 0) {
                    //compressed
                    m_walk = false;
                    val &= ~typedBlockCompressedFlag;
                }
                else {
                    //uncompressed
                    m_walk = true;
                }

                m_currentCount = val - m_host.m_countBase;
            }
            else {
                //just walk the current block
                --m_currentCount;

                if (m_walk) {
                    //for uncompressed values advance with each step
                    m_bufPos += m_host.m_bpp;
                }
            }
            return *this;
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness> RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>::operator++ (Internal::PostIncrementType)
        {
            RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness> prev(*this);
            operator ++();
            return prev;
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        const typename RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>::Accessor RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>::operator* () const
        {
            return Accessor(*this);
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        UInt32 RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>::GetOffset() const
        {
            return (m_bufPos + m_host.m_offset) >> 3;
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        bool RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>::IsAtEnd() const
        {
            return (m_currentCount == 0) && (m_bufPos >= m_host.m_end);
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        HostType RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>::get() const
        {
            FEATSTD_DEBUG_ASSERT(m_bufPos + m_host.m_bpp <= m_host.GetBufferSize());
            HostType pixel = Internal::Access::GetBitmapBits<HostType, WordType, Endianness>(
                m_host.GetBufferStart(),
                m_bufPos,
                m_host.m_bpp);
            return Internal::EndiannessConverter<HostType, PixelEndianness>::ToHost(pixel);
        }

        //Byte accessor
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        HostType RunLengthDecodingAccessor<HostType, WordType, Endianness, PixelEndianness>::get() const
        {
            return m_source.get();
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        RunLengthDecodingAccessor<HostType, WordType, Endianness, PixelEndianness>::RunLengthDecodingAccessor(const RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>& source) :
            m_source(source) {}

        //Container
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        RunLengthDecodingContainer<HostType, WordType, Endianness, PixelEndianness>::RunLengthDecodingContainer(const WordType* buffer, UInt32 size, UInt32 bpp, UInt32 countBase) :
            m_buffer(buffer),
            m_size(size),
            m_bpp(bpp),
            m_countBase(countBase),
            m_offset(0),
            m_end(0)
        {
            const UInt32* header = reinterpret_cast<const UInt32*>(buffer);
            UInt32 headerSize = 
                Internal::Access::GetBitmapBytes<UInt32, UInt32, LittleEndian>(header, 0);
            UInt32 bufferSize =
                Internal::Access::GetBitmapBytes<UInt32, UInt32, LittleEndian>(header, 32);
            UInt32 bitSize = (size - headerSize) << 3;
            if (bitSize > bufferSize) {
                bitSize = bufferSize;
            }
            m_end = bitSize;
            // Decoding iterator points to the address before the start
            // of the start of the data by bpp bits.
            m_offset = (((headerSize + sizeof(WordType)) << 3) + bpp) - 1 ;
        }

        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness> RunLengthDecodingContainer<HostType, WordType, Endianness, PixelEndianness>::Begin() const
        {
            return RunLengthDecodingIterator<HostType, WordType, Endianness, PixelEndianness>(*this, 0);
        }
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        void RunLengthDecodingContainer<HostType, WordType, Endianness, PixelEndianness>::End() const
        {
        }
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        UInt32 RunLengthDecodingContainer<HostType, WordType, Endianness, PixelEndianness>::GetBufferSize() const
        {
            return reinterpret_cast<const UInt32*>(m_buffer)[1];
        }
        template <typename HostType, typename WordType, typename Endianness, typename PixelEndianness>
        const WordType* RunLengthDecodingContainer<HostType, WordType, Endianness, PixelEndianness>::GetBufferStart() const
        {
            return FeatStd::Internal::PointerAdd(m_buffer, FeatStd::Internal::PointerToPointer<const UInt32*>(m_buffer)[0]);
        }
    } //namespace GenericBitmapConvertor
} //namespace Candera

#endif // GENERIC_RUN_LENGTH_ENCODED_BITMAP_H

