//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERAPLATFORM_SIMULATED_DISPLAY_H)
    #define CANDERAPLATFORM_SIMULATED_DISPLAY_H

#include <CanderaPlatform/OS/CanderaTypes.h>

namespace Candera {

/** @addtogroup CommonDevice
 *  @{
 */

/**
 *  @brief This interface may be implemented by a package instance simulating displays on another native display. The application
 *  can use this interface to assume control about this simulated diplay in order to control certain aspects which
 *  usually are performed by the native display.
 */
class SimulatedDisplay {
    public:
        /**
         *  Constructor
         */
        SimulatedDisplay() {}

        /**
         *  Destructor
         */
        virtual ~SimulatedDisplay() {}

        typedef void (*CallbackFunction)(void* callbackData);


        /**
         *  If a display is simulated inside another display window. The application might want to specify a view port
         *  so that the display maintains the correct aspect ratio in relation to the native display. The application
         *  has the ability to zoom and pan the simulated display according to it's needs.
         *  @param left     The left corner of the view port in pixels relative to the native display / window.
         *  @param bottom   The bottom corner of the view port in pixels relative to the native display / window.
         *  @param width    The width of the display in pixels.
         *  @param height   The height of the display in pixels.
         */
        virtual void SetViewport(Int left, Int bottom, Int width, Int height) = 0;

        /**
         *  If the simulated display has an own drawing context, this context is enabled. The application can then
         *  issue custom rendering on the simulated display surface.
         *  @return         true if activation was successful. false is returned if the simulated display does not support this
         *                  feature or if activation fails.
         */
        virtual bool ActivateContext() = 0;

        /**
         *  If the display dimensions shall change of the simulated display this method needs to be called so that
         *  certain pixel related values can be scaled appropriately to the specified view port.
         *  @param width    The width of the simulated display to set.
         *  @param height   The height of the simulated display to set.
         */
        virtual void SetDisplayDimensions(Int width, Int height) = 0;

        /**
         *  By default automatic update of the display is enabled (as it would occur at native displays). Native displays
         *  however are kicked by the display controller timer whereas a simulated display is kicked whenever a change of
         *  content is detected.
         *  To manually control display update automatic kick can be disabled.
         *  @param enable   Enables auto kick.
         */
        virtual void SetAutoKickEnabled(bool enable) = 0;

        /**
         *  Manually push the contents of the display to the native display.
         *  @return         true if the off-screen-display was sent to native display, false otherwise.
         */
        virtual bool KickDisplayUpdate() = 0;

        /**
         * Sets a callback that will be executed before the display will be kicked
         * @param function      Function to be called on pre kick display.
         * @param callbackData  Data to pass to function.
         */
        virtual void SetPreKickDisplayCallback(CallbackFunction function, void* callbackData) = 0;

        /**
         * Retrieves the callback along with the callback data (if there are any set)
         * @param callback      Call function that's called on pre kick display.
         * @param callbackData  Data that was passed last to the callback.
         */
        virtual void GetPreKickDisplayCallback(CallbackFunction& callback, void* &callbackData) = 0;
    };

/** @}*/ //end of CommonDevice

}   // namespace Candera

#endif  // CANDERAPLATFORM_SIMULATED_DISPLAY_H
