//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERAPLATFORM_RENDERDEVICE_2D_H)
    #define CANDERAPLATFORM_RENDERDEVICE_2D_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Common/Bitmap.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/TextEngine/GlyphBitmap.h>
#include <Candera/System/MetaInfo/EnumDataTypeBase.h>

namespace Candera {

/** @addtogroup CommonDevice
 *  @{
 */

class Matrix3x2;
class Vector3;
class Vector4;
class Matrix4;
class Rectangle;
class GenericBitmapFormat;
class Camera2D;

class BitmapImage2D;
class VertexBuffer2D;

/**
*  Surfaces are arbitrary image objects which map to a specified screen or off-screen boundary. Surface derived
*  objects can be either used to draw to or used to read pixel data, typically referenced by x, and y coordinates.
*/
typedef Handle SurfaceHandle;
typedef Handle ContextHandle2D;
typedef Handle GeometryHandle;
typedef UInt BufferHandle;

class VectorialRenderData {

public:
    UInt32 vertexCount;
    BufferHandle vertexBuffer;
    UInt8* vertexData;

    UInt32 indexCount;
    BufferHandle indexBuffer;
    UInt8* indexData;

    VectorialRenderData() :
        vertexCount(0),
        vertexBuffer(0),
        vertexData(0),
        indexCount(0),
        indexBuffer(0),
        indexData(0),
        m_targetPathData(0),
        m_targetPathSegments(0),
        m_segmentsCount(0),
        m_pathDataCount(0)
    {
    }

    void AppendNewData(VectorialRenderData& newData, Int16 x, Int16 y, Int16 w, Int16 h);
    void ScaleVertexCoords(Float scaleX, Float scaleY);
    void TranslateVertexCoords(Float translateX, Float translateY);

    float* m_targetPathData;
    UInt8* m_targetPathSegments;
    UInt32 m_segmentsCount;
    UInt32 m_pathDataCount;

private:


};

namespace Internal {
    class RenderDevice2DHelper;
}

/**
 * @brief   The RenderDevice2D encapsulates calls to the graphics device driver interface.
 */
class RenderDevice2D {
    public :
        typedef MemoryManagement::Disposer<const void*>::DisposerFunction DisposerFn;

        /**
         * List of surface types.
         */
        enum SurfaceType{
            DestinationSurface = 0, ///< Destination surface.
            SourceSurface = 1,      ///< Source surface.
            MaskSurface = 2         ///< Mask surface.
        };

        /**
         * Factor used in modulating the surface color before blend.
         */
        enum BlendFactor {
            Zero = 0,                   ///< Zero
            One  = 1,                   ///< One
            SourceColor = 2,            ///< SourceColor
            InverseSourceColor = 3,     ///< InverseSourceColor
            SourceAlpha = 4,            ///< SourceAlpha
            InverseSourceAlpha = 5,     ///< InverseSourceAlpha
            DestColor = 6,              ///< DestColor
            InverseDestColor = 7,       ///< InverseDestColor
            DestAlpha = 8,              ///< DestAlpha
            InverseDestAlpha = 9        ///< InverseDestAlpha
        };

        static inline const Char* GetBlendFactorEditor() {
            return ENUM_DATA_TYPE_EDITOR_BEGIN(Filter)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(Zero, 0)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(One, 1)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(SourceColor, 2)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(InverseSourceColor, 3)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(SourceAlpha, 4)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(InverseSourceAlpha, 5)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(DestColor, 6)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(InverseDestColor, 7)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(DestAlpha, 8)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(InverseDestAlpha, 9)
            ENUM_DATA_TYPE_EDITOR_END();
        }

        /**
         * Operation used when blending surfaces.
         */
        enum BlendOperation {
            Add = 0,             ///< The result will be the sum of the objects. (Src + Dst)
            Subtract = 1,        ///< The result will be the difference of the objects. (Src - Dst)
            ReverseSubtract = 2, ///< The result will be the reversed difference of the objects. (Dst - Src)
            BitwiseAnd = 3       ///< The result will be the bitwise and of the objects. (Src & Dst)
        };

        static inline const Char* GetBlendOperationEditor() {
            return ENUM_DATA_TYPE_EDITOR_BEGIN(Filter)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(Add, 0)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(Subtract, 1)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(ReverseSubtract, 2)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(BitwiseAnd, 3)
            ENUM_DATA_TYPE_EDITOR_END();
        }



        /**
          *  BlendMode combines the source and destination blend factors with the operation used in blending equation.
          *  Color blending equation: final color = source color *  colorBlendFactorSrc (+colorBlendOperation+) destination color *  colorBlendFactorDst.
          *  alpha blending equation: final alpha = source alpha *  alphaBlendFactorSrc (+alphaBlendOperation+) destination alpha *  alphaBlendFactorDst.
          *  Default values are: colorBlendFactorSrc = SourceAlpha, colorBlendFactorDst = InverseSourceAlpha, colorBlendOperation = Add,
          *                      alphaBlendFactorSrc = One, alphaBlendFactorDst = Zero, alphaBlendOperation = Add.
          */
        struct BlendMode {
            BlendFactor colorBlendFactorSrc;
            BlendFactor colorBlendFactorDst;
            BlendOperation colorBlendOperation;
            BlendFactor alphaBlendFactorSrc;
            BlendFactor alphaBlendFactorDst;
            BlendOperation alphaBlendOperation;

            BlendMode() : colorBlendFactorSrc(SourceAlpha), colorBlendFactorDst(InverseSourceAlpha), colorBlendOperation(Add),
                          alphaBlendFactorSrc(One), alphaBlendFactorDst(Zero), alphaBlendOperation(Add) {}
        };

        /**
         * Filtering mode used for minification and/or magnification of surfaces.
         */
        enum Filter {
            NearestFilter = 0,
            BilinearFilter = 1,
            AutomaticFilter = 2
        };

        static inline const Char* GetFilterEditor() {
            return ENUM_DATA_TYPE_EDITOR_BEGIN(Filter)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(NearestFilter, 0)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(BilinearFilter, 1)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(AutomaticFilter, 2)
            ENUM_DATA_TYPE_EDITOR_END();
        }

        /**
         * Filtering mode used for mipmapping.
         */
        enum MipMapFilter {
            DisabledMipMapFilter,   ///< Disables mipmapping.
            NearestMipMapFilter,    ///< MipMapping uses nearest mipmap level.
            LinearMipMapFilter      ///< MipMapping uses bilinear mipmap interpolation of the two nearest mipmap levels.
        };

        static inline const Char* GetMipMapFilterEditor() {
            return ENUM_DATA_TYPE_EDITOR_BEGIN(MipMapFilter)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(DisabledMipMapFilter, 0)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(NearestMipMapFilter, 1)
                ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(LinearMipMapFilter, 2)
            ENUM_DATA_TYPE_EDITOR_END();
        }

        /**
         *  Different pack order types.
         */
        enum PackOrder {
            UnknownPackOrder,                   ///< Order unknown.
            DisplayPackOrder,                   ///< Left to right, top to bottom.
            VerticallyFlippedDisplayPackOrder   ///< Left to right, bottom to top.
        };

        /**
         * Link a surface object to one of the surface types.
         * @param context   2D context obtained from RenderTarget2D.
         * @param sidx      Surface type.
         * @param sname     Surface handle returned by surface manager.
         * @return          true on success, false on failure.
         */
        static bool SetSurface(ContextHandle2D context, SurfaceType sidx, SurfaceHandle sname);


        /**
        * Assign the vectorialData to be used by this context.
        * @param context        2D context obtained from RenderTarget2D.
        * @param vectorialData  object which holds the vectorial data.
        * @return               true on success, false on failure.
        */
        static bool SetVectorialRenderData(ContextHandle2D context, const VectorialRenderData& vectorialData);

        /**
         * Set a transformation matrix for a certain surface.
         * @param context   2D context obtained from RenderTarget2D.
         * @param sidx      Surface type.
         * @param mat       Transformation matrix.
         * @return          true on success, false on failure.
         */
        CANDERA_DEPRECATED_3_2_1("Function has been moved. Please use Renderer2D::SetTransformationMatrix(ContextHandle2D context, SurfaceType sidx, const Matrix3x2& mat) instead.",
        static bool SetTransformationMatrix(ContextHandle2D context, SurfaceType sidx, const Matrix3x2& mat));

        /**
         * Set the blend operation used when blitting a certain surface.
         * @param context 2D context obtained from RenderTarget2D.
         * @param sidx          Surface type.
         * @param op            Blend operation.
         * @param srcFactor     Source blend factor.
         * @param dstFactor     Destination blend factor.
         * @return              true on success, false on failure.
         */
        static bool SetBlendOperation(ContextHandle2D context, SurfaceType sidx, BlendOperation op, BlendFactor srcFactor, BlendFactor dstFactor);

        /**
         * Set the blend operation used for the color channels when blitting a certain surface.
         * @param context       2D context obtained from RenderTarget2D.
         * @param sidx          Surface type.
         * @param op            Blend operation.
         * @param srcFactor     Source blend factor.
         * @param dstFactor     Destination blend factor.
         * @return              true on success, false on failure.
         */
        static bool SetColorBlendOperation(ContextHandle2D context, SurfaceType sidx, BlendOperation op, BlendFactor srcFactor, BlendFactor dstFactor);

        /**
         * Set the blend operation used for the alpha channel when blitting a certain surface.
         * @param context 2D context obtained from RenderTarget2D.
         * @param sidx          Surface type.
         * @param op            Blend operation.
         * @param srcFactor     Source blend factor.
         * @param dstFactor     Destination blend factor.
         * @return              true on success, false on failure.
         */
        static bool SetAlphaBlendOperation(ContextHandle2D context, SurfaceType sidx, BlendOperation op, BlendFactor srcFactor, BlendFactor dstFactor);

        /**
         * Set the color used for modulating surface color, or for filling when no surface is specified.
         * @param context 2D context obtained from RenderTarget2D.
         * @param sidx      Surface type.
         * @param r         Red color channel.
         * @param g         Green color channel.
         * @param b         Blue color channel.
         * @param a         Alpha color channel.
         * @return          true on success, false on failure.
         */
        static bool SetSurfaceConstColor(ContextHandle2D context, SurfaceType sidx, Float r, Float g, Float b, Float a);

        /**
         * Set the matrix used for surface color transformation.
         * @param context   2D context obtained from RenderTarget2D.
         * @param sidx      Surface type.
         * @param matrix    Color transformation matrix.
         * @return          true on success, false on failure.
         */
        static bool SetColorTransformation(ContextHandle2D context, SurfaceType sidx, const Matrix4& matrix);

        /**
         * Set the active area of a certain surface used in clearing, filling and blitting.
         * The active area of the destination surface specifies the view port clipping rectangle.
         * The active area of the source surface is the area used for filling if no surface is used.
         * If a surface is used, and no transformation is applied to the surface, the active area of
         * the source surface represents the only area taken into account when blitting.
         * In all other situations, the implementation is platform specific.
         * @param context   2D context obtained from RenderTarget2D.
         * @param sidx      Surface type.
         * @param left      X Axis start of active area. (default is 0)
         * @param top       Y Axis start of active area. (default is 0)
         * @param width     X Axis size of active area. If negative it is replaced by maximum Float value. (default is -1)
         * @param height    Y Axis size of active area. If negative it is replaced by maximum Float value. (default is -1)
         * @return          true on success, false on failure.
         */
        static bool SetActiveArea(ContextHandle2D context, SurfaceType sidx, Float left, Float top, Float width, Float height);

        /**
         * Get the active area of the specified surface.
         * If the area was set using negative values for width and height, the correct values are computed.
         * @param context   2D context obtained from RenderTarget2D.
         * @param sidx      Surface type.
         * @param left      Pointer to Left coordinate of return rectangle (out parameter).
         * @param top       Pointer to Top coordinate of return rectangle (out parameter).
         * @param width     Pointer to Width of return rectangle (out parameter).
         * @param height    Pointer to Height of return rectangle (out parameter).
         * @return          true on success, false on failure.
         */
        static bool GetActiveArea(ContextHandle2D context, SurfaceType sidx, Float *left, Float *top, Float *width, Float *height);

        /**
         * Set filtering mode used when minifying or magnifying images.
         * @param context   2D context obtained from RenderTarget2D.
         * @param filter    filtering mode to be set.
         * @return          true on success, false on failure.
         */
        static bool SetFilter(ContextHandle2D context, Filter filter);

        /**
         * Set filtering mode used when sampling mip-maps.
         * @param context   2D context obtained from RenderTarget2D.
         * @param filter    filtering mode to be set.
         * @return          true on success, false on failure.
         */
        static bool SetMipMapFilter(ContextHandle2D context, MipMapFilter filter);

        /**
        * Returns if the implementation of RenderDevice2D supports MipMapping or not.
        * @return Returns if the implementation of RenderDevice2D supports MipMapping (true) or not (false).
        */
        static bool IsMipMappingSupported();

        /**
         * Blit the surfaces attached, using the states set to the context.
         * @param context   2D context obtained from RenderTarget2D.
         * @return          true on success, false on failure.
         */
        CANDERA_DEPRECATED_3_2_1("Function has been moved. Please use Renderer2D::Blit(ContextHandle2D context) instead.",
        static bool Blit(ContextHandle2D context));

        /**
        * Set vectorial mode flag for the context.
        * @param context   2D context obtained from RenderTarget2D.
        * @param enabled   true if vectorial mode must become active, false otherwise.
        * @return          true on success, false on failure.
        */
        static bool SetVectorialModeActive(ContextHandle2D context, bool enabled);

        /**
         * Clear the active area of the destination surface to the specified color.
         * @param context 2D context obtained from RenderTarget2D.
         * @param r     Red color channel.
         * @param g     Green color channel.
         * @param b     Blue color channel.
         * @param a     Alpha channel.
         * @return      true on success, false on failure.
         */
        CANDERA_DEPRECATED_3_2_1("Function has been moved. Please use Renderer2D::Clear(ContextHandle2D context, Float r, Float g, Float b, Float a) instead.",
        static bool Clear(ContextHandle2D context, Float r, Float g, Float b, Float a));

        /**
         * Get the area modified by the last blit or clear command.
         * Changing the view port to this context invalidates the value.
         * @param context   2D context obtained from RenderTarget2D.
         * @param left      Pointer to Left coordinate of return rectangle (out parameter).
         * @param top       Pointer to Top coordinate of return rectangle (out parameter).
         * @param width     Pointer to Width of return rectangle (out parameter).
         * @param height    Pointer to Height of return rectangle (out parameter).
         * @return          A rectangle defining the view port space updated area.
         */
        static bool GetUpdatedArea(ContextHandle2D context, Float *left, Float *top, Float *width, Float *height);

        /**
         *  Upload a bitmap image 2D.
         *  The surface handles generated during upload are attached to the
         *  image.
         *  @param image    Image to unload.
         *  @return         true on success, false otherwise.
         */
        CANDERA_DEPRECATED_3_3_0("Use Renderer2D::UploadBitmapImage2D instead.",
            static bool UploadBitmapImage2D(BitmapImage2D& image)
         );
        /**
         *  Unload a bitmap image 2D.
         *  All resources created during upload will be released.
         *  @param image    Image to unload.
         *  @return         true on success, false otherwise.
         */
        CANDERA_DEPRECATED_3_3_0("Use Renderer2D::UnloadBitmapImage2D instead.",
            static bool UnloadBitmapImage2D(BitmapImage2D& image)
        );

        /**
        *  Creates a number of surfaces and store the data within the table specified by pSurface
        *  @param count     Number of surfaces to create.
        *  @param pSurface  Pointer to the first element of a table containing at least count items.
        *  @return          true on success, false otherwise.
        */
        static bool CreateSurfaces(Int count, SurfaceHandle *pSurface);

        /**
        *   Creates hardware level handles for the vectorial Data.
        *   @param vectorialData    vectorial data object for which hardware handles will be created.
        *   @return                 true on success, false otherwise.
        */
        static bool CreateVectorialRenderData(VectorialRenderData& vectorialData);

        /**
        *  Creates and assigns a buffer to a surface. If the optional parameter addressBuffer
        *  is specified, the buffer is populated with data retrieved from addressBuffer.
        *  @param surface               Surface to which to assign the buffer
        *  @param uiWidth               Width of the buffer.
        *  @param uiHeight              Height of the buffer.
        *  @param iPixelFormat          Pixel format of the buffer.
        *  @param eColorPackAlignment   Pack alignment of the source data (unused if addressBuffer is 0).
        *  @param eColorPackOrder       Pack order of the source data (unused if addressBuffer is 0).
        *  @param addressBuffer         Source data to be copied to the newly created buffer.
        *  @return                      true on success, false otherwise.
        */
        static bool CreateBuffer(SurfaceHandle surface, UInt32 uiWidth, UInt32 uiHeight, Int iPixelFormat, Bitmap::PackAlignment eColorPackAlignment, RenderDevice2D::PackOrder eColorPackOrder, const void* addressBuffer);

        /**
         *  Creates and assigns a buffer to a surface. This function is provided to
         *  offer device packages a chance to optimize surface generation when rendering text.
         *  @param surface              Surface to which to assign the buffer
         *  @param width                Width of the source data.
         *  @param height               Height of the source data.
         *  @param stride               Stride in bytes of the source data.
         *  @param format               Format as defined by GlyphBitmap.
         *  @param addressBuffer        Source data to be copied to the newly created buffer.
         *  @return                     True on success, false otherwise.
         */
        static bool CreateBuffer(SurfaceHandle surface, UInt32 width, UInt32 height, UInt32 stride,
                                 TextRendering::GlyphBitmap::Format format, const void* addressBuffer);

        /**
        *  Creates and assign data to a vectorial data object.
        *  @param vectorialData         Vectorial data object to be populated with generated data.
        *  @param pathCommandsBuffer    The command buffer used to generate the vectorial data.
        *  @param left                  x offset to be set to the generated data.
        *  @param top                   y offset to be set to the generated data.
        *  @param width                 width within which to generate data.
        *  @param height                height withing which to generate data.
        *  @return                      True on success, false otherwise.
        */
        static bool GenerateVectorialRenderData(VectorialRenderData& vectorialData, const void* pathCommandsBuffer, Int32 left, Int32 top, Int32 width, Int32 height);

        /**
        *  Uploads vectorial data to video memory
        *  @param vectorialData         Vectorial data object whose data will be uploaded to video memory.
        *  @return                      True on success, false otherwise.
        */
        static bool UploadVectorialData(VectorialRenderData vectorialData);

        /**
        *  Assigns the data from addressBuffer to the surface. If the address range pointed to by addressBuffer
        *  is in the proper memory space, addressBuffer will be used directly as buffer. If the address range
        *  is not proper, a buffer will be automatically created, the data copied, and the addressBuffer disposed.
        *  If disposer is not 0, addressBuffer will be disposed of before the buffer is detached, or the surface
        *  is destroyed. If addressBuffer is NULL, the surface will behave like having assigned a white buffer,
        *  having the properties described by the parameters.
        *  Changing the data within the addressBuffer while the buffer is still attached results in undefined
        *  behavior.
        *  @param surface               Surface to which to attach the buffer.
        *  @param width               Width of the buffer.
        *  @param height              Height of the buffer.
        *  @param iPixelFormat          PixelFormat of the buffer.
        *  @param eColorPackAlignment   Pack alignment of the source data (unused if addressBuffer is 0).
        *  @param eColorPackOrder       Pack order of the source data (unused if addressBuffer is 0).
        *  @param addressBuffer         Source data to be copied to the newly created buffer.
        *  @param disposerFn            Disposer function used to dispose the addressBuffer.
        *  @return                      true on success, false otherwise.
        */
        static bool AssignBuffer(SurfaceHandle surface, UInt32 width, UInt32 height, Int iPixelFormat,
                                Bitmap::PackAlignment eColorPackAlignment, RenderDevice2D::PackOrder eColorPackOrder, const void* addressBuffer,
                                DisposerFn disposerFn);

        /**
        * Updates a certain region from the given surface. The given data.
        *
        *  @param surface       Surface in which texture subimage is updated.
        *  @param xOffset       The x-Offset inside the Texture.
        *  @param yOffset       The y-Offset inside the Texture.
        *  @param width         The width to update. The width relates also to the width of the source data.
        *  @param height        The height to update. The height relates also to the height of the source data.
        *  @param data          The source data to update. The source data must have the same pack order and pack
        *                       alignment as the data supplied to RenderDevice2D::CreateBuffer, which was used to
        *                       create the surface.
        *  @return              True if succeeded, False if failed.
        */
        static bool UpdateSubImage(SurfaceHandle surface, Int xOffset, Int yOffset, UInt width, UInt height, const UInt8* data);

        /**
        *  Detaches the buffer assigned to the surface.
        *  @param surface   Targeted surface.
        *  @return          true on success, false otherwise.
        */
        static bool DetachBuffer(SurfaceHandle surface);

        /**
        *  Destroys count surfaces from pSurface.
        *  @param count     Number of surfaces to destroy.
        *  @param pSurface  Pointer to the first element of a table containing at least count surface handles.
        *  @return          true on success, false otherwise.
        */
        static bool DestroySurfaces(Int count, SurfaceHandle* pSurface);

        /**
        *   Destroys hardware level handles for the vectorial Data.
        *   @param vectorialRenderData vectorial data object for which hardware handles will be destroyed.
        *   @return                    true on success, false otherwise.
        */
        static bool DestroyVectorialDataHandles(VectorialRenderData& vectorialRenderData);

        /**
        *  Deletes vectorial data from a vectorial data object.
        *  @param vectorialRenderData   Vectorial data object to have it's data deleted.
        *  @return                      True on success, false otherwise.
        */
        static bool DestroyVectorialDataBuffers(VectorialRenderData& vectorialRenderData);
        /**
        *  Get width of the surface.
        *  @param pSurface  Surface to get width from.
        *  @return          Width of the surface.
        */
        static UInt32 GetWidth(SurfaceHandle pSurface);
        /**
        *  Get height of the surface.
        *  @param pSurface  Surface to get height from.
        *  @return          Height of the surface.
        */
        static UInt32 GetHeight(SurfaceHandle pSurface);
        /**
        *  Get RenderDevice2D::PackOrder of the surface.
        *  @param pSurface  Surface to get RenderDevice2D::PackOrder from.
        *  @return          RenderDevice2D::PackOrder of the surface.
        */
        static RenderDevice2D::PackOrder GetPackOrder(SurfaceHandle pSurface);

        /**
         * Get Pixel Format of the surface.
         * @param pSurface  Surface to get pixel format from.
         * @return          Pixel format of the surface.
         */
        static Int GetPixelFormat(SurfaceHandle pSurface);

        /**
         * Gets a new off-screen context. Upon calling this function, any old context handle retrieved
         * with this function will become invalid and should not be used. The new context should not
         * influence the surface attached to an old context. A second call to this function since the
         * last call to GetNewOffscreenSurfaceHandle also invalidates the off-screen surface handle.
         * The new off-screen context is made active. In order to return to drawing to the regular render
         * target you need to reactivate it.
         * The size of the off-screen surface attached to the context is always grater or equal to the size
         * of the render target it replaces.
         * Typical usage would be: \code
         *     GetNewOffscreenContextHandle()
         *     draw to context
         *     GetNewOffscreenSurfaceHandle()
         *     repeat
         *         GetNewOffscreenContextHandle()
         *         draw to context using last off-screen surface
         *         GetNewOffscreenSurfaceHandle()
         *     activate render target
         *     draw to render target using last off-screen surface
         *  \endcode
         * @param contextHandle     Context handle for the owner of the off-screen surface.
         * @param rectangle         Minimum rectangle that needs to be covered by the off-screen render target.
         * @return                  ContextHandle2D to a new off-screen context.
         */
        static ContextHandle2D GetNewOffscreenContextHandle(ContextHandle2D contextHandle, const Rectangle& rectangle);

        /**
         * Gets a surface handle attached to the current off-screen context.
         * This handle should not be used as long as the attached off-screen context is active.
         * The surface should be retrieved only as long as the context is active.
         * @param contextHandle context handle for the owner of the off-screen surface.
         *
         * @return SurfaceHandle to a new off-screen surface.
         */
        static SurfaceHandle GetNewOffscreenSurfaceHandle(ContextHandle2D contextHandle);

        /**
         * Uploads off-screen surfaces.
         * Should be called from render target upload, in order to prepare the off-screen surfaces
         * for effect chaining.
         *
         * @param contextHandle context handle for the owner of the off-screen surface.
         * @param width  Width of the off-screen surface.
         * @param height Height of the off-screen surface.
         *
         * @return True if uploading of off-screen surfaces succeeded. False if creating or uploading of any off-screen surface failed.
         */
        static bool UploadOffscreenSurfaces(ContextHandle2D contextHandle, UInt32 width, UInt32 height);

        /**
         * Unloads off-screen surfaces.
         * For each successful upload an unload should be called. Calls are counted, end only the last
         * unload frees the actual data.
         * @param contextHandle context handle for the owner of the off-screen surface.
         *
         * @return False if there is no off-screen surface to Unload. True if upload count is decreased. When upload count
         *         reached zero off-screen surfaces are unloaded. In this case also true is returned.
         */
        static bool UnloadOffscreenSurfaces(ContextHandle2D contextHandle);

        /**
         *  Attach a native surface handle to a Candera 2D surface handle (a native handle is
         *  a texture handle the in case of an OpenGL based graphic package.)
         *  @param surfaceHandle surface create with RenderDevice2D::CreateSurface
         *  @param uiWidth width of the surface
         *  @param uiHeight height of the surface
         *  @param iPixelFormat Pixel format of the surface
         *  @param nativeHandle native handle of the surface, to be attached to the surface handle
         *  @return True if successful, otherwise false.
         */
        static bool AttachNativeHandle(SurfaceHandle surfaceHandle, UInt32 uiWidth, UInt32 uiHeight, Int iPixelFormat, Handle nativeHandle);

        /**
         *  Converts bitmaps form one Bitmap Format to another, including platform specific formats.
         *  @param dstWidth       width of the destination bitmap
         *  @param dstHeight      height of the destination bitmap
         *  @param dstPixelFormat format of the destination bitmap
         *  @param dstBuffer      buffer of the destination bitmap. The caller must guarantee the buffer is large enough
         *  @param srcWidth       width of the source bitmap
         *  @param dstPackAlignment
         *  @param dstBuffer
         *  @param srcHeight      height of the source bitmap
         *  @param srcPixelFormat format of the source bitmap
         *  @param srcPackAlignment
         *  @param srcBuffer      buffer of the source bitmap
         *  @return               True if successful, otherwise false.
         */
        static bool ConvertBitmap(UInt32 dstWidth, UInt32 dstHeight, Int dstPixelFormat, Bitmap::PackAlignment dstPackAlignment, UInt8* dstBuffer,
                                  UInt32 srcWidth, UInt32 srcHeight, Int srcPixelFormat, Bitmap::PackAlignment srcPackAlignment, const UInt8* srcBuffer);


        /**
         * Enables perspective warping for the passed context, if available.
         * @param context The context where perspective warping should be enabled.
         * @param enable  True to enable perspective warping, false to disable.
         * @return        True if perspective warping is available on the passed context / on the current target.
         *                False otherwise.
         */
        static bool SetPerspectiveWarpingEnabled(ContextHandle2D context, bool enable);

        /**
         * Queries if perspective warping is enabled or not.
         * @param context The context to query if perspective warping is enabled or not.
         * @return True if perspective warping is enabled on the passed context. False if it is disabled.
         *         On targets / context where perspective warping is not available the result is always false, even if
         *         SetPerspectiveWarpingEnabled(ContextHandle2D context, bool enable); was set to true.
         */
        static bool IsPerspectiveWarpingEnabled(ContextHandle2D context);

        /**
         * Sets the transformation matrix for perspective warping, which is basically a full 3D transformation.
         * This transformation matrix only applies to source surfaces.
         * This matrix only gets evaluated if
         *    - Perspective warping is available on the passed context / target.
         *    - Perspective warping was enabled before using SetPerspectiveWarpingEnabled function.
         * @param context The context to set the perspective transformation matrix to.
         * @param mat The transformation matrix to set.
         * @return True if the matrix was set, otherwise false. False includes the case where perspective warping is not available.
         */
        static bool SetPerspectiveTransformationMatrix(ContextHandle2D context, const Matrix4& mat);

        /**
         * Sets the projection matrix for perspective warping, which is basically a full 3D projection.
         * This projection matrix only applies to source surfaces.
         * This matrix only gets evaluated if
         *    - Perspective warping is available on the passed context / target.
         *    - Perspective warping was enabled before using SetPerspectiveWarpingEnabled function.
         * @param context The context to set the perspective projection matrix to.
         * @param mat The transformation matrix to set.
         * @return True if the matrix was set, otherwise false. False includes the case where perspective warping is not available.
         */
        static bool SetPerspectiveProjectionMatrix(ContextHandle2D context, const Matrix4& mat);

        /**
         * Sets the viewport used for perspective warping.
         * Classic 2D viewports behave more like a scissor rectangle together with an applied offset.
         * In 3D the viewport also describes how the whole content is scaled in x and y dimensions.
         * In order to allow 2D camera independent perspective viewports it can be set separately.
         * The 2D camera viewport defines the origin of the rendering and the scissoring rectangle for the perspective warping nodes.
         * If the viewport dimensions are set to default values (x, y, -1, -1), render target size is used as viewport size, while the 2D
         * camera defines the origin of the rendering, and the scissor rectangle.
         * @param context The context to set the perspective viewport to.
         * @param viewport The viewport matrix to set.
         * @return True if the viewport was set, otherwise false. False includes the case where perspective warping is not available.
         */
        static bool SetPerspectiveViewport(ContextHandle2D context, const Rectangle& viewport);

        /**
         * Calculates a full 3D transformation matrix out of position, rotation, scale and pivot.
         * @param matrix Out parameter that holds the created matrix.
         * @param position: World position of the images pivot point.
         * @param rotation: Rotation around pivot point.
         * @param scale     Scale around pivot point.
         * @param pivot:    Pivot point of the image. This point is the center of rotations and scale. x, y is in pixels. z to be handled like in OpenGL.
         * @return:         True if the matrix was computed. False otherwise, including that the current platform doesn't support perspective warping.
         */
        static bool CalculatePerspectiveTransformationMatrix(Matrix4& matrix, const Vector3& position, const Vector3& rotation, const Vector3& scale, const Vector3& pivot);


        /**
         * Calculates a full 3D projection matrix out of near plane, far plane, field of view angle and aspect ratio.
         * @param matrix Out parameter that holds the created matrix.
         * @param camera      Camera that determines the current aspect ration.
         * @param viewport
         * @param nearZ       Near plane of the projection frustum.
         * @param farZ        Far plane of the projection frustum.
         * @param fovY        Field of view Y - angle of the projection frustum.
         * @param aspectRatio
         * @return            True if the matrix was computed. False otherwise, including that the current platform doesn't support perspective warping.
         */
        static bool CalculatePerspectiveProjectionMatrix(Matrix4& matrix, const Camera2D* camera, const Rectangle& viewport, Float nearZ, Float farZ, Float fovY, Float aspectRatio);

        /**
         * Returns the size the pixel array will occupy in VRAM.
         * @param bitmap
         * @param memoryPool Defines the memory pool for which this bitmap is counted.
         *                   Default value (=0) is for "VideoMemory".
         * @return Returns the size the pixel array will occupy in VRAM.
         */
        static UInt32 GetSize(const Bitmap* bitmap, UInt memoryPool = 0);

        /**
         * Set the scissor rectangle.
         * @param context           The context to set the scissor rectangle.
         * @param scissorRectangle  The scissor rectangle to set.
         * @return                  True on success, false on failure.
         */
        static bool SetScissorRectangle(ContextHandle2D context, const Rectangle& scissorRectangle);

        /**
         * Set the scissor rectangle.
         * @param context           The context to get the scissor rectangle.
         * @param scissorRectangle  The out parameter to return the scissor rectangle in.
         * @return                  True on success, false on failure.
         */
        static bool GetScissorRectangle(ContextHandle2D context, Rectangle& scissorRectangle);

        /**
         * Enable or disable scissoring.
         * @param context    The context to enable or disable scissoring.
         * @param isEnabled  True to enable scissoring, false to disable scissoring.
         * @return           True on success, false on failure.
         */
        static bool SetScissorEnabled(ContextHandle2D context, bool isEnabled);

        /**
         * Returns whether scissor is enabled or disabled.
         * @param context    The context to enable or disable scissoring.
         * @return           True if scissor is enabled, false if scissor is disabled.
         */
        static bool IsScissorEnabled(ContextHandle2D context);


        //Functions for handling VertexBuffer2D.

        /**
         * Sets the geometry to the given context. Unsetting means setting geometry = 0.
         * @param context  The context to set the geometry to.
         * @param geometry The geometry to set.
         * @return Whether setting was successful(true) or not (false).
         */
        static bool SetGeometry(ContextHandle2D context, GeometryHandle geometry, const Rectangle& boundingRectangle = Rectangle());

        /**
        * Gets the geometry to the given context.
        * @param context The context to query the geometry from.
        * @return Whether querying the geometry was successful(true) or not (false).
        */
        static GeometryHandle GetGeometry(ContextHandle2D context);

        /**
         * Updates the 2D vertex buffer in memory.
         * @param vb The vertex buffer to update.
         * @param offset The offset to start updating from in bytes.
         * @param size The size of block to write into the vertex buffer.
         * @data The new vertex data.
         * @return Whether updating was succesful(true) or not(false).
         */
        static bool UpdateVertexBuffer2D(VertexBuffer2D& vb, Int offset, Int size, const UInt8* data);

        /**
        * Returns the size the VertexBuffer2D will occupy in the defined memorypools.
        * @param vb The VertexBuffer2D to check for size.
        * @param memoryPool Defines the memory pool for which this bitmap is counted.
        *                   Default value (=0) is for "VideoMemory".
        * @return Returns the size the VertexBuffer will occupy in the defined memory pool in bytes.
        */
        static UInt32 GetVertexBufferSize(const VertexBuffer2D* vb, UInt memoryPool = 0);

        /**
         * Queries if the given PrimitiveType is supported by the device.
         * @param primitiveType The primitive type to check.
         * @result Whether the primitive type is supported or not.
         */
        static bool IsPrimitiveTypeSupported(const Int primitiveType);

        /**
        * Queries if index buffers supported by the device.
        * @result Whether index buffers are supported or not.
        */
        static bool IsIndexBufferSupported();

    private:
        friend class Renderer2D;

        static bool UploadVertexBuffer2D(VertexBuffer2D& vb);
        static bool UnloadVertexBuffer2D(VertexBuffer2D& vb);

        friend class Candera::Internal::RenderDevice2DHelper;
        static bool HasBuffer(SurfaceHandle surface);

};//class RenderDevice2D

/** @}*/ //end of CommonDevice

namespace Internal {
    class RenderDevice2DHelper
    {
    public:
        static bool HasBuffer(SurfaceHandle surface)
        {
            return RenderDevice2D::HasBuffer(surface);
        }
    };
}

} // namespace Candera

#endif  // CANDERAPLATFORM_RENDERDEVICE_2D_H
