//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERAPLATFORM_RENDERDEVICE_H)
    #define CANDERAPLATFORM_RENDERDEVICE_H

#include <Candera/Environment.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/EngineBase/Common/Bitmap.h>
#include <Candera/System/Container/Vector.h>

#ifdef CANDERA_3D_ENABLED
    #include <Candera/Engine3D/Core/VertexBuffer.h>
    #include <Candera/Engine3D/Core/ShaderParamNames.h>
    #include <Candera/Engine3D/Core/Shader.h>
    #include <Candera/Engine3D/Core/Light.h>
    #include <Candera/Engine3D/Core/CubeMapTextureImage.h>
#ifdef CANDERA_SHADER_PROGRAM_PERSIST_INTERFACE_ENABLED
    #include <CanderaPlatform/Device/Common/OpenGLES/GlShaderStorageInterface.h>
#endif
    #include <Candera/Engine3D/Core/SynchronizationFence.h>
#else
    #include <CanderaPlatform/Device/Common/Stubs/DummyCandera3D.h>
#endif

namespace Candera {

/** @addtogroup CommonDevice
 *  @{
 */

class RenderTarget3D;
class Texture;
class TextureImage;
class Color;
class EGLRenderTarget;
class Viewport;
class Vector4;
class Camera;
class Rectangle;
class RenderBuffer;
struct RenderStateCache;
struct TextureStateCache;
class ContextProvider3D;
class BitmapTextureImage;
class DirectTextureImage;
class ExternalTextureImage;
class VertexBuffer;
class Scene;
class Query;

/**
 * @brief   The RenderDevice encapsulates calls to the graphics device driver interface and render API (OpenGL ES).
 */
class RenderDevice {
    public:
        /**
         * Render states managed by this RenderDevice class.
         */
        enum RenderState
        {
            BlendingEnable,                 ///< Blending enabled
            CullFaceMode,                   ///< Cull face mode.
            DepthWriteEnable,               ///< Depth write enable.
            DepthTestEnable,                ///< Depth test enable.
            DepthComparisonFunction,        ///< Depth comparison function.
            DitheringEnable,                ///< Dithering enable.
            ScissorTestEnable,              ///< Scissor test enable.
            SampleCoverageMaskEnable,       ///< Sample coverage mask enable.
            SampleAlphaToCoverageMaskEnable,///< Sample alpha to coverage mask enable.
            StencilTestEnable,              ///< Stencil test enable.
            WindingOrder,                   ///< Winding order.
            CoverageWriteEnable,            ///< Supported by extension "GL_NV_coverage_sample".
            RasterizerDiscardEnable,        ///< Supported by OpenGL ES3.0 and later.
            NumberOfRenderStates            ///< Number of render states.
        };

        /**
         * Values that can be passed to RenderDevice::ResetCurrentRenderStateCache for single cached state reseting.
         *  Each cached render state must be reset if it is changed through other means than by calling the RenderDevice
         *  corresponding method.
         */
        enum RenderStateCacheResyncOption {
            ResyncClearColorOption,                         ///<See RenderDevice::ActivateClearColor
            ResyncClearDepthOption,                         ///<See RenderDevice::ActivateDepthClearValue
            ResyncClearStencilOption,                       ///<See RenderDevice::ActivateStencilClearValue
            ResyncColorWriteOption,                         ///<See RenderDevice::ActivateColorWriteEnabled
            ResyncBlendColorOption,                         ///<See RenderDevice::ActivateBlendColor
            ResyncBlendModeOption,                          ///<See RenderDevice::ActivateBlendMode and RenderDevice::ActivateBlendModeSeparate
            ResyncSampleCoverageOption,                     ///<See RenderDevice::ActivateSampleCoverage
            ResyncStencilFunctionOption,                    ///<See RenderDevice::ActivateStencilFunction
            ResyncStencilOperationOption,                   ///<See RenderDevice::ActivateStencilOperation
            ResyncStencilWriteMaskOption,                   ///<See RenderDevice::ActivateStencilWriteMask
            ResyncCullFaceOption,                           ///<See RenderDevice::ActivateRenderState with CullFaceMode render state
            ResyncDitheringEnabledOption,                   ///<See RenderDevice::ActivateRenderState with DitheringEnable render state
            ResyncWindingOrderOption,                       ///<See RenderDevice::ActivateRenderState with WindingOrder render state
            ResyncDepthWriteEnabledOption,                  ///<See RenderDevice::ActivateRenderState with DepthWriteEnable render state
            ResyncDepthTestEnabledOption,                   ///<See RenderDevice::ActivateRenderState with DepthTestEnable render state
            ResyncDepthComparisonFunctionOption,            ///<See RenderDevice::ActivateRenderState with DepthComparisonFunction render state
            ResyncStencilTestEnabledOption,                 ///<See RenderDevice::ActivateRenderState with StencilTestEnable render state
            ResyncScissorTestEnabledOption,                 ///<See RenderDevice::ActivateRenderState with ScissorTestEnable render state
            ResyncBlendingEnabledOption,                    ///<See RenderDevice::ActivateRenderState with BlendingEnable render state
            ResyncSampleCoverageMaskOption,                 ///<See RenderDevice::ActivateRenderState with SampleCoverageMaskEnable render state
            ResyncSampleAlphaToCoverageMaskEnabledOption,   ///<See RenderDevice::ActivateRenderState with SampleAlphaToCoverageMaskEnable render state
            ResyncRasterizerDiscardEnabledOption,           ///<See RenderDevice::ActivateRenderState with RasterizerDiscardEnable render state
            ResyncDepthBiasOption,                          ///<See RenderDevice::ActivateDepthBias
            ResyncViewportOption,                           ///<See RenderDevice::ActivateViewport
            ResyncScissorOption,                            ///<See RenderDevice::ActivateScissorRectangle
            ResyncActiveShaderOption,                       ///<See RenderDevice::ActivateShader
            ResyncActiveTextureImagesOption,                ///<See RenderDevice::ActivateTextureImage
            ResyncActiveVertexBufferOption,                 ///<See RenderDevice::ActivateVertexBuffer
            ResyncVertexAttribArrayOption,                  ///<Caches enabled/disabled state of vertex attribute arrays
            ResyncAll
        };

        /**
         * DeviceInfo specifies the information to retrieve by function GetDeviceInfo.
         */
        enum DeviceInfo
        {
            VendorInfo = 0,                 ///< The vendor information formatted for human consumption. No specific format set.
            RendererInfo = 1,               ///< The renderer information formatted for human consumption. No specific format set.
            ApiVersionInfo = 2,             ///< The API version info. In OpenGL ES this is "OpenGL ES <version> <vendor-specific information>".
            ShadingLanguageVersionInfo = 3, ///< The shading language version. In  OpenGL ES this is "OpenGL ES GLSL ES <version> <vendor-specific information>".
            ExtensionsInfo = 4              ///< All extensions supported by device. This is a space-separated list.
        };

        enum HintTarget
        {
            MipMapGeneration = 0,           ///< Indicates the quality of filtering when generating mipmap images.
            DerivativeCalculation = 1       ///< Indicates the accuracy of the derivative calculation in some built-in shader functions (dFdx, dFdy and fwidth).
        };

        enum HintMode
        {
            Fastest = 0,                    ///< The most efficient option should be chosen.
            Nicest = 1,                     ///< The most correct, or highest quality, option should be chosen.
            DontCare = 2                    ///< No preference.
        };

        /// Destructor
        ~RenderDevice();

        /* Texture functions ***********************************************************************************************/

        /**
         * Get the device specific bits per pixel of a given bitmap pixel format.
         * @param  pixelFormat  The pixel format of a bitmap.
         * @return              The number of bits per pixel that is being used on the device for the given bitmap pixel format.
         */
        static UInt8 GetBitsPerPixel(Bitmap::PixelFormat pixelFormat);

        /**
         * Returns the size the pixel array will occupy in VRAM.
         * @return Returns the size the pixel array will occupy in VRAM.
         */
        static UInt32 GetSize(const Bitmap* bitmap, TextureImage::TextureMemoryPool memoryPool = TextureImage::VideoMemory);

        /**
         * Upload a Texture given to VRAM.
         * Handle to texture memory is generated during upload and will be set to the given texture.
         * Postcondition on success: Texture is active.
         * @param  textureImage The texture image to upload to VRAM.
         * @param  unit         Texture to be activated and associated to textureImage.
         * @return              True if uploading and activating texture succeeded. False otherwise, also fails if
         *                      bitmap of texture image is not valid.
         */
        CANDERA_DEPRECATED_3_3_0("Use Renderer::UploadBitmapTextureImage instead.",
            static bool UploadBitmapTextureImage(BitmapTextureImage& textureImage, UInt unit)
        );

        /**
         * Upload a CubeMap texture given to VRAM.
         * Handle to texture memory is generated during upload and will be set to the given texture.
         * Postcondition on success: Texture is active.
         * If mipmapping is enabled, custom bitmap chains will only be uploaded if every bitmap has one such chain attached.
         * Otherwise the mipmaps are generated.
         * @param  textureImage The texture image to upload to VRAM.
         * @param  unit         Texture to be activated and associated to textureImage.
         * @return              True if uploading all six cube map faces and activating the texture succeeded. False otherwise, also fails if
         *                      any bitmap of any face is not valid.
         */
        CANDERA_DEPRECATED_3_3_0("Use Renderer::UploadCubeMapTextureImage instead.",
            static bool UploadCubeMapTextureImage(CubeMapTextureImage& textureImage, UInt unit)
        );

        /**
         * Unload a Texture from graphics device memory (VRAM).
         * @param  textureImage     Texture image to unload from VRAM.
         * @return                  True if texture image was unloaded.
         */
        CANDERA_DEPRECATED_3_3_0("Use Renderer::UnloadBitmapTextureImage instead.",
            static bool UnloadBitmapTextureImage(BitmapTextureImage& textureImage)
        );

         /**
          * Unload a CubeMap Texture from graphics device memory (VRAM).
          * @param  textureImage     Texture image to unload from VRAM.
          * @return                  True if texture image was unloaded.
          */
        CANDERA_DEPRECATED_3_3_0("Use Renderer::UnloadCubeMapTextureImage instead.",
            static bool UnloadCubeMapTextureImage(CubeMapTextureImage& textureImage)
        );

        /**
         * Activate and bind a texture to the texture unit given.
         * Postcondition: TextureImage is bound to unit given and unit is active.
         * @param textureImage Texture image to bind to the texture unit.
         * @param unit         Texture unit to bind texture image to.
         * @return             True if activating the given texture unit and binding the texture succeeded.
         */
        static bool ActivateTextureImage(const TextureImage& textureImage, UInt unit);

        /**
         * Deactivate a Texture at the texture unit given.
         * Postcondition: Texture unit doesn't have an assigned TextureImage.
         * @param unit      Texture unit to be cleared.
         * @param target    Target to be cleared.
         * @return          True if texture unit doesn't have an assigned TextureImage.
         */
        static bool DeactivateTexture(UInt unit, TextureImage::TextureTargetType target = TextureImage::Texture2D);

        /**
         * Set subimage within texture.
         * Note: TextureImage is set active.
         * @param textureImage Texture to be updated.
         * @param level        Mip level to update.
         * @param xOffset      X Index of texel to start updating texture from.
         * @param yOffset      Y Index of texel to start updating texture from.
         * @param width        Width of the image subregion to update.
         * @param height       Height of the image subregion to update.
         * @param data         The actual pixel data for the subregion of the image.
         * @param unit         Texture unit to be activated and updated.
         * @param compressedSize The size of the data array in bytes, needed for updating compressed images.
         *                     Note: Function will fail if image is compressed and size is wrong.
         * @return             True if Texture in texture unit "unit" was updated.
         */
        static bool SetTextureSubimage(const BitmapTextureImage& textureImage, UInt level, Int xOffset, Int yOffset,
                UInt width, UInt height, const UInt8* data, UInt unit = 0, UInt compressedSize = 0);

         /**
          * Set subimage within CubeMap texture face.
          * Note: TextureImage is set active.
          * @param textureImage Texture to be updated.
          * @param face         CubeMap face to update image.
          * @param level        Mip level to update.
          * @param xOffset      X Index of texel to start updating texture from.
          * @param yOffset      Y Index of texel to start updating texture from.
          * @param width        Width of the image subregion to update.
          * @param height       Height of the image subregion to update.
          * @param data         The actual pixel data for the subregion of the image.
          * @param unit         Texture unit to be activated and updated.
          * @param compressedSize The size of the data array in bytes, needed for updating compressed images.
          *                     Note: Function will fail if image is compressed and size is wrong.
          * @return             True if CubeMap texture face "face" in texture unit "unit" was updated.
         */
        static bool SetTextureSubimage(const CubeMapTextureImage& textureImage, CubeMapTextureImage::TargetFace face, UInt level,
            Int xOffset, Int yOffset, UInt width, UInt height, const UInt8* data, UInt unit = 0, UInt compressedSize = 0);

        /**
         * Copies pixels from framebuffer to texture.
         * Note: TextureImage is set active.
         * @param textureImage Texture to copy to.
         * @param xTex         X Index of texel to start updating texture from.
         * @param yTex         Y Index of texel to start updating texture from.
         * @param xFramebuffer X coordinate of a rectangles lower left corner, describing where to read from in the framebuffer.
         * @param yFramebuffer Y coordinate of a rectangles lower left corner, describing where to read from in the framebuffer.
         * @param width        Width in pixels of the region to read from in the framebuffer.
         * @param height       Height in pixels of the region to read from in the framebuffer.
         * @param unit         Texture unit to which texture shall be bound.
         * @return             True if image was copied from framebuffer to texture.
         */
        static bool CopyFromFramebufferToTexture(const BitmapTextureImage& textureImage, Int32 xTex, Int32 yTex,
                Int32 xFramebuffer, Int32 yFramebuffer, Int32 width, Int32 height, UInt unit = 0);

         /**
         * Copies pixels from framebuffer to a CubeMap texture face.
         * Note: TextureImage is set active.
         * @param textureImage Texture to copy to.
         * @param face         CubeMap face to copy framebuffer to.
         * @param xTex         X Index of texel to start updating texture from.
         * @param yTex         Y Index of texel to start updating texture from.
         * @param xFramebuffer X coordinate of a rectangles lower left corner, describing where to read from in the framebuffer.
         * @param yFramebuffer Y coordinate of a rectangles lower left corner, describing where to read from in the framebuffer.
         * @param width        Width in pixels of the region to read from in the framebuffer.
         * @param height       Height in pixels of the region to read from in the framebuffer.
         * @param unit         Texture unit to which texture shall be bound.
         * @return             True if image was copied from framebuffer to the CubeMap texture face "face".
         */
        static bool CopyFromFramebufferToCubeMapTextureFace(const CubeMapTextureImage& textureImage, CubeMapTextureImage::TargetFace face, Int32 xTex, Int32 yTex,
            Int32 xFramebuffer, Int32 yFramebuffer, Int32 width, Int32 height, UInt unit = 0);

        /**
         * Set texture wrap mode of texture given.
         * Precondition: TextureImage has to be activated first.
         * @param texture   Texture object containing wrap mode settings to set in OpenGL ES.
         * @return          True if setting wrap mode succeeded.
         */
        CANDERA_DEPRECATED_3_1_1("Use RenderDevice::ActivateTexture to set texture sampling states.",
            static bool SetTextureWrapMode(const Texture& texture)
        );

        /**
         * Set texture filter mode of texture given. This includes magnification, minification, mipmap filter. Further, the maximum
         * degree of anisotropy for anisotropic filtering is set, if supported by device.
         * Note: When mipmapping is disabled and mipmap filter is set by application nonetheless, the mipmap filter is ignored.
         * Precondition: TextureImage has to be activated first.
         * @param texture   Texture object containing settings for magnification, minification and mipmap filters to be set in OpenGL ES.
         * @return          True if texture filters could be set.
         */
        CANDERA_DEPRECATED_3_1_1("Use RenderDevice::ActivateTexture to set texture sampling states.",
            static bool SetTextureFilter(const Texture& texture)
        );

        /**
         * Set the texture level of detail (lod) clamping and bias parameters. MinLod and maxLod define the lower and upper limits for the calculated
         * mipmap level. The lod bias defines an offset from the calculated mipmap level.
         *
         * For example, if according to calculations mipmap level 1 should be used, but minLod is set to 2 then mipmap level 2 will be sampled.
         * Furthermore if additionally lodBias is set to 2 then mipmap level 4 will be sampled.
         *
         * @param texture   Texture object containing settings.
         * @return          True if setting texture lod parameters succeeded. On unsupported platforms this will always return true.
         */
        CANDERA_DEPRECATED_3_1_1("Use RenderDevice::ActivateTexture to set texture sampling states.",
            static bool SetTextureLodMode(const Texture& texture)
        );

        /**
         * Set or update the texture and sampling parameters such as filter modes, lod and wrap modes. Precondition: TextureImage has to be activated first.
         * If supported, shared sampler objects are used for sampling parameters, otherwise texture sampling parameters will be used.
         *
         * @param texture   Texture object containing settings.
         * @param unit
         * @return          True if setting texture and sampling parameters succeeded. False otherwise.
         */
        static bool ActivateTexture(const Texture& texture, UInt unit);

        /** Vertex Buffer functions *****************************************************************************************/

        /**
         * Upload VertexBuffer to graphics device memory (VRAM).
         * @param vertexBuffer  VertexBuffer object containing vertex geometry and index buffer to upload.
         * @return              True if vertex geometry and index buffer in vertex buffer are valid
         *                      and could be uploaded to VRAM.
         */
        CANDERA_DEPRECATED_3_3_0("Use Renderer::UploadVertexBuffer instead.",
            static bool UploadVertexBuffer(VertexBuffer& vertexBuffer)
        );

        /**
         * Unload VertexBuffer to graphics device memory (VRAM).
         * @param vertexBuffer  VertexBuffer object containing memory handles to VRAM memory to delete.
         * @return              True if VertexBuffer data are deleted from VRAM.
         */
        CANDERA_DEPRECATED_3_3_0("Use Renderer::UnloadVertexBuffer instead.",
            static bool UnloadVertexBuffer(VertexBuffer& vertexBuffer)
        );

        /**
         * Set the VertexBuffer given as active vertex buffer in render device.
         * @param  vertexBuffer     VertexBuffer containing handle to VRAM memory to bind as current array buffer.
         * @return                  True if vertexBuffer is bound as current array buffer.
         */
        static bool ActivateVertexBuffer(VertexBuffer& vertexBuffer);

        /**
         * Set subdata within vertex buffer
         * @param vertexBuffer  The vertex buffer on which subdata is modified.
         * @param offset        The offset into the buffer data store.
         * @param size          The size in bytes of the data that is modified in the buffer data store.
         * @param data          The data to modify in data store.
         * @return              True if vertexBuffer has been updated successfully.
         */
        static bool SetVertexBufferSubData(VertexBuffer& vertexBuffer, Int offset, Int size, const UInt8* data);

        /** Shader functions ************************************************************************************************/

        /**
         * Set the Shader given as active shader in render device.
         * @param programHandle     Handle to the shader program that should be activated.
         * @return                   True if shader is activated.
         */
        CANDERA_DEPRECATED_3_4_0("Use Renderer::ActivateShader(const Shader& shader, Handle programHandle) instead.",
            static bool ActivateShader(Handle programHandle)
            );

        /**
         * Provides a graphic platform dependent way to query if a shader object is available as source code or pre-compiled binary.
         * @param shader            The shader data to query the buildtype from.
         * @param shaderBuildType   Out parameter that returns the buildtype of the shader data.
         * @return                  Boolean that describes whether querying the shader build type was successful or not.
         */
        static bool GetShaderBuildType(const void* shader, Shader::BuildType& shaderBuildType);

        /**
         * Provides a graphic platform dependent way to query if a shader object is available as source code or pre-compiled binary.
         * @param shaderResourceHandle   The shader data to query the buildtype from.
         * @param shaderBuildType        Out parameter that returns the buildtype of the shader data.
         * @return                       Boolean that describes whether querying the shader build type was successful or not.
         */
        static bool GetShaderBuildType(const ResourceDataHandle& shaderResourceHandle, Shader::BuildType& shaderBuildType);

        /**
         * Provides a platform independent way to upload shader objects that get compiled from provided shader source.
         * Note: On some OpenGL ES 2.0 platforms this method might fail, as providing a shader compiler inside the driver is not mandatory.
         * @param objectType    Whether the shader object to upload and compile is a vertex or fragment shader.
         * @param shader        The shader object to upload and compile.
         * @param handle        Out parameter that returns the created shader handle.
         * @return              Boolean that describes whether the upload was successful or not.
         *                      If false the created handle is deleted.
         */
        static bool UploadShaderSource(Shader::ObjectType objectType, const void* shader, Handle& handle);

        /**
        * Provides a platform dependent way to upload shader objects that are provided as precompiled data.
        * @param objectType    Whether the shader object to upload is a vertex or fragment shader.
        * @param shader        The shader object to upload.
        * @param handle        Out parameter that returns the created shader handle.
        * @return              Boolean that describes whether the upload was successful or not. Also returns false if binary upload is not supported.
        *                      If false the created handle is deleted.
        */
        static bool UploadShaderBinary(Shader::ObjectType objectType, const void* shader, Handle& handle);

        /**
         * Provides a platform dependent way to upload complete shader programs, that come as binary data.
         * @param shader    The shader program to upload.
         * @param handle    Out parameter that returns the created program handle.
         * @return          Boolean that describes whether the upload was successful or not. Also returns false if binary upload is not supported.
         *                  If false the handle is deleted.
         */
        static bool UploadShaderBinaryProgram(const void* shader, Handle& handle);

#ifdef CANDERA_SHADER_PROGRAM_PERSIST_INTERFACE_ENABLED
        /**
        * Provides a platform dependent way to upload complete shader programs, that come as binary data from the shader storage.
        * @param shader           The shader program to upload.
        * @param storageObject    Shader storager object that contains binary shader program.
        * @return                 Boolean that describes whether the upload was successful or not. Also returns false if binary upload is not supported.
        */
        static bool UploadStoredShaderBinaryProgram(Shader& shader, GlShaderStorageInterface::StorageObject &storageObject);
        /**
        * Provides a platform dependent way to retrieve complete shader programs from the OpenGLES implementation.
        * @param shader    The shader program to upload.
        * @param Handle    Out parameter that returns the retrieved shader program.
        * @return          Boolean that describes whether the retrieval was successful or not. Also returns false if binary download is not supported.
        */
        static bool RetrieveShaderBinaryProgram(Shader& shader, GlShaderStorageInterface::StorageObject &storageObject);
#endif
        /**
         * Upload Shader to graphics device memory (VRAM).
         * @param shader  Shader object containing vertex and fragment shader or shader program to upload.
         * @return              True if shader objects are valid
         *                      and could be uploaded to VRAM.
         */
        CANDERA_DEPRECATED_3_3_0("Use Renderer::UploadShader instead.",
            static bool UploadShader(Shader& shader)
        );

        /**
         * Link vertex and fragment shader to a shader program.
         * @param  vertexShaderHandle   Handle to vertex shader to be linked.
         * @param  fragmentShaderHandle Handle to fragment shader to be linked.
         * @param  programHandle        Created handle to program to link vertex and fragment shader to, handle is out parameter.
         * @param  transformVaryings
         * @param  transformVaryingsCount
         * @return                      True if linking of vertex and fragment shaders to a shader program succeeded, False otherwise. If False the program is
         *                              deleted.
         */
        static bool LinkShader(Handle& vertexShaderHandle, Handle& fragmentShaderHandle, Handle& programHandle,
            const Char* const* transformVaryings = 0, SizeType transformVaryingsCount = 0);

        /**
         * Retrieve location of an attribute in the shader given.
         * @param programHandle     Handle to shader program to retrieve attribute location from.
         * @param attributeSemantic Gets resolved to attributeName, which is used for location query.
         * @return                  Location of shader attribute in the shader program. -1 indicates an error.
         */
        static Int GetAttributeLocation(Handle programHandle, ShaderParamNames::AttributeSemantic attributeSemantic);

        /**
         * Retrieve location of an attribute in the shader given.
         * @param programHandle     Handle to shader program to retrieve attribute location from.
         * @param attributeName     Name of attribute variable whose location will be queried.
         * @return                  Location of shader attribute in the shader program. -1 indicates an error.
         */
        static Int GetAttributeLocation(Handle programHandle, const Char* attributeName);

        /**
         * Returns the active attribute count in a shader program and stores it in the out-param count.
         * @param programHandle Handle to shader program to which receive active attribute count from.
         * @param count         Retrieved active attribute count, out parameter.
         * @return              True if active attribute count was retrieved and no error occurred.
         */
        static bool GetActiveAttributeCount(Handle programHandle, Int& count);

        /**
         * Returns the length of the longest active attribute variable name for programHandle including null termination character.
         * If no active attribute variables exist, 0 is returned.
         * @param programHandle Handle to the shader program.
         * @param maxLength Returns the length of longest attribute variable name.
         * @return Success if no error occurred.
         */
        static bool GetActiveAttributeMaxLength(Handle programHandle, Int& maxLength);

        /**
         * Returns an active attribute name of a shader program according the given index.
         * @param programHandle      Specifies the program object to be queried
         * @param index              Specifies the index of the attribute variable to be queried.
         * @param nameBufferSize     The maximum number of characters the RenderDevice is allowed to write in the character buffer indicated by name including the null terminator.
         * @param name               Returns a null terminated string containing the name of the attribute variable.
         * @param nameLength         Returns the actual number of characters written by RenderDevice in name string excluding the null terminator.
         * @param size               Returns the size of the attribute variable.
         * @param type               Returns the data type of the queried attribute.
         * @return                   True if query succeeded.
         */
        static bool GetActiveAttribute(Handle programHandle, Int index, Int nameBufferSize, Char* name /*out*/, Int& nameLength /*out*/, Int& size /*out*/, Shader::UniformType& type /*out*/);


        /**
         * Delete shader in graphic device memory (VRAM).
         * @param vertexShaderHandle    Handle to vertex shader to be deleted.
         * @param fragmentShaderHandle  Handle to fragment shader to be deleted.
         * @param programHandle         Handle to shader program to be deleted.
         * @return                      True if shaders were deleted.
         */
        CANDERA_DEPRECATED_3_3_0("Use Renderer::DeleteShader instead.",
            static bool DeleteShader(Handle vertexShaderHandle, Handle fragmentShaderHandle, Handle programHandle)
        );

        /**
         * Set uniforms of the shader given.
         * @param programHandle Handle to program whose uniform will be set.
         * @param name          Name of uniform to be set.
         * @param data          Data to be set as value of uniform.
         * @param type          Uniform data type.
         * @param count         Size of data to be set, shall be retrieved from GetActiveUniform in case of arrays.
         *                      Default value is 1, single values (e.g. one Float, one Matrix,...) count always as 1.
         * @return              True if uniform exists and value is set, false otherwise.
         */
        CANDERA_DEPRECATED_3_2_1("Function has been moved. Please use Renderer::SetUniformInRenderDevice(Handle programHandle, const Char* name, const void* data, Shader::UniformType type, UInt count) instead.",
        static bool SetUniform(Handle programHandle, const Char* name, const void* data, Shader::UniformType type, UInt count = 1));

        /**
         * Set uniforms of the active shader when the location of the uniform is known.
         * If the uniform data type can be determined at compile time, the optimized template
         * version of SetUniform should be used.
         * @param location Location of uniform to be set. Can be retrieved from GetUniformLocation.
         * @param data     Data to be set as value of uniform.
         * @param type     Uniform data type.
         * @param count    Size of data to be set, shall be retrieved from GetActiveUniform.
         *                 Default value is 1, single values (e.g. one Float, one Matrix,...) count always as 1.
         * @return         True if uniform exists and value is set, false otherwise.
         */
        static bool SetUniform(Int location, const void* data, Shader::UniformType type, UInt count = 1);

        /**
         * Set uniforms of the active shader when the location of the uniform is known. The uniform data type
         *  is passed as template argument.
         * @param location Location of uniform to be set. Can be retrieved from GetUniformLocation.
         * @param data     Data to be set as value of uniform.
         * @param count    Size of data to be set, shall be retrieved from GetActiveUniform.
         *                 Default value is 1, single values (e.g. one Float, one Matrix,...) count always as 1.
         * @return         True if uniform exists and value is set, false otherwise.
         */
        template <Shader::UniformType Type>
        static bool SetUniform(Int location, const void* data, UInt count = 1);

        /**
         * Retrieve location of an uniform in the shader given.
         * @param programHandle Handle to program from which a uniform location is queried.
         * @param name          Name of the uniform to get the location of.
         * @param location      Location of queried uniform as out parameter. If uniform doesn't exist or an error occurred
         *                      location becomes -1.
         * @return              location != -1
         */
        static bool GetUniformLocation (Handle programHandle, const Char* name, Int& location);

        /**
         * Retrieve location of an uniform in the shader given.
         * @param programHandle Handle to program from which a uniform location is queried.
         * @param name          Name of the uniform semantic to get the location of.
         * @param location      Location of queried uniform semantic as out parameter. If uniform doesn't exist or an error occurred
         *                      location becomes -1.
         * @return              location != -1
         */
        static bool GetUniformLocation (Handle programHandle, ShaderParamNames::UniformSemantic name, Int& location);

        /**
        * Returns the active uniform count in a shader program and stores it in the out-param count.
        * @param programHandle Handle to shader program to which receive active uniform count from.
        * @param count         Retrieved active uniform count, out parameter.
        * @return              True if active uniform count was retrieved and no error occurred.
        */
        static bool GetActiveUniformCount(Handle programHandle, Int& count);

        /**
         * Returns the length of the longest active uniform variable name for programHandle including null termination character.
         * If no active uniform variables exist, 0 is returned.
         * @param programHandle Handle to the shader program.
         * @param maxLength Returns the length of longest uniform variable name.
         * @return Success if no error occurred.
         */
        static bool GetActiveUniformMaxLength(Handle programHandle, Int& maxLength);

        /**
         * Returns an active uniform name of a shader program according the given index.
         * @param programHandle      Specifies the program object to be queried
         * @param index              Specifies the index of the uniform variable to be queried.
         * @param nameBufferSize     The maximum number of characters the RenderDevice is allowed to write in the character buffer indicated by name including the null terminator.
         * @param name               Returns a null terminated string containing the name of the uniform variable. A uniform array reports the uniform name only.
         *                           Structures and array of structures are reduced to its fundamental components containing "." and "[]" operators.
         *                           Thus, structures, an array of structures, or a subcomponent of a vector or matrix are no valid names.
         * @param nameLength         Returns the actual number of characters written by RenderDevice in name string excluding the null terminator.
         * @param size               Returns the size of the uniform variable. Uniform variables other than arrays have size 1. Arrays of structures are
         *                           reduced to its fundamental components including "." and "[]" operators, which have size 1, if they are no arrays.
         * @param type               Returns the data type of the queried uniform.
         * @return                   True if query succeeded.
         */
        static bool GetActiveUniform(Handle programHandle, Int index, Int nameBufferSize, Char* name /*out*/, Int& nameLength /*out*/, Int& size /*out*/, Shader::UniformType& type /*out*/);


        /**
         * Bind attribute
         * @param attribute     Attribute index of array to be bound to vertex element.
         * @param vertexElement Description of vertex element format describing the vertex attribute to bind.
         * @param stride        Delta between attribute data of two adjacent vertices.
         * @return              True if attribute was bound, false if attribute doesn't exist or another error occurred activating it.
         */
        static bool BindAttribute(Int attribute, const VertexGeometry::VertexElementFormat& vertexElement, UInt stride);

        /**
         * Bind attribute array pointer to vertex element
         * @param attribute     Attribute index of array to be bound to vertex element.
         * @param vertexElement Description of vertex element format describing the vertex attribute to bind.
         * @param stride        Delta between attribute data of two adjacent vertices.
         * @param base          Specifies a pointer to the first component of
         *                      the first generic vertex attribute in the array. The initial value is 0.
         * @return              True if attribute was bound, false if attribute doesn't exist or another error occurred activating it.
         */
        static bool BindClientAttribute(Int attribute, const VertexGeometry::VertexElementFormat& vertexElement, UInt stride, const void* base);

        /**
         * Bind attribute
         * @param attribute     Attribute index of array to be bound to vertex element.
         * @param vertexElement Description of vertex element format describing the vertex attribute to bind.
         * @param stride        Delta between attribute data of two adjacent vertices.
         * @param divisor
         * @return              True if attribute was bound, false if attribute doesn't exist or another error occurred activating it.
         */
        static bool BindInstanceAttribute(Int attribute, const VertexGeometry::VertexElementFormat& vertexElement, UInt stride, UInt divisor);

        /**
         * Bind attribute array pointer to vertex element
         * @param attribute     Attribute index of array to be bound to vertex element.
         * @param vertexElement Description of vertex element format describing the vertex attribute to bind.
         * @param stride        Delta between attribute data of two adjacent vertices.
         * @param base          Specifies a pointer to the first component of
         *                      the first generic vertex attribute in the array. The initial value is 0.
         * @param divisor
         * @return              True if attribute was bound, false if attribute doesn't exist or another error occurred activating it.
         */
        static bool BindInstanceClientAttribute(Int attribute, const VertexGeometry::VertexElementFormat& vertexElement, UInt stride, const void* base, UInt divisor);

        /**
         * Unbind array buffer
         * @return True if unbinding array buffer succeeded.
         */
        static bool UnbindArrayBuffers();

        /**
         * Bind single attribute value (also known as dangling uniform) to be used for all attributes with the attribute name given.
         * A constant vertex attribute is the same for all vertices of a primitive.
         * @param programHandle Handle to shader program where constant vertex attribute shall be set.
         * @param name          Name of attribute in shader program.
         * @param data          Data to be set as value of attribute.
         * @param type          Constant Attribute data type.
         * @return              True if attribute exists and value is set, false otherwise.
         */
        static bool SetConstantVertexAttribute(Handle programHandle, const Char* name, const void* data, Shader::ConstantAttributeType type);

        /**
         * Bind single attribute value (also known as dangling uniform) to be used for all attributes with the attribute name given.
         * A constant vertex attribute is the same for all vertices of a primitive.
         * @param index         Attribute index to set data on.
         * @param data          Data to be set as value of attribute.
         * @param type          Constant Attribute data type.
         * @return              True if setting the attributes constant value succeeds.
         */
        static bool SetConstantVertexAttribute(Int index, const void* data, Shader::ConstantAttributeType type);

        /**
         * Bind single attribute value (also known as dangling uniform) to be used for all attributes with the attribute name given.
         * A constant vertex attribute is the same for all vertices of a primitive.
         * @param index         Attribute index to set data on.
         * @param data          Data to be set as value of attribute.
         * @return              True if setting the attributes constant value succeeds.
         */
        template <Shader::ConstantAttributeType Type>
        static bool SetConstantVertexAttribute(Int index, const void* data);


        /**
         * Activates generic vertex buffer attribute with given index in render device.
         * @param index Index of vertex attribute to be activated.
         * @return      True if attribute array was activated.
         */
        static bool ActivateAttribute(UInt index);

        /**
         * Deactivates generic vertex buffer attribute with given index in render device.
         * Note: If generic vertex buffer attribute is disabled, the constant vertex attribute with same name is used instead.
         * @param index Index of vertex attribute to be activated.
         * @return      True if attribute array was deactivated.
         */
        static bool DeactivateAttribute(UInt index);

        /** RenderBuffer functions ********************************************************************************************/

        /**
         * Upload RenderBuffer to graphics device memory (VRAM).
         * The handle to the RenderBuffer in VRAM is set to the specified RenderBuffer object
         * @param renderBuffer   Specifies the RenderBuffer object to upload.
         * @return               True if uploading render buffer succeeded.
         */
        CANDERA_DEPRECATED_3_3_0("Use Renderer::UploadRenderBuffer instead.",
            static bool UploadRenderBuffer(RenderBuffer& renderBuffer)
        );

        /**
         * Unload RenderBuffer from graphics device memory (VRAM).
         * @param renderBuffer   Specifies the RenderBuffer object to unload.
         * @return               True if renderBuffer was deleted in VRAM, false if renderBuffer doesn't have any video memory handle.
         */
        CANDERA_DEPRECATED_3_3_0("Use Renderer::UnloadRenderBuffer instead.",
            static bool UnloadRenderBuffer(RenderBuffer& renderBuffer)
        );

        /**
         * Set the RenderBuffer given as active RenderBuffer in render device.
         * @param renderBuffer   RenderBuffer to be bind as current render buffer.
         * @return               True if render buffer is activated, False otherwise, e.g. if render buffer object
         *                       has an invalid VRAM handle.
         */
        static bool ActivateRenderBuffer(const RenderBuffer& renderBuffer);


        /* FrameBufferRenderTarget functions **********************************************************************************/

        ////**
        //* Upload FrameBufferRenderTarget to graphics device memory (VRAM).
        //* The handle to the FrameBufferRenderTarget in VRAM is set to the specified FrameBufferRenderTarget object
        //* @param frameBufferRenderTarget specifies the FrameBufferRenderTarget object to upload.
        //*/
        //static bool UploadFrameBufferRenderTarget(FrameBufferRenderTarget& frameBufferRenderTarget);

        ////**
        //* Unload FrameBufferRenderTarget from graphics device memory (VRAM).
        //*/
        //static bool UnloadFrameBufferRenderTarget(FrameBufferRenderTarget& frameBufferRenderTarget);

        ////**
        //* Set the FrameBufferRenderTarget given as active FrameBufferRenderTarget in render device.
        //*/
        //static bool ActivateFrameBufferRenderTarget(const FrameBufferRenderTarget& frameBufferRenderTarget);

         /**
          * Reads pixels from the framebuffer and copies them into a given data buffer.
          * The data buffer size must take into consideration the probable padding bytes resulted from the pack alignment.
          * @param x             X-coordinate of the Lower left corner of the rectangle to read from the frame buffer.
          * @param y             Y-coordinate of the Lower left corner of the rectangle to read from the frame buffer.
          * @param width         Width of the rectangle to read from the frame buffer.
          * @param height        Height of the rectangle to read from the frame buffer.
          * @param pixelFormat   Format and type of pixels to read, e.g. RgbaUnsignedBytePixelFormat.
          * @param packAlignment Defines the byte alignment of the pixels to read.
          * @param dstData       Pointer to an array of pixels where values read from the framebuffer are stored. Has to be preallocated.
          * @return              If reading of framebuffer succeeded(true).
          */
        static bool ReadPixels(Int32 x, Int32 y, Int32 width, Int32 height, Bitmap::PixelFormat pixelFormat, Bitmap::PackAlignment packAlignment, UInt8* dstData);

        /** Viewport and Camera functions ****************************************************************************************/

        /**
         * Set the rectangle as active viewport in render device.
         * @param viewport              Normalized viewport rectangle.
         * @param renderTargetWidth     Width of the render target.
         * @param renderTargetHeight    Height of the render target.
         */
        static void ActivateViewport(const Rectangle& viewport, Float renderTargetWidth, Float renderTargetHeight);

        /**
         * Set the given camera viewport as active viewport in render device.
         * @param camera    Camera object containing viewport data to be activated.
         */
        static void ActivateViewport(const Camera& camera);

        /**
         * Set camera given as active camera used for subsequent render calls.
         * @param camera    Camera to be set active for subsequent render calls.
         */
        static void SetActiveCamera(const Camera* camera);

        /**
         * Retrieve camera currently active.
         * @return  The currently active camera.
         */
        static const Camera* GetActiveCamera();

        /** Light functions *****************************************************************************************************/

        /**
         *  Retrieves the number of active lights.
         *  @return     The number of active lights;
         */
        static SizeType GetNumberOfActiveLights();
        /**
         *  Retrieves the first active light in the light list.
         *  @return     The first active light in the light list or 0 if non exist.
         */
        static const Light* GetFirstActiveLight();

        /**
         *  Retrieves the next active light in the light list of this scene.
         *  @return The next active light in the light list or 0 if non exist.
         */
        static const Light* GetNextActiveLight();

        /** Render state functions **********************************************************************************************/

        /**
         * Set render state given as active render state of current render context.
         * @param state     RendeState to set value to.
         * @param value     Value for state.
         * @return          True if value was set to RenderState. False otherwise, e.g. an invalid RenderState was specified.
         */
        static bool ActivateRenderState(RenderState state, UInt32 value);

        /**
         * Get active render state value of current render context.
         * @param state     State to get currently set value.
         * @return          Render state value that can be casted to native Candera type if successful, 0xFFffFFff otherwise.
         */
        static UInt32 GetRenderState(RenderState state);

        /**
         * If "enable" is true for a color component, the corresponding component of the frame buffer will be written.
         * @param enableRed     Enables(true) / Disables(false) modifying of red component in render buffer.
         * @param enableGreen   Enables(true) / Disables(false) modifying of green component in render buffer.
         * @param enableBlue    Enables(true) / Disables(false) modifying of blue component in render buffer.
         * @param enableAlpha   Enables(true) / Disables(false) modifying of alpha component in render buffer.
         * @return              True if enabling / disabling color components succeeded.
         */
        static bool ActivateColorWriteEnabled(bool enableRed, bool enableGreen, bool enableBlue, bool enableAlpha);


        /**
         * Sets color to clear color buffer to.
         * @param color     Color to clear color buffer to.
         * @return          True if setting clear color succeeded.
         */
        static bool ActivateClearColor(const Color& color);

        /**
        * Sets depth value to clear depth buffer to.
        * @param depth  Depth value to clear depth buffer to.
        * @return       True if setting depth clear value succeeded.
        */
        static bool ActivateDepthClearValue(Float depth);

        /**
        * Sets stencil value to clear stencil buffer to.
        * @param value  Stencil value to clear stencil buffer to.
        * @return       True if setting stencil clear value succeeded.
        */
        static bool ActivateStencilClearValue(Int32 value);

        /**
         * Set constant color used for subsequent blending operations against this color.
         * @param color     The blend color to set.
         * @return          True if setting blend color succeeded.
         */
        static bool ActivateBlendColor(const Color& color);

        /**
         * Set blend factors and operation as active for subsequent render operations in render device for RGBA values.
         * @param  sourceFactor Source fragment coefficient for blending equation.
         * @param  destFactor   Destination fragment coefficient for blending equation.
         * @param  operation    Operator of blending equation.
         * @return              True if setting blend mode succeeded.
         */
        static bool ActivateBlendMode(RenderMode::BlendFactor sourceFactor, RenderMode::BlendFactor destFactor, RenderMode::BlendOperation operation);

        /**
         * Set blend factors and operation as active for subsequent render operations in render device for RGB and alpha separately.
         * @param sourceRGBFactor   Source RGB coefficient for blending equation.
         * @param destRGBFactor     Destination RGB coefficient for blending equation.
         * @param operationRGB      Operator of blending equation for RGB components.
         * @param sourceAlphaFactor Source alpha coefficient for blending equation.
         * @param destAlphaFactor   Destination alpha coefficient for blending equation.
         * @param operationAlpha    Operator of blending equation for alpha component.
         * @result                  True if setting blend mode succeeded.
         */
        static bool ActivateBlendModeSeparate(RenderMode::BlendFactor sourceRGBFactor, RenderMode::BlendFactor destRGBFactor, RenderMode::BlendOperation operationRGB,
                                              RenderMode::BlendFactor sourceAlphaFactor, RenderMode::BlendFactor destAlphaFactor, RenderMode::BlendOperation operationAlpha);

        /**
         * Set custom sample coverage mask for multisampling anti-aliasing.
         * @param coverage  Tells the percentage of the samples that a fragment mask selects to pass.
         * @param invert    Is false if samples selected by fragment mask shall pass and true if samples that would.
         *                  have been discarded shall pass.
         * @return          True if setting sample coverage mask for multisampling anti-aliasing succeeded.
         */
        static bool ActivateSampleCoverage(Float coverage, bool invert);


        /**
         * Set stencil function and reference value for stencil testing.
         * Equation for Stencil Test: (refValue & mask) compFunc (stencil & mask).
         * @param compFunc  specifies the comparison function for the stencil test.
         * @param refValue  specifies the reference value for the stencil test.
         * @param mask      specifies the mask that is bitwise ANDed with both the reference value and the stencil value stored.
         * @param face      specifies the stencil plane associated with the provided stencil function.
         * @return          True if setting stencil function succeeded.
         */
        static bool ActivateStencilFunction(RenderMode::ComparisonFunction compFunc, Int32 refValue, UInt32 mask,
                                                  RenderMode::StencilPlane face = RenderMode::FrontAndBackFace);

         /**
          * Set stencil operations applied to the stencil buffer for three conditions:
          * Condition 1: The fragment fails the stencil test. (Subsequent depth test and other tests will not be applied.)
          * Condition 2: The fragment fails the depth test after the stencil test passed.
          * Condition 3: The fragment passes both the stencil and depth test.
          * @param stencilFail  Defines the operation applied if Condition 1 is true.
          * @param depthFail    Defines the operation applied if Condition 2 is true.
          * @param depthPass    Defines the operation applied if Condition 3 is true.
          * @param face         Specifies whether front and / or back stencil plane is updated.
          * @return             True if setting stencil operations succeeded.
          */
        static bool ActivateStencilOperation(RenderMode::StencilOperation stencilFail, RenderMode::StencilOperation depthFail,
                                                   RenderMode::StencilOperation depthPass, RenderMode::StencilPlane face = RenderMode::FrontAndBackFace);


        /**
         * Set the write mask applied to values written into the stencil buffer.
         * @param mask  Specifies a bit mask to enable and disable writing of individual bits in the stencil planes. Initially, all bits are set to 1's.
         * @param face  Specifies whether front and / or back stencil write mask is updated.
         * @return      True if setting stencil write mask succeeded.
         */
        static bool ActivateStencilWriteMask(UInt32 mask, RenderMode::StencilPlane face = RenderMode::FrontAndBackFace);

        /**
         * Activates Coverage Sampling with coverage operation given (See OpenGLES extension GL_NV_coverage_sample).
         * @param operation     Defines the coverage operation used for coverage sampling.
         * @return              true if coverage sampling with operation given has been activated, false otherwise.
         */
        static bool ActivateCoverageOperation(RenderMode::CoverageOperation operation);

        /**
         * Set normalized scissor rectangle in render device.
         * @param rectangle     Is the normalized scissor rectangle, where x:0.0, y:0.0 is left, upper and x:1.0, y:1.0 is right, bottom corner of render target.
         *                      Thus, a rectangle with left:0.5, top:0.0, width: 0.5, height 0.5 addresses the right upper quarter of the render target.
         * @param renderTarget  Defines the render target the normalized scissor rectangle is related to.
         * @return              True if setting scissor rectangle succeeded.
         */
        static bool ActivateScissorRectangle(const Rectangle& rectangle, const RenderTarget3D* renderTarget);

        /**
         * Set normalized scissor rectangle in render device.
         * @param rectangle           Is the normalized scissor rectangle, where x:0.0, y:0.0 is left, upper and x:1.0, y:1.0 is right, bottom corner of render target.
         *                            Thus, a rectangle with left:0.5, top:0.0, width: 0.5, height 0.5 addresses the right upper quarter of the render target.
         * @param renderTargetWidth   The width of the render target to apply scissoring in pixels.
         * @param renderTargetHeight  The height of the render target to apply scissoring in pixels.
         * @return                    True if setting scissor rectangle succeeded.
         */
        static bool SetScissorRectangle(const Rectangle& rectangle, Float renderTargetWidth, Float renderTargetHeight);

        /**
        * Retrieve normalized scissor rectangle in render device.
        * @param renderTarget   RenderTarget used to normalize absolute scissor rectangle coordinates.
        * @return               The normalized scissor rectangle related to the render target given.
        */
        static const Rectangle GetScissorRectangle(const RenderTarget3D* renderTarget);

        /**
         * Set depth bias active. The depth bias is added to the z-value of the primitive when performing the depth test.
         * @param scaleFactor   Scale factor component of depth offset equation. See OpenGL ES documentation for glPolygonOffset.
         * @param units         Units component of depth offset equation. See OpenGL ES documentation for glPolygonOffset.
         * @return              True if setting depth bias succeeded.
         */
        static bool ActivateDepthBias(Float scaleFactor, Float units);


        /** Render functions ***************************************************************************************************/

        /**
         * Clear background with Color defined.
         * Before clearing      The clear color is set using ActivateClearColor.
         * @param color         Color to which background shall be cleared.
         * @return              True if clearing background succeeded.
         */
        static bool ClearColorBuffer(const Color& color);

        /**
         * Clear depth buffer with Float value defined.
         * Before clearing      The depth clear value is set using ActivateDepthClearValue.
         * @param depth         Value to clear depth buffer to.
         * @return              True if clearing depth buffer succeeded.
         */
        static bool ClearDepthBuffer(Float depth);

        /**
         * Clear stencil buffer with Int32 value defined.
         * Before clearing the stencil clear value is set using ActivateStencilClearValue.
         * @param value     Value to clear stencil buffer to.
         * @return          True if clearing depth buffer succeeded.
         */
        static bool ClearStencilBuffer(Int32 value);

        /**
         * Clears a framebuffer with all specified buffer types, with the values previously set in ActivateClearColor, ActivateDepthClearValue
         * and ActivateStencilClearValue.
         * @param color     Enables/Disables clearing the color buffer.
         * @param depth     Enables/Disables clearing the depth buffer.
         * @param stencil   Enables/Disables clearing the stencil buffer.
         * @return          True if frame buffer was cleared successfully.
         */
        static bool ClearFrameBuffer(bool color, bool depth, bool stencil);

        /**
         * Clear coverage buffer
         * @return  True if clearing coverage buffer succeeded.
         */
        static bool ClearCoverageBuffer();

        /**
         * Render VertexBuffer.
         * @param vertexBuffer  VertexBuffer object containing vertex geometry and index buffer to render.
         * @return              True if VertexBuffer was rendered successfully.
         */
#ifdef CANDERA_3D_ENABLED
        CANDERA_DEPRECATED_3_2_1("Function has been moved. Please use Renderer::Draw(const VertexBuffer& vertexBuffer) instead.",
            static bool DrawVertexBufffer(const VertexBuffer& vertexBuffer)) { return DrawVertexBuffer(vertexBuffer,
            vertexBuffer.GetElementStart(), vertexBuffer.GetElementCount()); }
#endif

        /**
         * Render an instanced VertexBuffer.
         * @param vertexBuffer  VertexBuffer object containing vertex geometry and index buffer to render.
         * @param instanceCount The number of times the vertex buffer shall be instanced.
         * @return              True if VertexBuffer was rendered successfully.
         */
#ifdef CANDERA_3D_ENABLED
        CANDERA_DEPRECATED_3_2_1("Function has been moved. Please use Renderer::Draw(const VertexBuffer& vertexBuffer, UInt instanceCount) instead.",
            static bool DrawInstancedVertexBuffer(const VertexBuffer& vertexBuffer, UInt32 instanceCount)) { return DrawInstancedVertexBuffer(
            vertexBuffer, instanceCount, vertexBuffer.GetElementStart(), vertexBuffer.GetElementCount()); }
#endif

        /**
         * Render triangles
         * @param vertexCount   Count of vertices to be drawn.
         * @return              True if triangles were rendered successfully.
         */
        static bool DrawTriangles(UInt32 vertexCount);

        /**
         * Render indexed triangles
         * @param indexCount    Count of indices.
         * @param indices       Pointer to location where indices are stored.
         * @return              True if indexed triangles were rendered successfully.
         */
        static bool DrawTrianglesIndexed(UInt32 indexCount, const void* indices = 0);

        /**
         * Render triangle strip
         * @param vertexCount   Count of vertices to be drawn.
         * @return              True if triangle strip was rendered successfully.
         */
        static bool DrawTriangleStrip(UInt32 vertexCount);

        /**
         * Render indexed triangle strip
         * @param indexCount    Count of indices.
         * @param indices       Pointer to location where indices are stored.
         * @return              True if indexed triangle strip was rendered successfully.
         */
        static bool DrawTriangleStripIndexed(UInt32 indexCount, const void* indices = 0);

        /**
         * Render triangle fan
         * @param vertexCount   Count of vertices to be drawn.
         * @return              True if triangle fan was rendered successfully.
         */
        static bool DrawTriangleFan(UInt32 vertexCount);

        /**
         * Render indexed triangle fan
         * @param indexCount    Count of indices.
         * @param indices       Pointer to location where indices are stored.
         * @return              True if indexed triangle fan was rendered successfully.
         */
        static bool DrawTriangleFanIndexed(UInt32 indexCount, const void* indices = 0);


        /**
         * Set line width for rendering
         * @param lineWidth     Width of a line in pixels.
         * @return              True if setting line width succeeded.
         */
        static bool SetLineWidth(Float lineWidth);

        /**
         * Render lines
         * @param vertexCount   Count of vertices to be drawn.
         * @return              True if lines were rendered successfully.
         */
        static bool DrawLines(UInt32 vertexCount);

        /**
         * Render indexed lines
         * @param indexCount    Count of indices.
         * @param indices       Pointer to location where indices are stored.
         * @return              True if indexed lines were rendered successfully.
         */
        static bool DrawLinesIndexed(UInt32 indexCount, const void* indices = 0);

        /**
         * Render line strip
         * @param vertexCount   Count of vertices to be drawn.
         * @return              True if line strip was rendered successfully.
         */
        static bool DrawLineStrip(UInt32 vertexCount);

        /**
         * Render indexed line strip
         * @param indexCount    Count of indices.
         * @param indices       Pointer to location where indices are stored.
         * @return              True if indexed line strip was rendered successfully.
         */
        static bool DrawLineStripIndexed(UInt32 indexCount, const void* indices = 0);

        /**
         * Render line loop
         * @param vertexCount   Count of vertices to be drawn.
         * @return              True if line loop was rendered successfully.
         */
        static bool DrawLineLoop(UInt32 vertexCount);

        /**
         * Render indexed line loop
         * @param indexCount    Count of indices.
         * @param indices       Pointer to location where indices are stored.
         * @return              True if indexed line loop was rendered successfully.
         */
        static bool DrawLineLoopIndexed(UInt32 indexCount, const void* indices = 0);

        /**
         * Render points
         * @param vertexCount   Count of vertices to be drawn.
         * @return              True if points were rendered successfully.
         */
        static bool DrawPoints(UInt32 vertexCount);

        /**
         * Render indexed points
         * @param indexCount    Count of indices.
         * @param indices       Pointer to location where indices are stored.
         * @return              True if indexed points were rendered successfully.
         */
        static bool DrawPointsIndexed(UInt32 indexCount, const void* indices = 0);

        /**
         * Allows the application to specify implementation-dependent performance hints.
         * @param target    Hint to be set, whether hint is for mipmapping or fragment shaders.
         * @param mode      Specifies which mode the feature should use, nicest, fastest or don't care.
         */
        static void Hint(HintTarget target, HintMode mode);


        /** Render Device Query functions ***************************************************************************************/

        /**
        * Upload query to graphics device memory. Query can be uploaded to one single context only.
        * @param query Query object gets handle to graphics device memory assigned.
        * @return      True if query handle is generated in graphics device memory. Returns false if query object is already uploaded in another context.
        */
        static bool UploadQuery(Query& query);


        /**
        * Unload query from graphics device memory.
        * @param query Query object holds the handle to graphics device memory to delete.
        * @return      True if query handle is deleted from graphics device memory. Returns false if query object is unloaded from another context.
        */
        static bool UnloadQuery(Query& query);

        /**
        * Begin a new query. BeginQuery and EndQuery together delimit the boundaries of a render device query.
        * @param query Defines the new render device query to start.
        * @return      True if query has been started successfully, false otherwise.
        */
        static bool BeginQuery(const Query& query);

        /**
        * End a currently running query. BeginQuery and EndQuery together delimit the boundaries of a render device query.
        * @param query Defines the render device query to end.
        * @return      True if query has been finished successfully, false otherwise.
        */
        static bool EndQuery(const Query& query);

        /**
        * Check if query object given in current context is complete and if result can be retrieved immediately without any delay (see function GetQueryResult).
        * Note: Frequent invocation of this function with same query object is guaranteed to return true eventually, but may result in a considerable performance loss.
        *       It is recommended to wait between one and three frames to check state, concrete number is platform dependent.
        *       A query result cannot be retrieved when query is not uploaded, has not been activated before, or is in active state (see function IsQueryActive).
        * @param query
        * @return bool True if query object result is available, false otherwise.
        */
        static bool IsQueryResultAvailable(const Query& query);

        /**
        *  Get query result of query object given in current context.
        *  If QueryType is QueryAnySamplesPassed or QueryAnySamplesPassedConservative the value is a boolean telling whether any samples have passed the depth test.
        *  If QueryType is QueryTransformFeedbackPrimitivesWritten the value is an integer value telling how many vertices are written into the bound transform feedback buffer(s).
        *  Note: Invoke function IsQueryResultAvailable to determine whether result is returned immediately or a delay will occur.
        *        The function GetQueryResult implicitly flushes the command pipeline.
        *  @return Query result holds the result according to query type. If query result is invalid it is set to value 0xFFffFFff.
        */
        static UInt32 GetQueryResult(const Query& query);

        /**
        * Retrieve whether query object is active in current context or not.
        * @param query Defines the query object that is tested being active or not.
        * @return      True if query object given is active, false otherwise.
        */
        static bool IsQueryActive(const Query& query);


        /** Render Device Capability functions **********************************************************************************/

        /**
         * Retrieves device information as an ANSII string, or null in case of no information is supplied.
         * @param deviceInfo    Parameter specifying which information should be returned.
         * @return              Device information as an ANSII string, or null in case of no information is supplied.
         */
        static const Char* GetDeviceInfo(DeviceInfo deviceInfo);

        /**
         *  Retrieves the number of texture units supported by the render device. Can be greater than
         *  the number supported by Candera.
         *  @return     Number of texture units supported by the render device. Can be greater than
         *              the number supported by Candera.
         */
        static UInt GetMaxTextureUnitsSupportedByDevice();
        /**
         * Retrieves the number of texture units supported by Candera. The number is guaranteed to
         * be less equal to the number of texture units supported by render device.
         * @return      Number of texture units supported by Candera. The number is guaranteed to
         *              be less equal to the number of texture units supported by render device.
         */
        static UInt GetMaxTextureUnitsSupportedByCandera();

        /**
         * Retrieves the maximum degree of anisotropy filter supported by hardware. If the textures maximum degree of anisotropy
         * filtering is set greater than the one supported by device, the device's max degree of anisotropy filtering is taken instead.
         * @return      The maximum degree of anisotropy filter supported by hardware. If the textures maximum degree of anisotropy
         *              filtering is set greater than the one supported by device, the device's max degree of anisotropy filtering is taken instead.
         */
        static Float GetMaxAnisotropySupportedByDevice();

        /**
         *  Retrieves the maximum texture size supported by the render device.
         *  @return     Maximum texture size supported by the render device.
         */
        static UInt GetMaxTextureSizeSupportedByDevice();

         /**
         *  Retrieves the maximum number of combined texture units supported by the render device.
         *  @return     Maximum number of combined texture units supported by the render device.
         */
        static UInt GetMaxCombinedTextureUnitsSupportedByDevice();

         /**
         *  Retrieves the maximum CubeMap texture size supported by the render device.
         *  @return     Maximum CubeMap texture size supported by the render device.
         */
        static UInt GetMaxCubeMapTextureSizeSupportedByDevice();

        /**
        * Retrieves the maximum number of four-element floating-point, integer or boolean vectors that can be held
        * in uniform variable storage for a fragment shader and are supported by the render device.
        * @return   Maximum number of four-element floating-point, integer or boolean vectors that can be held
        *           in uniform variable storage for a fragment shader and are supported by the render device.
        */
        static UInt GetMaxFragmentUniformVectorsSupportedByDevice();

         /**
         *  Retrieves the maximum RenderBuffer size supported by the render device.
         *  Size means the largest width and height that the renderbuffer might get.
         *  @return     Maximum RenderBuffer size supported by the render device.
         */
        static UInt GetMaxRenderBufferSizeSupportedByDevice();

        /**
        * Retrieves the maximum number of four-element floating-point vectors for interpolating
        * varying variables between vertex and fragment shaders.
        * @return   Maximum number of four-element floating-point vectors for interpolating
        *           varying variables between vertex and fragment shaders.
        */
        static UInt GetMaxVaryingVectorsSupportedByDevice();

        /**
         * Retrieves the maximum number of 4-component generic vertex attributes accessible to a vertex shader.
         * @return  Maximum number of 4-component generic vertex attributes accessible to a vertex shader.
         */
        static UInt GetMaxVertexAttribsSupportedByDevice();

        /**
         *  Retrieves the maximum number of texture units available to be used in the vertex shader. Can be greater than
         *  the number supported by Candera.
         *  @return     Maximum number of texture units available to be used in the vertex shader. Can be greater than
         *              the number supported by Candera.
         */
        static UInt GetMaxVertexTextureUnitsSupportedByDevice();

        /**
        * Retrieves the maximum number of four-element floating-point, integer or boolean vectors that can be held
        * in uniform variable storage for a vertex shader and are supported by the render device.
        * @return   Maximum number of four-element floating-point, integer or boolean vectors that can be held
        *           in uniform variable storage for a vertex shader and are supported by the render device.
        */
        static UInt GetMaxVertexUniformVectorsSupportedByDevice();

         /**
         *  Retrieves the maximum supported width and height of the view port. Return two values, width and height.
         *  @param  width   Out parameter that stores the maximum supported width of the viewport.
         *  @param  height  Out parameter that stores the maximum supported height of the viewport.
         */
        static void GetMaxViewportSizeSupportedByDevice(UInt32& width, UInt32& height);

        /**
         * Retrieves the maximum value of an index in indexed vertex buffers.
         * @return   Maximum value of an index in indexed vertex buffers.
         */
        static UInt32 GetMaxIndexElementSupportedByDevice();

        /**
         * @return   Whether support for DXT textures is available or not.
         */
        static bool IsDxtCompressionSupportedByDevice();

        /**
         *  Does not return until the effects of all previously called commands are complete.
         */
        static void Finish();

        /**
         * Empties all render command buffers, causing all issued render commands to be executed as quickly as they are accepted by the render device.
         * Returns at any time and does not wait until all issued commands are executed.
         */
        static void Flush();

        /**
         * Retrieves if caching of render states is enabled or not.
         * @return True when render state caching is enabled.
         */
        static bool IsRenderStateCacheEnabled();

        /**
         * Determine whether Instancing is supported by the RenderDevice.
         * @return True if Instancing is supported, false otherwise.
         */
        static bool IsInstancingSupported();

        // Synchronization

        /**
         * Reset the RenderStateCache object of the current context. Method should be called whenever
         *  the render states are changed bypassing RenderDevice and the cache needs to be synchronized
         *  with the new render states.
         * @param resetOption   Optional parameter to specify which render state to be reset. By default all
         *                      values are reset.
         */
        static void ResyncCurrentRenderStateCache(RenderStateCacheResyncOption resetOption = ResyncAll);

        /**
         * Create SynchronizationFence object.
         *
         * @param [out] fenceHandle Handle of the new synchronization fence.
         * @return true if fence successfully created, false otherwise.
         */
        static bool CreateSynchronizationFence(SynchronizationFence::Handle& fenceHandle);

        static const UInt64 c_IgnoreWaitTimeOut;      ///>Value that can be passed to ClientWaitSynchronizationFence as timeout parameter to disable timeout.

        /**
         * Wait until the fence command is completed or the timeout expires.
         *
         * @param fenceHandle Handle of the fence that is expected to be completed when method returns.
         * @param timeout Timeout in nanoseconds, after which the method will return even if the fence command is not yet completed.
         * @param flush Flag specifying if any existing buffered commands should be sent to the RenderDevice command queue to be processed
         *  (including the fence command). Flag should be set to true if the timeout is ignored.
         * @return Wait result.
         *
         * @see SynchronizationFence::WaitResult
         */
        static SynchronizationFence::WaitResult ClientWaitSynchronizationFence(SynchronizationFence::Handle fenceHandle, UInt64 timeout = c_IgnoreWaitTimeOut, bool flush = true);

        /**
         * Check whether the fence command is completed or not.
         *
         * @param fenceHandle Handle of the fence that should be checked for completion.
         * @return true if command is completed, false otherwise.
         */
        static bool IsSynchronizationFenceSignaled(SynchronizationFence::Handle fenceHandle);

        /**
         * Destroy fence.
         *
         * The fence is destroyed only after no thread is waiting for command completion.
         *
         * @param fenceHandle Handle of the fence that should be destroyed.
         */
        static void DestroySynchronizationFence(SynchronizationFence::Handle fenceHandle);


        // Direct texture

        /**
         * Upload a Texture given to VRAM.
         * Handle of texture is generated during upload and will be set
         * to the given texture.
         * Postcondition on success: Texture is active.
         * @param  textureImage The texture image to upload to VRAM.
         * @param  unit         Texture unit to which the texture is bound.
         * @return              True if uploading and activating texture
         *                      succeeded. False otherwise.
         */
        CANDERA_DEPRECATED_3_3_0("Use Renderer::UploadDirectTextureImage instead.",
            static bool UploadDirectTextureImage(DirectTextureImage& textureImage, UInt unit)
        );

        /**
         * Unload a Texture from graphics device memory (VRAM).
         * @param  textureImage     Texture image to unload from VRAM.
         * @return                  True if texture image was unloaded.
         */
        CANDERA_DEPRECATED_3_3_0("Use Renderer::UnloadDirectTextureImage instead.",
            static bool UnloadDirectTextureImage(DirectTextureImage& textureImage)
        );

        // External texture


        CANDERA_DEPRECATED_3_3_0("Use Renderer::UploadExternalTextureImage instead.",
            static bool UploadExternalTextureImage(ExternalTextureImage& textureImage, UInt unit)
            );

        CANDERA_DEPRECATED_3_3_0("Use Renderer::UnloadExternalTextureImage instead.",
            static bool UnloadExternalTextureImage(ExternalTextureImage& textureImage)
            );

        /**
         * Lock the content of the texture image, so that it's available to the
         * CPU. Address of the texture is generated and will be set
         * to the given texture.
         * @param  textureImage Texture image to Lock.
         * @return              True if texture image was successfully locked.
         */
        static bool LockDirectTextureImage(DirectTextureImage& textureImage);

        /**
         * Unlock the texture image. After unlocking the texture may be used
         * for drawing. In effect this function lets the GPU know that texture data has changed.
         * Postcondition on success: Texture is active.
         * @param  textureImage Texture image to unlock.
         * @param  unit
         * @return              True if texture image was successfully unlocked.
         */
        static bool UnlockDirectTextureImage(DirectTextureImage& textureImage, UInt unit);

        static const Float c_defaultMinLod;     ///< Default MinLod texture sampling parameter.
        static const Float c_defaultMaxLod;     ///< Default MaxLod texture sampling parameter.

    private:
        friend class Camera;
        friend class ContextProvider3D;
        friend class Scene;
        friend class Renderer;

        typedef Internal::LinkedList<Light, &Light::LightListNode> LightList;

        static const RenderDevice::LightList* m_activeLights;   // Current active light list set by Scene.
        static const Light* m_activeLightIterator;              // Used to iterate of the light list.

        // Current active render state cache set by render context.
        static RenderStateCache *m_currentRenderStateCache;

        /**
         * Constructor is private as RenderDevice is a singleton.
         */
        RenderDevice();

        /**
         * Render VertexBuffer.
         * @param vertexBuffer  VertexBuffer object containing vertex geometry and index buffer to render.
         * @param elementStart  Index of the first element to be drawn.
         * @param elementCount  The number of elements (starting with elementStart) to be drawn.
         * @return              True if VertexBuffer was rendered successfully.
         */
        static bool DrawVertexBuffer(const VertexBuffer& vertexBuffer, SizeType elementStart, SizeType elementCount);

        /**
         * Render an instanced VertexBuffer.
         * @param vertexBuffer   VertexBuffer object containing vertex geometry and index buffer to render.
         * @param instanceCount  The number of times the vertex buffer shall be instanced.
         * @param elementStart   Index of the first element to be drawn.
         * @param elementCount   The number of elements (starting with elementStart) to be drawn.
         * @return               True if VertexBuffer was rendered successfully.
         */
        static bool DrawInstancedVertexBuffer(const VertexBuffer& vertexBuffer, UInt32 instanceCount, SizeType elementStart, SizeType elementCount);

        /**
         * Set lights given active for subsequent render calls.
         * @param lightList     A list of lights to set active.
         */
        static void SetActiveLights(const RenderDevice::LightList* lightList);

        /**
         * Helper function for texture binding.
         * @param textureHandle Handle to texture which shall be bound.
         * @param unit          Texture unit to which texture shall be associated.
         * @param target        Target to which texture shall be associated.
         * @return              True if updating texture handle succeeded.
         */
        static bool UpdateTextureBinding(Handle textureHandle, UInt unit, TextureImage::TextureTargetType target);

        /**
         * Upload a Texture given to VRAM.
         * Handle to texture memory is generated during upload and will be set to the given texture.
         * Postcondition on success: Texture is active.
         * @param  textureImage The texture image to upload to VRAM.
         * @param  unit         Texture to be activated and associated to textureImage.
         * @return              True if uploading and activating texture succeeded. False otherwise, also fails if
         *                      bitmap of texture image is not valid.
         */
        static bool UploadBitmapTextureImageInternal(BitmapTextureImage& textureImage, UInt unit);

        /**
         * Unload a Texture from graphics device memory (VRAM).
         * @param  textureImage     Texture image to unload from VRAM.
         * @return                  True if texture image was unloaded.
         */
        static bool UnloadBitmapTextureImageInternal(BitmapTextureImage& textureImage);

        /**
         * Destroy a Texture in graphics device memory (VRAM).
         * @param  handle  The handle of the texture to be destroyed.
         * @return         True if destruction was successful.
         */
        static bool DestroyTexture(Handle handle);

        /**
         * Upload a BitmapTextureImage to VRAM defined by the given texture handle.
         * Postcondition on success: Texture is active.
         * @param  handle       The handle of the texture in VRAM to upload the BitmapTextureImage to.
         * @param  textureImage The texture image to upload to VRAM.
         * @param  unit         Texture to be activated and associated to textureImage.
         * @return              True if uploading and activating texture succeeded. False otherwise, also fails if
         *                      bitmap of texture image is not valid.
         */
        static bool ReuseTexture(Handle handle, BitmapTextureImage& textureImage, UInt unit);

        /**
         * Uploads a single cube map face. Image origin has to be in the upper left corner (in contrary to
         * 2D textures).
         * @param face      CubeMap face to upload image to.
         * @param bitmap    Bitmap containing image data to upload to CubeMap.
         * @param mipmap    Whether Bitmap contains mipmap levels which should be uploaded.
         * @param levels    Number of mipmap levels.
         * @return          True if uploading of CubeMap face succeeded.
         */
        static bool UploadCubeMapFace(CubeMapTextureImage::TargetFace face, const Bitmap* bitmap, bool mipmap, UInt levels);


        /**
         * These helper functions are required because the structure RenderStateCache
         * is defined in the cpp file and not visible for other classes.
         * @return  Pointer to the created RenderStateCache struct.
         */
        static RenderStateCache* CreateRenderStateCache();
        static void DestroyRenderStateCache(RenderStateCache* renderStateCache);

        /**
         * Called by ContextProvider shortly before context is destroyed. Can be used as a cleanup function to unload any
         * objects and free resources.
         * @param contextResourcePool The ContextResourcePool to be destroyed.
         */
        static void OnContextUnload(SizeType contextIndex);

        /**
         * This function should be only used internally to access the RenderStateCache object of the current context.
         * @return  RenderStateCache of the current context.
         */
        static RenderStateCache* GetCurrentRenderStateCache();

        /**
         * Set the RenderStateCache object of the current context. It's called when another
         * context is made current.
         * @param currentRenderStateCache       RenderStateCache to be set.
         */
        static void SetCurrentRenderStateCache(RenderStateCache* currentRenderStateCache);


        /**
         * These helper functions are required because the structure TextureStateCache
         * is defined in the cpp file and not visible for other classes.
         * @return  Pointer to the created TextureStateCache struct.
         */
        static TextureStateCache* CreateTextureStateCache();
        static void DestroyTextureStateCache(TextureStateCache* textureStateCache);

        /**
         * Helper function for setting the TextureStateCache on the passed TextureImage. The previously set TextureStateCache
         *  is destroyed.
         */
        static void SetTextureStateCache(TextureImage& textureImage, TextureStateCache* textureStateCache);

#ifdef CGIDEVICE_OPENGLES_30
        /**
         * Unload a device buffer from VRAM.
         * @param deviceHandle  The device handle of the buffer to unload.
         * @return              True if the buffer was successfully unloaded from VRAM.
         */
        static bool UnloadBuffer(Handle deviceHandle);


        // --------------
        // UNIFORM BUFFER
        // --------------

        /**
         * Upload a uniform buffer to VRAM.
         * @param data          A pointer to the data to upload.
         * @param bufferSize    The size of the data to upload in bytes.
         * @param deviceHandle  A reference to the handle where the uniform buffer of the device
         *                      will be returned in.
         * @return              True if the data was successfully uploaded to VRAM.
         */
        static bool UploadUniformBuffer(const void* data, SizeType bufferSize, /*out*/ Handle& deviceHandle);

        /**
         * Update the data of a uniform buffer in VRAM.
         * @param deviceHandle  The device handle of the uniform buffer to update.
         * @param data          A pointer to the data to update the uniform buffer with.
         * @param bufferSize    The size of the data to update the uniform buffer with in bytes.
         * @return              True if the data was successfully updated in VRAM.
         */
        static bool UpdateUniformBuffer(Handle deviceHandle, const void* data, SizeType bufferSize);

        /**
         * Bind a uniform buffer to the uniform block of a shader.
         * @param bufferHandle       The device handle of the uniform buffer to bind to the shader program.
         * @param programHandle      The device handle of the shader program to bind the uniform buffer to.
         * @param uniformBlockIndex  The uniform block index of the shader to bind the uniform buffer to.
         * @param bindingPoint       The binding point to bind the uniform block from the shader with the
         *                           uniform buffer.
         * @return                   True if the uniform buffer was successfully bound to the uniform block
         *                           of the shader.
         */
        static bool BindUniformBufferToBlock(Handle bufferHandle, Handle programHandle, UInt uniformBlockIndex, UInt bindingPoint);

        /**
         * Return the active uniform block count of the shader program.
         * @param programHandle  Handle to shader program to which receive active uniform block count from.
         * @param count          Retrieved active uniform block count, out parameter.
         * @return               True if active uniform block count was successfully retrieved.
         */
        static bool GetActiveUniformBlockCount(Handle programHandle, /*out*/ Int& count);

        /**
         * Return the length of the longest active uniform block name of the shader program including
         * the null termination character. If no active uniform block exists, 0 is returned.
         * @param programHandle  Handle of the shader program.
         * @param maxLength      A reference to the variable to return the length of longest active
         *                       uniform block name in.
         * @return               True if the operation was successful.
         */
        static bool GetActiveUniformBlockMaxNameLength(Handle programHandle, Int& maxLength);

        /**
         * Return the active uniform block name of the shader program corresponding to the given index.
         * @param programHandle      The handle of the shader program to get the active uniform block name from.
         * @param uniformBlockIndex  The uniform block index to get the active uniform block name for.
         * @param nameBufferSize     The maximum number of characters (including null terminator) the RenderDevice
         *                           is allowed to write to uniformBlockName.
         * @param nameLength         A reference to the variable to store the actual number of characters written
         *                           (excluding null terminator) by the RenderDevice.
         * @param uniformBlockName   A pointer to a buffer of size nameBufferSize to store the name of the active
         *                           uniform block specified by the given uniform block index in.
         * @return  True, if the operation was successful.
         */
        static bool GetActiveUniformBlockName(Handle programHandle, UInt uniformBlockIndex, Int nameBufferSize,
            /*out*/ Int& nameLength, Char* uniformBlockName);

        /**
         * Return the active uniform block size of the shader program corresponding to the given index.
         * @param programHandle      The handle of the shader program to get the active uniform block size from.
         * @param uniformBlockIndex  The uniform block index to get the active uniform block size for.
         * @param blockSize          A reference to the variable to store the size of the active uniform block.
         * @return  True, if the operation was successful.
         */
        static bool GetActiveUniformBlockSize(Handle programHandle, UInt uniformBlockIndex, /*out*/ Int& blockSize);


        // ------------
        // PIXEL BUFFER
        // ------------

        /**
         * Create a pixel buffer. If data was provided, upload the data.
         * @param data          A pointer to the data to upload, or 0.
         * @param bufferSize    The size of the pixel buffer to create. If data is provided, this is also the size
         *                      of the data.
         * @param isPixelPack   A flag to indicate if the pixel buffer is used to transfer data from (pack), or
         *                      to (unpack) the GPU.
         * @param deviceHandle  A reference to the handle where the pixel buffer of the device
         *                      will be returned in.
         * @return              True if the pixel buffer was successfully created, and provided data uploaded.
         */
        static bool UploadPixelBuffer(const void* data, SizeType bufferSize, bool isPixelPack, /*out*/ Handle& deviceHandle);

        /**
         * Bind a pixel buffer.
         * @param deviceHandle  The device handle of the pixel buffer to bind.
         * @param isPixelPack   A flag to indicate if the pixel buffer is used to transfer data from (pack), or
         *                      to (unpack) the GPU.
         * @return              True if the pixel buffer was successfully bound.
         */
        static bool BindPixelBuffer(Handle deviceHandle, bool isPixelPack);

        /**
         * Unbind a pixel buffer.
         * @param isPixelPack  A flag to indicate if the pixel buffer is used to transfer data from (pack), or
         *                     to (unpack) the GPU.
         * @return             True if the pixel buffer was successfully unbound.
         */
        static bool UnbindPixelBuffer(bool isPixelPack);

        /**
         * Map the currently bound pixel buffer.
         * @param offset       The offset into the pixel buffer to map.
         * @param length       The length starting at offset of the pixel buffer to map.
         * @param isPixelPack  A flag to indicate if the pixel buffer is used to transfer data from (pack), or
         *                     to (unpack) the GPU.
         * @return  The address of the mapped pixel buffer at offset if the operation was successful, 0 otherwise.
         */
        static void* MapPixelBuffer(SizeType offset, SizeType length, bool isPixelPack);

        /**
         * Unmap the currently bound pixel buffer.
         * @param isPixelPack  A flag to indicate if the pixel buffer is used to transfer data from (pack), or
         *                     to (unpack) the GPU.
         * @return             True if the pixel buffer was successfully unmapped.
         */
        static bool UnmapPixelBuffer(bool isPixelPack);


        // -------------------------
        // TRANSFORM FEEDBACK BUFFER
        // -------------------------

        /**
         * Bind a transform feedback buffer at the given binding point.
         * @param deviceHandle  The device handle of the transform feedback buffer to bind.
         * @param bindingPoint  The binding point to bind the transform feedback buffer to.
         * @return              True if the transform feedback buffer was successfully bound to
         *                      the binding point.
         */
        static bool BindTransformFeedbackBuffer(Handle deviceHandle, UInt bindingPoint);

        /**
         * Unbind the transform feedback buffer at the given binding point.
         * @param bindingPoint  The binding point to unbind the transform feedback buffer from.
         * @return              True if the transform feedback buffer was successfully unbound
         *                      from the binding point.
         */
        static bool UnbindTransformFeedbackBuffer(UInt bindingPoint);

        enum TransformFeedbackPrimitiveType
        {
            Points,
            Lines,
            Triangles,
            TransformFeedbackPrimitiveTypeCount
        };

        /**
         * Begin transform feedback operation.
         * @param primitiveType  The output type of the primitives that will be recorded into the
         *                       buffer that is bound for transform feedback. This type must be
         *                       the same as the one used for rendering.
         * @return               True if transform feedback operation was started successfully.
         */
        static bool BeginTransformFeedback(TransformFeedbackPrimitiveType primitiveType);

        /**
         * End transform feedback operation.
         * @return  True if transform feedback operation was ended successfully.
         */
        static bool EndTransformFeedback();

        /**
         * Map the currently bound transform feedback buffer.
         * @param offset       The offset into the transform feedback buffer to map.
         * @param length       The length starting at offset of the transform feedback buffer to map.
         * @return             The address of the mapped transform feedback buffer at offset if the
         *                     operation was successful, 0 otherwise.
         */
        static void* MapTransformFeedbackBuffer(SizeType offset, SizeType length);

        /**
         * Unmap the currently bound transform feedback buffer.
         * @return             True if the transform feedback buffer was successfully unmapped.
         */
        static bool UnmapTransformFeedbackBuffer();
#endif
};

template<> bool RenderDevice::SetUniform<Shader::Float>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::FloatVec2>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::FloatVec3>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::FloatVec4>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::Integer>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::IntegerVec2>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::IntegerVec3>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::IntegerVec4>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::Bool>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::BoolVec2>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::BoolVec3>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::BoolVec4>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::FloatMat2>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::FloatMat3>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::FloatMat4>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::Sampler2D>(Int location, const void* data, UInt count);
template<> bool RenderDevice::SetUniform<Shader::SamplerCube>(Int location, const void* data, UInt count);

template<> bool RenderDevice::SetConstantVertexAttribute<Shader::FloatAttribute>(Int index, const void* data);
template<> bool RenderDevice::SetConstantVertexAttribute<Shader::FloatVec2Attribute>(Int index, const void* data);
template<> bool RenderDevice::SetConstantVertexAttribute<Shader::FloatVec3Attribute>(Int index, const void* data);
template<> bool RenderDevice::SetConstantVertexAttribute<Shader::FloatVec4Attribute>(Int index, const void* data);

 /** @} */ // end of CommonDevice

} // namespace Candera

#endif  // CANDERAPLATFORM_RENDERDEVICE_H

