//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERAPLATFORM_GRAPHIC_DEVICE_UNIT_H)
    #define CANDERAPLATFORM_GRAPHIC_DEVICE_UNIT_H

#include <Candera/EngineBase/Common/BaseStringBufferAppenders.h>
#include <CanderaPlatform/OS/CanderaTypes.h>

#include <CanderaPlatform/Device/Common/Base/DevicePackageDescriptor.h>
#include <CanderaPlatform/Device/Common/Base/DeviceMetaInfoHost.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnitMetaInfo.h>

#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget3D.h>

#include <CanderaPlatform/Device/Common/Internal/GraphicDeviceUnitOwnerAccess.h>

#include <CanderaAssetLoader/AssetLoaderBase/AssetLoaderBaseDataTypes.h>

namespace Candera {

/** @addtogroup CommonDevice
 *  @{
 */

class ImageSource2D;
class ImageSource3D;
class Window;

/**
 *  @brief GraphicDeviceUnit is an arbitrary object specific to the graphics 
 *  platform and can provide one or multiple instances of render targets and 
 *  ImageSources to the application.
 */
class GraphicDeviceUnit : public DeviceMetaInfoHost {
    public:
        GraphicDeviceUnit() : m_display(0), m_name(0) {}
        virtual ~GraphicDeviceUnit() { m_display = 0; m_name = 0; }

        /**
         * A name may be attached to a Graphic Device Unit to facilitate
         * identification. The lifetime of the string must be handled by the
         * caller.
         * @param val String holding the name of the GraphicDeviceUnit.
         */
        void SetName(const Char* val) { m_name = val; }
        const Char* GetName() const { return m_name; }

        /**
         *  Some objects need to be attached to a display. In order to Upload
         *  these objects a display needs to be specified by calling this
         *  method before calling Upload.
         *  Remark: In order for the association to work, the
         *  ContextResourcePool of the display must be compatible with the
         *  ContextResourcePool of the RenderTarget3D of this GDU.
         *  @param display  Numeric identifier of the attached display.
         */
        void SetDisplay(Int display) { m_display = display; }
        Int GetDisplay() const { return m_display; }

        /**
         *  Allocates resources and creates the object. Any cast method may not 
         *  be called unless a successful creation of the object.
         *  Some attributes are required to be set before create is called. Some
         *  attributes cannot be changed at a later time.
         *  @return     True if all requirements to upload GraphicDeviceUnit are
         *              met and uploading of it succeeded.
         */
        virtual bool Upload() = 0;

        /**
         *  Deallocates resources of the object.h
         */
        virtual void Unload() = 0;

        /**
         *  Casts the object to a ImageSource2D object. May return 0 if this
         *  interface is not supported.
         *  @param index    Index of requested object, when more then one 
         *                  interface is provided.
         *  @return         Pointer to the requested object.
         */
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Virtual function does not override a base-class function.")
        virtual ImageSource2D* ToImageSource2D(Int index = 0) { 
            FEATSTD_UNUSED(index);
            return 0;
        }
        const ImageSource2D* ToImageSource2D(Int index = 0) const;

        /**
         *  Casts the object to a RenderTarget2D object. May return 0 if this
         *  interface is not supported.
         *  @param index    Index of requested object, when more then one 
         *                  interface is provided.
         *  @return         Pointer to the requested object.
         */
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Virtual function does not override a base-class function.")
        virtual RenderTarget2D* ToRenderTarget2D(Int index = 0) {
            FEATSTD_UNUSED(index);
            return 0;
        }
        const RenderTarget2D* ToRenderTarget2D(Int index = 0) const;

        /**
         *  Casts the object to a ImageSource3D object. May return 0 if this
         *  interface is not supported.
         *  @param index    Defines the type of ImageSource3D to retrieve (e.g.
         *                  ColorBuffer, DepthBuffer). The indices are defined
         *                  by constants.
         *  @return         Pointer to the requested object.
         */
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Virtual function does not override a base-class function.")
        virtual ImageSource3D* ToImageSource3D(Int index = 0) {
            FEATSTD_UNUSED(index);
            return 0;
        }
        const ImageSource3D* ToImageSource3D(Int index = 0) const;

        /**
         *  Casts the object to a RenderTarget3D object. May return 0 if this
         *  interface is not supported.
         *  @param index    Index of requested object, when more then one
         *                  interface is provided.
         *  @return         Pointer to the requested object.
         */
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Virtual function does not override a base-class function.")
        virtual RenderTarget3D* ToRenderTarget3D(Int index = 0) {
            FEATSTD_UNUSED(index);
            return 0;
        }
        const RenderTarget3D* ToRenderTarget3D(Int index = 0) const;

        /**
         *  Casts the object to a Window object. May return 0 if this interface 
         *  is not supported.
         *  @return         Pointer to the requested object.
         */
        virtual Window* ToWindow() { return 0; }
        const Window* ToWindow() const;

        /**
         *  Returns size of this graphic device unit in bytes.
         *  @return     An estimation of the size this graphic device unit 
         *              occupies in memory.
         */
        virtual UInt GetSize() const { return 0; }

        /**
         *  Returns a type handle for the graphic device unit that can be used 
         *  in a factory to create an object with the same type.
         *  @return     Graphic device unit type handle.
         */
        virtual GraphicDeviceUnitTypeHandle GetUnitType() const = 0;

        /// @cond excluded from doxygen
        CdaForeignGduDef(GraphicDeviceUnit)
            CdaGduProperties()

                CdaGduProperty(AutoSwapEnabled, bool,
                    IsAutoSwapEnabled,
                    SetAutoSwapEnabled)
                    CdaDescription("Enable auto-swapping on the 3D render "
                                   "target of this gdu.")
                    CdaVisibilityTest(ToRenderTarget3D)
                    CdaUpdateType(ContinuousUpdateType)
                    CdaCategory("Render Target")
                CdaGduPropertyEnd()

                CdaGduProperty(AutoSwapEnabled2D, bool,
                    IsAutoSwap2DEnabled,
                    SetAutoSwap2DEnabled)
                    CdaDescription("Enable auto-swapping on the 2D render "
                                   "target of this gdu.")
                    CdaVisibilityTest(ToRenderTarget2D)
                    CdaUpdateType(ContinuousUpdateType)
                    CdaCategory("Render Target")
                CdaGduPropertyEnd()
                
                CdaGduProperty(AutoClearEnabled, bool,
                    IsAutoClearEnabled,
                    SetAutoClearEnabled)
                    CdaDescription("Enable auto-clearing on the 3D render "
                                   "target of this gdu.")
                    CdaVisibilityTest(ToRenderTarget3D)
                    CdaUpdateType(ContinuousUpdateType)
                    CdaCategory("Render Target")
                CdaGduPropertyEnd()
                
                CdaGduProperty(ClearMode, SharedClearMode::SharedPointer,
                    GetClearMode,
                    SetClearMode)
                    CdaDescription("Set clear settings on the 3D render "
                                   "target of this gdu.")
                    CdaFieldVisibilityTest(IsAutoClearEnabled())
                    CdaUpdateType(ContinuousUpdateType)
                    CdaCategory("Render Target")
                CdaGduPropertyEnd()

                CdaGduProperty(AutoClearEnabled2D, bool,
                    IsAutoClear2DEnabled,
                    SetAutoClear2DEnabled)
                    CdaDescription("Enable auto-clearing on the 2D render "
                                   "target of this gdu.")
                    CdaVisibilityTest(ToRenderTarget2D)
                    CdaUpdateType(ContinuousUpdateType)
                    CdaCategory("Render Target")
                CdaGduPropertyEnd()
                
                CdaGduProperty(ClearMode2D, SharedClearMode2D::SharedPointer,
                    GetClearMode2D,
                    SetClearMode2D)
                    CdaDescription("Set clear settings on the 2D render "
                                   "target of this gdu.")
                    CdaFieldVisibilityTest(IsAutoClear2DEnabled())
                    CdaUpdateType(ContinuousUpdateType)
                    CdaCategory("Render Target")
                CdaGduPropertyEnd()

            CdaGduPropertiesEnd()
        CdaGduDefEnd()
        /// @endcond

    protected:
        friend class Internal::GraphicDeviceUnitOwnerAccess;
        virtual Internal::GraphicDeviceUnitOwnerAccess GetOwnerAccess() { return Internal::GraphicDeviceUnitOwnerAccess(); }
        virtual void SetOwner(RenderTarget3D* /*owner*/) {}

        bool IsUploadPresequisiteFullfiled() const;

    private:
        void SetAutoSwapEnabled(bool enabled);
        bool IsAutoSwapEnabled() const;

        void SetAutoSwap2DEnabled(bool enabled);
        bool IsAutoSwap2DEnabled() const;

        void SetAutoClearEnabled(bool enabled);
        bool IsAutoClearEnabled() const;

        void SetAutoClear2DEnabled(bool enabled);
        bool IsAutoClear2DEnabled() const;

        void SetClearMode(SharedClearMode::SharedPointer clearMode);
        SharedClearMode::SharedPointer GetClearMode() const;

        void SetClearMode2D(SharedClearMode2D::SharedPointer clearMode);
        SharedClearMode2D::SharedPointer GetClearMode2D() const;


        Int m_display;

        const Char* m_name;
};

/** @}*/ //end of CommonDevice

}   // namespace Candera

#endif  // CANDERAPLATFORM_GRAPHIC_DEVICE_UNIT_H
