//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################
#ifndef CANDERA_ImportedTextureImage_h
#define CANDERA_ImportedTextureImage_h

#include <Candera/Environment.h>
#include <FeatStd/Util/Optional.h>
#include <FeatStd/Container/Vector.h>
#include <Candera/Engine3D/Core/TextureImage.h>
#include <CanderaPlatform/Device/Common/Base/ImageSource3D.h>
#include <FeatStd/Diagnostics/Measurable.h>

namespace Candera {
    class ExternalTextureImage : public TextureImage, public FeatStd::Diagnostics::Measurable {
        FEATSTD_TYPEDEF_BASE(TextureImage);
    public:
        ExternalTextureImage();
        virtual ~ExternalTextureImage(){}

        /**
        *  Defines if MipMap-chain shall be attached to TextureImage.
        *  By default this is disabled.
        *  @param enableMipMapping True to enable, false to disable MipMapping.
        *  @return True if successful, false if Uploaded.
        */
        bool SetMipMappingEnabled(bool enableMipMapping);

        // Overridden function of TextureImage.
        virtual bool IsMipMappingEnabled() const override;

        // Overridden function of TextureImage.
        virtual ImageSource3D* ToImageSource3D() override;




        /**
        *  Sets the width of the texture image. Width must be 16 pixel aligned.
        *  The default value of this parameter is 64.
        *  Call this function before Upload. If called on uploaded image it
        *  will fail.
        *  @param width Width of the texture image.
        *  @return True if successful, false if Uploaded.
        */
        bool SetWidth(Int width);

        /**
        *  Retrieves width set to this Texture object.
        *  @return Width of the texture image.
        */
        Int GetWidth() const;

        /**
        *  Sets the height of the texture image.
        *  The default value of this parameter is 64.
        *  @param height Height of the texture image.
        *  @return True if successful, false if Uploaded.
        */
        bool SetHeight(Int height);

        /**
        *  Retrieves height set to this Texture object.
        *  @return Height of the texture image.
        */
        Int GetHeight() const;



        FEATSTD_RTTI_DECLARATION();
    protected:

        virtual void DisposeInternal() override;
        FeatStd::Initialized<Int> m_width;
        FeatStd::Initialized<Int> m_height;
    private:


        class ExternalTextureImageSource3DInstance : public ImageSource3D {
            friend class ExternalTextureImage;

        public:
            ExternalTextureImageSource3DInstance() : m_externalTextureImage(0) {}
            ~ExternalTextureImageSource3DInstance() { m_externalTextureImage = 0; }

            virtual Handle GetVideoMemoryHandle() const { return (m_externalTextureImage != 0)?m_externalTextureImage->GetVideoMemoryHandle():0; }
            virtual bool IsMipMappingEnabled() const { return (m_externalTextureImage != 0)?m_externalTextureImage->IsMipMappingEnabled():false; }
            virtual Int GetHeight() const { return (m_externalTextureImage != 0)?static_cast<Int>(m_externalTextureImage->GetHeight()):0; }
            virtual Int GetWidth() const { return (m_externalTextureImage != 0)?static_cast<Int>(m_externalTextureImage->GetWidth()):0; }
            virtual void Sync() {}
            virtual void WaitSync() {}
            void SetExternalTextureImage(const ExternalTextureImage* image) { m_externalTextureImage = image; }

            virtual TextureImage::TextureTargetType GetTextureTarget() const override{ return (m_externalTextureImage != 0 ? m_externalTextureImage->GetTextureTargetType() : TextureImage::Texture2D); }

        private:
            const ExternalTextureImage* m_externalTextureImage;


        };

        bool m_mipMappingEnabled;
        protected:
            ExternalTextureImageSource3DInstance m_imageSource3DInstance;

    };
}

#endif // CANDERA_ImportedTextureImage_h
