//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifdef CANDERA_2D_ENABLED

#if !defined(CANDERAPLATFORM_EFFECT_2D_META_INFORMATION_H)
    #define CANDERAPLATFORM_EFFECT_2D_META_INFORMATION_H

#include <CanderaPlatform/Device/Common/Base/DevicePackageInterface.h>
#include <Candera/Engine2D/Property/Property.h>
#include <Candera/System/MetaInfo/MetaInfo.h>
#include <Candera/System/MetaInfo/PublicMacros.h>

namespace Candera {

    class Effect2D;
    class Image2D;

    namespace MetaInfo {

        /** @addtogroup CommonDevice
         *  @{
         */
        typedef PropertyMetaInfo<Effect2D> Effect2DPropertyMetaInfo;
        
        /**
         *  @brief Meta info extensions for the Effect2D.
         */
        class Effect2DMetaInfo : public ObjectMetaInfo<Effect2D> {
            public:

                /**
                 * Types of brush effects.
                 */
                enum EffectType {
                    Invalid,        ///< Invalid brush effect.

                    BrushEffect,    ///< Brushes define their input based alone on their parameters and need to be top in the effect chain.
                    InplaceEffect,  ///< InplaceEffects require an input and produce output. They can be second up to last in the effect chain.
                    BlendEffect,    ///< BlendEffects require input and are capable of reading back the target they render. They are required to be last in the effect chain.
                    CombinedEffect  ///< combined effects are a combination of an arbitrary number of other specific effects in a specific sequence in order to have them processed in a more optimized way like a single pass.
                };

                Effect2DMetaInfo(const Char* name) : ObjectMetaInfo<Effect2D>(name) { }

                virtual ~Effect2DMetaInfo() {}

                virtual MemoryManagement::SharedPointer<Effect2D> Create() = 0;

            #if defined(CANDERA_META_DESCRIPTION) // only for scene composer or editors which need to care about ordering effects
                virtual EffectType GetEffectType() = 0;
                virtual EffectType GetFirstCombinedEffectType() { return Invalid; }
                virtual EffectType GetLastCombinedEffectType() { return Invalid; }
            #endif
        };

        class Effect2DSetMetaInfo : public MetaInfoBase {
            public:
                Effect2DSetMetaInfo(const Char *name) : MetaInfoBase(name) { }

                /** Lookup Effect2D type meta information object with the given name.
                 *  @param name     Name of widget type.
                 *  @return         Pointer to meta information object if found, null otherwise
                 */
                virtual Effect2DMetaInfo* LookupItem(const Char *name) const = 0;

                /** Get Effect2D type meta information object at given index.
                 *  @param idx  Index of the item to retrieve.
                 *  @return     Pointer to meta information object.
                 */
                virtual Effect2DMetaInfo* GetItem(Int32 idx) const = 0;

                /** Return the number of Effect2D meta information objects in the widget set.
                 *  @return     Number of widgets in the widget set.
                 */
                virtual Int32 GetItemCount() const = 0;
        };

        namespace Internal {
            template<>
            struct DataType<Image2D*>
            {
                static bool ConvertFromString(Image2D* &val, const Char *buf)
                {
                    val = FeatStd::aligned_cast<Image2D*>(const_cast<Char*>(buf));

                    return true;
                }

            #ifdef CANDERA_META_DESCRIPTION
                static const Char* GetEditor() { return "builtin://Image2DEditor"; }
            #endif
            };

        }


/** @}*/ //end of CommonDevice

    }   // namespace MetaInfo

}   // namespace Candera

#define EFFECT2D_TYPE_BRUSH Candera::MetaInfo::Effect2DMetaInfo::BrushEffect
#define EFFECT2D_TYPE_INPLACE Candera::MetaInfo::Effect2DMetaInfo::InplaceEffect
#define EFFECT2D_TYPE_BLEND Candera::MetaInfo::Effect2DMetaInfo::BlendEffect
#define EFFECT2D_TYPE_COMBINED Candera::MetaInfo::Effect2DMetaInfo::CombinedEffect
#define EFFECT2D_TYPE_INVALID Candera::MetaInfo::Effect2DMetaInfo::Invalid

#if defined (CANDERA_META_DESCRIPTION)
    #define _CdaEffectType(effectType) virtual Candera::MetaInfo::Effect2DMetaInfo::EffectType GetEffectType() { return effectType; }
    #define _CdaCombinedEffect(firstEffectType, lastEffectType) \
        virtual EffectType GetFirstCombinedEffectType() { return firstEffectType; }\
        virtual EffectType GetLastCombinedEffectType() { return lastEffectType; }
#else
    #define _CdaEffectType(effectType)
    #define _CdaCombinedEffect(firstEffectType, lastEffectType)
#endif

/** Required definition to add meta information to an Effect2D. Each effect may only belong to only one type of effects */
#define CdaEffect2DDef(type, typeName, effectType) \
    _CdaMetaInfoObjectProlog(type, typeName, Candera::Effect2D, MetaInfo, Candera::MetaInfo::Effect2DMetaInfo) \
        _CdaEffectType(effectType) \
        virtual Candera::MemoryManagement::SharedPointer<Candera::Effect2D> Create() { return type::Create(); }

#define CdaEffect2DCombinedDef(type, typeName, firstCombinedEffectType, lastCombinedEffectType)\
    _CdaMetaInfoObjectProlog(type, typeName, Candera::Effect2D, MetaInfo, Candera::MetaInfo::Effect2DMetaInfo) \
        _CdaEffectType(EFFECT2D_TYPE_COMBINED) \
        _CdaCombinedEffect(firstCombinedEffectType, lastCombinedEffectType) \
        virtual Candera::MemoryManagement::SharedPointer<Candera::Effect2D> Create() { return type::Create(); }

#define CdaEffect2DAbstractDef(type, typeName) \
    _CdaMetaInfoObjectProlog(type, typeName, Candera::Effect2D, MetaInfo, Candera::MetaInfo::Effect2DMetaInfo) \
        _CdaEffectType(EFFECT2D_TYPE_INVALID) \
        virtual Candera::MemoryManagement::SharedPointer<Candera::Effect2D> Create() { return Candera::MemoryManagement::SharedPointer<Candera::Effect2D>(); }

/** Closes the effect meta information definition of an Effect2D */
#define CdaEffect2DDefEnd() \
    _CdaMetaInfoObjectEpilog() \
    virtual Candera::MetaInfo::Effect2DMetaInfo* GetMetaInfo() const { return &Candera::MetaInfo::AggregatableInstanceProvider<MetaInfo>(); }


/** Opens a property definition section within a effect definition section. @see CdaEffect2DDef */
#define CdaEffect2DProperties() \
    _CdaPropertySetProlog(Candera::MetaInfo::Effect2DPropertyMetaInfo)

/** Closes a property definition section within a effect definition section. @see CdaEffect2DDef */
#define CdaEffect2DPropertiesEnd() \
    _CdaPropertySetEpilog()

/** Imports properties from some member effect
    @param name name of the import
    @param type type of the member
    @param fieldName name of the member
  */
#define CdaEffect2DImportProperties(name, type, fieldName) \
    _CdaStructureImportProperties(name, type, fieldName, MetaInfo)

/** Imports properties from some base effect
    @param name name of the import
    @param type type of the base effect
  */
#define CdaEffect2DImportBaseClassProperties(name, type) \
    _CdaBaseClassImportProperties(name, type, MetaInfo)


/** Opens a property definition section within a properties definition section. @see CdaProperties
    @param name name of the property (e.g. TextColor)
    @param type type of the property (e.g. Color) */
#define CdaEffect2DProperty(name, type, prop) \
    _CdaFieldPropertyDef(name, type, prop)

/**
*  Opens a property definition section within a properties definition section (@see CdaProperties)
*     @param name name of the property (e.g. Position)
*     @param type type of the property (e.g. Point3D)
*     @param getter getter function of the widget to get property value (e.g. GetPosition)
*     @param setter setter function of the widget to set property value (e.g. SetPosition)
*/
#define CdaIndirectEffect2DProperty(name, type, getter, setter) \
    _CdaPropertyProlog(name, type) \
    typedef type PropertyType; \
    _CdaPropGetter(getter) \
    _CdaPropSetter(setter) \
    _CdaIndirectAPSFactory(name, setter, getter)

/** Opens a property definition section within a properties definition section. @see CdaProperties
    @param name name of the property (e.g. TextColor)
    @param type type of the property (e.g. Color) */
#define CdaEffect2DDisposableProperty(name, type, prop) \
    _CdaFieldPropertyDef(name, type, prop)

/** Closes a property definition section within */
#define CdaEffect2DPropertyEnd() \
    _CdaPropertyEpilog()


/** Opens a Effect2D set definition.
    @param name of the widget set */
#define CdaEffect2DSet(name) \
    typedef struct name : public Candera::MetaInfo::Effect2DSetMetaInfo { \
        _CdaMetaInfoCtors(name, name, Candera::MetaInfo::Effect2DSetMetaInfo)

/** Closes a widget set definition. */
#define CdaEffect2DSetEnd() \
    } _CdaUniqueName(Effect2DSet); \
    Candera::MetaInfo::Effect2DSetMetaInfo* GetEffect2DSet() { \
        static _CdaUniqueName(Effect2DSet) effectSet; \
        return &effectSet; \
    }

/** Opens an effect enumeration section within an effect set section*/
#define CdaEffect2DCollection() _CdaStaticAggregationProlog(Effects, Candera::MetaInfo::Effect2DMetaInfo)

/** Closes an effect enumeration section within an effect set section */
#define CdaEffect2DCollectionEnd() _CdaStaticAggregationEpilog(Effects, Effects)

/** Adds the given effect to the effect set within a #CdaEffect2DCollection section
    @param name of the widget type */
#define CdaEffect2D(typeName) \
    _CdaStaticAggregation(typeName::MetaInfo)

namespace Candera {

    /** Provides the widget set meta information
        @return pointer to widget meta information */
    Candera::MetaInfo::Effect2DSetMetaInfo* GetEffect2DSet();

}   // namespace Candera

#endif  // CANDERAPLATFORM_EFFECT_2D_META_INFORMATION_H

#endif //CANDERA_2D_ENABLED
