//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERAPLATFORM_DEVICE_PACKAGE_TRACE_H
    #define CANDERAPLATFORM_DEVICE_PACKAGE_TRACE_H

#include <Candera/Environment.h>
#include <Candera/System/Diagnostics/Log.h>

/**
 * Macro used to declare a new error tracer.
 * To implement this class one needs to implement 3 methods:
 * bool IsErrorSet();
 * bool WasLastErrorSet();
 * const Char* GetLastErrorText();
 * Use CANDERA_DEVICE_PACKAGE_TRACE_CLASS to generate the class name
 * for the implementation.
 * @param DeviceType error tracer name
 */
#define CANDERA_DEVICE_PACKAGE_TRACE(DeviceType)            \
    class FEATSTD_CONCAT2(DeviceType, DevicePackageTrace) { \
    public:                                                 \
        static bool IsErrorSet();                           \
        static bool WasLastErrorSet();                      \
        static const Char* GetLastErrorText();              \
    };

/**
 * Macro used to generate the error tracer name. Used in
 * implementations of error tracers
 * @param DeviceType error tracer type name
 */
#define CANDERA_DEVICE_PACKAGE_TRACE_CLASS(DeviceType) \
    FEATSTD_CONCAT2(DeviceType, DevicePackageTrace)

/**
 * Macros used to set log messages.
 * @param arg text to associate with the error trace
 */
#define CANDERA_DEVICE_LOG_DEBUG(args, ...) \
    FEATSTD_LOG_REALM(Diagnostics::LogRealm::CanderaPlatformDevice, FeatStd::Diagnostics::LogLevel::Debug,   args, ## __VA_ARGS__)
#define CANDERA_DEVICE_LOG_INFO(args, ...) \
    FEATSTD_LOG_REALM(Diagnostics::LogRealm::CanderaPlatformDevice, FeatStd::Diagnostics::LogLevel::Info,    args, ## __VA_ARGS__)
#define CANDERA_DEVICE_LOG_WARN(args, ...) \
    FEATSTD_LOG_REALM(Diagnostics::LogRealm::CanderaPlatformDevice, FeatStd::Diagnostics::LogLevel::Warning, args, ## __VA_ARGS__)
#define CANDERA_DEVICE_LOG_ERROR(args, ...) \
    FEATSTD_LOG_REALM(Diagnostics::LogRealm::CanderaPlatformDevice, FeatStd::Diagnostics::LogLevel::Error,   args, ## __VA_ARGS__)
#define CANDERA_DEVICE_LOG_FATAL(args, ...) \
    FEATSTD_LOG_REALM(Diagnostics::LogRealm::CanderaPlatformDevice, FeatStd::Diagnostics::LogLevel::Fatal,   args, ## __VA_ARGS__)

/**
 * Macros used to trace errors.
 * @param DeviceType error tracer type name
 * @param arg text to associate with the error trace
 */
#define CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(DeviceType, arg) \
    CANDERA_DEVICE_CHECK_AND_LOG(DeviceType, Diagnostics::LogRealm::CanderaPlatformDevice, FeatStd::Diagnostics::LogLevel::Debug, arg)
#define CANDERA_DEVICE_CHECK_AND_LOG_INFO(DeviceType, arg) \
    CANDERA_DEVICE_CHECK_AND_LOG(DeviceType, Diagnostics::LogRealm::CanderaPlatformDevice, FeatStd::Diagnostics::LogLevel::Info, arg)
#define CANDERA_DEVICE_CHECK_AND_LOG_WARN(DeviceType, arg) \
    CANDERA_DEVICE_CHECK_AND_LOG(DeviceType, Diagnostics::LogRealm::CanderaPlatformDevice, FeatStd::Diagnostics::LogLevel::Warning, arg)
#define CANDERA_DEVICE_CHECK_AND_LOG_ERROR(DeviceType, arg) \
    CANDERA_DEVICE_CHECK_AND_LOG(DeviceType, Diagnostics::LogRealm::CanderaPlatformDevice, FeatStd::Diagnostics::LogLevel::Error, arg)
#define CANDERA_DEVICE_CHECK_AND_LOG_FATAL(DeviceType, arg) \
    CANDERA_DEVICE_CHECK_AND_LOG(DeviceType, Diagnostics::LogRealm::CanderaPlatformDevice, FeatStd::Diagnostics::LogLevel::Fatal, arg)

#define CANDERA_DEVICE_CHECK_AND_LOG(DeviceType, logRealm, logLevel, arg)                                                           \
    if (CANDERA_DEVICE_PACKAGE_TRACE_CLASS(DeviceType)::IsErrorSet()) {                                                             \
        FEATSTD_LOG_REALM(logRealm, logLevel, "%s: %s", arg, CANDERA_DEVICE_PACKAGE_TRACE_CLASS(DeviceType)::GetLastErrorText());   \
    }

#endif  // CANDERAPLATFORM_DEVICE_PACKAGE_TRACE_H
