//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERAPLATFORM_CONTEXT_RESOURCE_POOL_H)
    #define CANDERAPLATFORM_CONTEXT_RESOURCE_POOL_H

#include <Candera/Environment.h>

#include <CanderaPlatform/Device/Common/Base/ContextProvider3D.h>
#include <Candera/System/Container/ForwardIterator.h>

namespace Candera {
/** @addtogroup CommonDevice
 *  @{
 */

/**
 *  @brief Context Resource Pool is used to share resources among different 
 *  Context Providers.
 *
 *  It allows grouping of render targets and associated device objects (Shaders,
 *  Texture Images, Vertex Buffers, etc.) in a common pool. The number of 
 *  Context Resource Pools at one time cannot exceed CANDERA_MAX_CONTEXT_COUNT.
 */
class ContextResourcePool {
private:
    ///> Type of the context provider list.
    typedef Internal::LinkedList<ContextProvider3D, &ContextProvider3D::m_contextProviderNode> ContextProviderList;
public:
    ///> Iterator of contexts stored within this Pool.
    typedef ForwardIterator<ContextProviderList::Iterator> ContextProviderIterator;

    /**
     * Destructs a ContextResourcePool object. Deliberately non-polymorphic.
     */
    ~ContextResourcePool();

    /**
     * Activate this ContextResourcePool object, and thus the shared 
     * context. All DeviceObjects uploaded after this call will be uploaded
     * to this context resource pool.
     * @return  true if the ContextResourcePool was activated properly.
     */
    bool Activate();

    /**
     * Retrieve the unique numeric identifier of the current context 
     * resource pool.
     * An index is a number between 0 and CANDERA_MAX_CONTEXT_COUNT - 1.
     * @return  The unique ID of this ContextResourcePool object.
     */
    Int GetIndex() const { return m_index; }

    /**
     * Get a native handle for the shared resource context provided by one 
     * of the subscribed context providers. This can be used to check 
     * whether a shared context is available.
     * @return  Handle to the shared native device context, 0 if no shared 
     *          context handle is available.
     */
    void* GetSharedContextHandle() const;

    /**
     * Retrieve an iterator to iterate through context providers.
     * This allows manipulation if each context provider in this pool.
     * @return  Iterator pointing to the first ContextProvider3D.
     */
    ContextProviderIterator GetContextProviderIterator() const;

    /**
     * Retrieve the next context resource pool.
     * As all ContextResourcePools are stored in a ring buffer, the
     * ContextResourcePool at the beginning acts as a sentinel
     * to terminate a loop.
     * @return  The next context resource pool.
     */
    ContextResourcePool& GetNext();

    /**
     * Retrieve the active context resource pool. It is guaranteed that one
     * resource pool is always active. If no resource pool was set active 
     * explicitly, the default resource pool is returned. If no 
     * ContextProvider was Uploaded to the active resource pool, no shared 
     * context handle is available and calls to rendering API (such as to 
     * RenderDevice) will fail.
     * @return  The active context resource pool.
     */
    static ContextResourcePool& GetActive();

    /**
     * Retrieve the default context resource pool. This resource pool is 
     * used whenever a resource pool is required and none is provided from 
     * outside. This is the resource pool with index 0.
     * @return  The default context resource pool.
     */
    static ContextResourcePool& GetDefault();

    /**
     *  Provides the number of available context resource pools.
     *  @return An integer containing the number of context resource pools.
     */
    static Int GetCount();

    /**
     * Retrieve a context resource pool by index.
     * @param index The index of the requested context resource pool.
     * @return  The context resource pool with the given index. 0 if
     *          index is out of bounds.
     */
    static ContextResourcePool* GetContextResourcePool(Int index);

private:
    friend class ContextProvider3D;
    friend struct ContextResourcePoolArrayItem;
    friend struct ContextResourcePoolSubject;

    //disallow copying of ContextResourcePool objects.
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1704, CANDERA_LINT_REASON_NONCOPYABLE)
    FEATSTD_MAKE_CLASS_UNCOPYABLE(ContextResourcePool);

    // Index of context resource pool. The index is in the range of 0 and 
    // CANDERA_MAX_CONTEXT_COUNT - 1.
    Int m_index;

    /**
     * Constructor used to create resource pools internally.
     */
    ContextResourcePool(Int index);

    /**
     * Adds a context provider to the current resource pool.
     * If it is the first context provider added, its contextHandle will act
     * as shared contextHandle for all other context providers added later.
     * @param contextProvider   Context provider to add to ContextResourcePool.
     * @return                  true if registration was successful, 
     *                          false otherwise.
     */
    void AddContextProvider3D(ContextProvider3D* contextProvider);

    /**
     * Removes a context provider from the current resource pool.
     * @param contextProvider   Object to remove from ContextResourcePool.
     * @return                  true if contextProvider given has been 
     *                          removed successfully, false otherwise.
     */
    void RemoveContextProvider3D(ContextProvider3D* contextProvider);

    /**
     * Set active context resource pool.
     * @param pool  Specifies the ContextResourcePool object that is 
     *              currently active.
     */
    static void SetActiveContextResourcePool(ContextResourcePool* pool);
};

/** @}*/ //end of CommonDevice
}   // namespace Candera
#endif  // CANDERAPLATFORM_CONTEXT_RESOURCE_POOL_H
