//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ContextResourcePool.h"
#include "DevicePackageTrace.h"

#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/Util/ObjectBufferTypeProvider.h>

namespace Candera
{

/*
 * Resource pool lifetime management
 */

/**
 * @brief Context resource pool acts as a proxy of this class.
 */
struct ContextResourcePoolSubject
{
    ContextResourcePool::ContextProviderList m_contextProviderList;
};

/**
 * @brief Wrapper class for ContextResourcePool, as part of the resource pool
 * array.
 *
 * This class provides the means of working around the deprecated
 * ContextResourcePool constructor.
 */
struct ContextResourcePoolArrayItem 
{
    ContextResourcePoolArrayItem(Int index) :
        m_subject(),
        m_proxy(index)
    {
    }

    ContextResourcePoolSubject m_subject;
    ContextResourcePool m_proxy;
};

static const Int c_contextCount = CANDERA_MAX_CONTEXT_COUNT;

/**
 * @brief The array of context resource pools.
 */
class ContextResourcePoolArray
{
public:
    ContextResourcePoolArray()
    {
        for (Int i = 0; i < c_contextCount; i++) {
            FEATSTD_LINT_NEXT_EXPRESSION(586, "placement operator new to construct a new object of type ContextResourcePoolArrayItem regarded to be safe.")
            (void)new(&Get(i)) ContextResourcePoolArrayItem(i);
        }
    }
    ~ContextResourcePoolArray()
    {
        for (Int i = 0; i < c_contextCount; i++) {
            FEATSTD_LINT_NEXT_EXPRESSION(586, "destructor call on object recently created by placement operator new regarded to be safe.")
            Get(i).~ContextResourcePoolArrayItem();
        }
    }
    ContextResourcePoolArrayItem& Get(Int i)
    {
        return FeatStd::Internal::PointerToPointer<ContextResourcePoolArrayItem*>(m_array)[i];
    }
private:
    typedef ContextResourcePoolArrayItem DestinationType[c_contextCount];
    typedef FeatStd::ObjectBufferTypeProvider<DestinationType>::Type HostType;

    HostType m_array;
};
/**
 * Retrieve an array of context resource pools.
 * @return A reference of the static array.
 */
static ContextResourcePoolArray& GetContextResourcePoolArray()
{
    static ContextResourcePoolArray s_array;
    return s_array;
}
/**
 * Retrieve a context resource pool from the array.
 * @return A reference to the contextResourcePool.
 */
static ContextResourcePool& GetContextResourcePoolByIndex(Int index)
{
    return GetContextResourcePoolArray().Get(index).m_proxy;
}
/**
 * Retrieve the subject of a context resource pool proxy from the array.
 * @return A reference to the ContextResourcePoolSubject.
 */
static ContextResourcePoolSubject& GetContextResourcePoolSubjectByIndex(Int index)
{
    return GetContextResourcePoolArray().Get(index).m_subject;
}
/**
 * Retrieve the size of the context resource pool array.
 * @return Number of elements in the array.
 */
static Int GetContextResourcePoolArraySize()
{
    return CANDERA_MAX_CONTEXT_COUNT;
}

/*
 * Default and Active resource pool control
 */
static ContextResourcePool* DefaultContextResourcePool()
{
    return &GetContextResourcePoolByIndex(0);
}

static ContextResourcePool* &ActiveContextResourcePool() {
    static ContextResourcePool* activeContextResourcePool = 
        DefaultContextResourcePool();
    return activeContextResourcePool;
}


ContextResourcePool::ContextResourcePool(Int index) :
    m_index(index)
{
}

ContextResourcePool::~ContextResourcePool()
{
    if (&GetActive() == this) {
        SetActiveContextResourcePool(0);
    }
}

bool ContextResourcePool::Activate()
{
    for(ContextProviderIterator it = GetContextProviderIterator(); it.IsValid(); ++it) {
        if (it->Activate()) {
            // Set active context resource pool if native context has been activated successfully.
            SetActiveContextResourcePool(this);
            return true;
        }
    }

    return false;
}

void* ContextResourcePool::GetSharedContextHandle() const
{
    for(ContextProviderIterator it = GetContextProviderIterator(); it.IsValid(); ++it) {
        void *contextHandle = it->GetContextHandle();
        if (contextHandle != 0) {
            return contextHandle;
        }
    }
    return 0;
}

ContextResourcePool::ContextProviderIterator ContextResourcePool::GetContextProviderIterator() const
{
    if (m_index >= 0) {
        ContextResourcePoolSubject& subject = 
            GetContextResourcePoolSubjectByIndex(m_index);
        return ContextProviderIterator(
            subject.m_contextProviderList.Begin(),
            subject.m_contextProviderList.End());
    }
    return ContextProviderIterator(
        ContextProviderList().End(),
        ContextProviderList().End());
}

ContextResourcePool& ContextResourcePool::GetActive()
{
    return *ActiveContextResourcePool();
}

ContextResourcePool& ContextResourcePool::GetDefault()
{
    return *DefaultContextResourcePool();
}

ContextResourcePool& ContextResourcePool::GetNext()
{
    if (m_index < 0) {
        return *this;
    }

    Int index = GetIndex() + 1;
    if (index == GetCount()) {
        index = 0;
    }
    return *GetContextResourcePool(index);
}

Int ContextResourcePool::GetCount()
{
    return GetContextResourcePoolArraySize();
}

ContextResourcePool* ContextResourcePool::GetContextResourcePool(Int index)
{
    return ((index >= 0) && (index < GetCount())) ?
        &GetContextResourcePoolByIndex(index) :
        0;
}

CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::ContextResourcePool::AddContextProvider3D, CANDERA_LINT_REASON_NONCONSTMETHOD)
void ContextResourcePool::AddContextProvider3D(ContextProvider3D* contextProvider)
{
    if (m_index < 0) {
        return;
    }
    ContextResourcePoolSubject& subject = 
        GetContextResourcePoolSubjectByIndex(m_index);
    subject.m_contextProviderList.Prepend(contextProvider);
}

CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::ContextResourcePool::RemoveContextProvider3D, CANDERA_LINT_REASON_NONCONSTMETHOD)
void ContextResourcePool::RemoveContextProvider3D(ContextProvider3D* contextProvider)
{
    if (m_index < 0) {
        return;
    }
    ContextResourcePoolSubject& subject = 
        GetContextResourcePoolSubjectByIndex(m_index);
    static_cast<void>(subject.m_contextProviderList.Remove(contextProvider));
}

void ContextResourcePool::SetActiveContextResourcePool(ContextResourcePool* pool)
{
    if (pool == 0) {
        ActiveContextResourcePool() = DefaultContextResourcePool();
    }
    else {
        if (pool->m_index >= 0) {
            ActiveContextResourcePool() = pool;
        }
    }
}
} //namespace Candera

