//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GLOBALIZATION_CONDITION_H)
#define CANDERA_GLOBALIZATION_CONDITION_H

#include <Candera/Environment.h>
#include <FeatStd/Util/String.h>
#include <FeatStd/Util/PointerUtil.h>

#include <Courier/DataBinding/DataItemValue.h>

namespace Candera { namespace Globalization { namespace Internal {
/// @addtogroup CanderaGlobalizationLocalizable
/// @{
typedef const Char* Handle;

static Handle GetHandleForOffset(const UInt16& offset) {
    //return reinterpret_cast<Handle>(&offset) + offset;
    return FeatStd::Internal::PointerAdd<Handle>(&offset, static_cast<OffsetType>(offset));
}

enum ValueType {
    InvalidValueType = 0,       ///< Invalid value type
    Int32ValueType   = 1,       ///< Int32 value type
    FloatValueType   = 2,       ///< Float value type
    StringValueType  = 3,       ///< String value type
    BooleanValueType = 4        ///< Boolean value type
};

/**
    * Each operand in a condition may either be a reference to a parameter
    * or a constant value. In both cases, the parameter index or constant
    * value is stored in the m_data member. The stored value must not be
    * bigger than 32 bit.
    */
class OperandValue {
public:
    OperandValue() : m_data(0) {}

    /**
        * @param data Either a constant value or a parameter index.
        */
    explicit OperandValue(const Int32& data) : m_data(data) {}

    /**
        * Retrieves the constant value or parameter index and casts it to the appropriate type
        * in the process.
        * @return   The constant value or parameter index.
        */
    template<typename T> const T* GetValue() const { return FeatStd::Internal::PointerToPointer<const T*>(&m_data); }

protected:
    Int32 m_data;
};

/**
    * Represents the block of a texts parameter definitions. Each defined Parameter
    * for a parameterized text has a type and a format string (how to convert the
    * value to a string).
    */
class Parameters {
    public:
        /**
            *  @param  hdl     Handle within the binary data.
            */
        Parameters(Handle hdl) : m_hdl(hdl) {}

        /**
            *  Retrieves the amount of parameters defined for the parameterized text.
            *  @return     The parameter count.
            */
        UInt8 GetCount() const { return GetHeader().m_paramCount; }

        /**
            *  Retrieves the format string for converting the parameter's value to a string.
            *  @param  index   The index of the parameter.
            *  @return     Pointer to the format string.
            */
        const Char* GetFormatString(UInt8 index) const { return GetHandleForOffset(GetData(index).m_FormatStringOffset); }

        /**
            *  Retrieves the type of the parameter.
            *  @param  index   The index of the parameter.
            *  @return         The type of the parameter with the given index.
            */
        ValueType GetType(UInt8 index) const { return static_cast<ValueType>(GetData(index).m_paramType); }

        /**
            *  Retrieves the key  of the data item to bind to this parameter.
            *  @param  index   The index of the parameter.
            *  @return         The key (hash) of the data item to bind.
            */
        UInt32 GetDataItemHash(UInt8 index) const { return GetData(index).m_dataItemHash; }

    private:
        struct Header {
            UInt8 m_paramCount;
            UInt8 m_padding[3];
        };

        struct Data {
            UInt32 m_dataItemHash;
            UInt16 m_FormatStringOffset;
            UInt8 m_paramType;
            UInt8 m_padding;
        };

        Handle m_hdl;

        const Header& GetHeader() const {  return *FeatStd::Internal::PointerToPointer<const Header*>(m_hdl); }
        const Data* GetData() const { return FeatStd::Internal::PointerToPointer<const Data*>(FeatStd::Internal::PointerAdd<const Char*>(m_hdl, sizeof(Header))); }
        const Data& GetData(UInt8 index) const { return GetData()[index]; }
};

class ParameterizedLocalizableText;

/**
    *  Operand can be a constant value or a parameter index, which is then a variable value.
    */
class Operand {
    public:
        /**
            *  @param  hdl     Handle within the binary data.
            *  @param  text    The parameterized text to be manipulated.
            */
        Operand(Handle hdl, const ParameterizedLocalizableText& text);

        bool operator<(const Operand& rhs) const;
        bool operator<=(const Operand& rhs) const;
        bool operator==(const Operand& rhs) const;
        bool operator!=(const Operand& rhs) const;

    private:
        friend class Condition;

        struct Data {
            UInt8 m_operandType;
            UInt8 m_valueType;
            UInt16 m_padding;
            OperandValue m_value;
        };

        Handle m_hdl;
        Courier::DataItemValue m_value;
        FeatStd::String m_strValue;
        const ParameterizedLocalizableText& m_text;

        enum Type {
            Invalid  = 0,   ///< Invalid.

            Constant = 1,   ///< The operand is a constant value.
            Variable = 2    ///< The operand is one of the variable parameters. The value is the index of the parameter.
        };

        Operand(const Operand&);
        Operand& operator=(const Operand&);

        const Data& GetData() const { return *FeatStd::Internal::PointerToPointer<const Data*>(m_hdl); }
        Type GetOperandType() const { return static_cast<Type>(GetData().m_operandType); }
        ValueType GetValueType() const { return static_cast<ValueType>(GetData().m_valueType); }
        const Courier::DataItemValue& GetValue() const;
};

/**
    *  Operator for the condition.
    */
class Operator {
    public:
        /**
            *  Set of defined operators.
            */
        enum Type {
            Invalid     = 0,    ///< Invalid.

            Less        = 1,    ///< The left operand is less than the right operand.
            LessOrEqual = 2,    ///< The left operand is less than or equal to the right operand.
            Equal       = 3,    ///< The left and right operands are equal.
            NotEqual    = 4     ///< The left and right operands are not equal.
        };

        Operator() : m_type(Invalid) {}
        Operator(UInt8 type) : m_type(type) {}

        Type GetType() const { return static_cast<Type>(m_type); }

    private:
        UInt8 m_type;
};

/**
    *  A FormatPattern hold all data to build the resulting string, including the information
    *  on which position to insert which parameter.
    */
class FormatPattern {
    public:
        /**
            *  @param  hdl     Handle within the binary data.
            *  @param  text    The parameterized text to be manipulated.
            */
        FormatPattern(Handle hdl, const ParameterizedLocalizableText& text) : m_hdl(hdl), m_text(text) {}

        /**
            *  This method is called if the corresponding condition is true to build the resulting string.
            */
        void FormatString();

    private:
        struct Header {
            UInt8 m_paramCount;
            UInt8 m_padding;
        };

        struct Data {
            UInt8 m_paramIndex;
            UInt8 m_padding;
            UInt16 m_paramPos;
        };

        Handle m_hdl;
        const ParameterizedLocalizableText& m_text;

        FormatPattern(const FormatPattern&);
        FormatPattern& operator=(const FormatPattern&);

        const Header& GetHeader() const { return *FeatStd::Internal::PointerToPointer<const Header*>(m_hdl); }
        const Data* GetData() const { return FeatStd::Internal::PointerToPointer<const Data*>(FeatStd::Internal::PointerAdd<const Char*>(m_hdl, sizeof(Header)));}
        const TChar* GetString() const {
            return FeatStd::Internal::PointerToPointer<const TChar*>(FeatStd::Internal::PointerAdd<const Char*>(m_hdl, sizeof(Header) + (GetParameterCount() * sizeof(Data))));
        }
        UInt8 GetParameterCount() const { return GetHeader().m_paramCount; }
        UInt16 GetParameterPosition(UInt8 index) const { return GetData()[index].m_paramPos; }
        UInt8 GetParameterIndex(UInt8 index) const { return GetData()[index].m_paramIndex; }
};

/**
    *  The Condition is a combination of left operand / operator / right operand. It the condition is true,
    *  the corresponding format pattern string can be retrieved.
    */
class Condition {
    public:
        /**
            *  @param  hdl     Handle within the binary data.
            *  @param  text    The parameterized text to be manipulated.
            */
        Condition(Handle hdl, const ParameterizedLocalizableText& text);

        /**
            *  @return     Boolean value whether the condition is true or false.
            */
        bool IsTrue() const;

        /**
            *  @return     The handle to the FormatPattern.
            */
        Handle GetFormatPatternHandle() const { return GetHandleForOffset(GetData().m_formatPatternOffset); }

    private:
        friend class ConditionalFormatPattern;

        struct Data {
            Operand::Data m_leftOperand;
            Operand::Data m_rightOperand;
            Operator m_operator;
            UInt8 m_padding;
            UInt16 m_formatPatternOffset;
        };

        Handle m_hdl;
        const ParameterizedLocalizableText& m_text;

        Condition(const Condition&);
        Condition& operator=(const Condition&);

        const Data& GetData() const { return *FeatStd::Internal::PointerToPointer<const Data*>(m_hdl);}
        Handle GetLeftOperandHandle() const { return reinterpret_cast<Handle>(&GetData().m_leftOperand); }
        Handle GetRightOperandHandle() const { return reinterpret_cast<Handle>(&GetData().m_rightOperand); }
        const Operator& GetOperator() const { return GetData().m_operator; };
};

/**
    *  ConditionalFormatPattern holds the conditions and the corresponding format patterns for the Parameterized Text.
    */
class ConditionalFormatPattern {
    public:
        /**
            *  @param  hdl     Handle within the binary data.
            *  @param  text    The parameterized text to be manipulated.
            */
        ConditionalFormatPattern(Handle hdl, const ParameterizedLocalizableText& text) : m_hdl(hdl), m_text(text) {}

        /**
            *  Resolved the conditions and creates builds the resulting string.
            *  The first condition in the list which is true defines the format pattern.
            */
        void ResolveConditions() const;

    private:
        struct Header {
            UInt8 m_conditionCount;
            UInt8 m_padding;
            UInt16 m_defaultFormatPatternOffset;
        };

        Handle m_hdl;
        const ParameterizedLocalizableText& m_text;

        ConditionalFormatPattern(const ConditionalFormatPattern&);
        ConditionalFormatPattern& operator=(const ConditionalFormatPattern&);

        const Header& GetHeader() const { return *FeatStd::Internal::PointerToPointer<const Header*>(m_hdl);}
        UInt8 GetConditionCount() const { return GetHeader().m_conditionCount; }
        Handle GetDefaultFormatPatternHandle() const { return GetHandleForOffset(GetHeader().m_defaultFormatPatternOffset); }
        Handle GetConditionHandle(UInt8 i) const { return m_hdl + sizeof(Header) + (i * sizeof(Condition::Data)); }
};
/// @}
}}}   // namespace Candera::Globalization::Internal
#endif  // CANDERA_GLOBALIZATION_CONDITION_H
