//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_STATEMACHINE_BEHAVIOR_H
#define CANDERA_STATEMACHINE_BEHAVIOR_H

#include <CanderaAssetLoader/AssetLoaderBase/AssetLoaderBaseDataTypes.h>

#include <CanderaBehavior/BehaviorBase/Behavior.h>

namespace Candera {
/** @addtogroup BehaviorBase
  * @{
  */
    namespace Internal {
        class StateMachineBehaviorBuilder;
        class StateMachineBehaviorActionCall;
        class StateMachineBehaviorData;
        class StateMachineBehaviorConditionEvaluation;
    }

    class DefaultAssetProvider;

    class StateMachineBehavior : public Behavior
    {
    public:
        /**
         * Retrieves the global StateMachine.
         * @return The global state machine.
         */
        static StateMachineBehavior* GetGlobalStateMachine();

        /**
         * Sets the global StateMachine.
         * @param stateMachine The StateMachine to be set.
         */
        static void SetGlobalStateMachine(StateMachineBehavior* stateMachine);

        /**
         * StateMachineBehavior default constructor.
         */
        StateMachineBehavior();

        /**
         * StateMachineBehavior destructor.
         */
        virtual ~StateMachineBehavior();

        /**
         * Finalize the widget to release all referred resources.
         */
        virtual void Finalize() override;

        /**
         * Invoked before rendering. Change node properties here.
         */
        virtual void Update() override;

        /**
         * Enable or disable the consumption of events. Consumed events are not dispatched any further.
         * @param consumeEvent True if events should be consumed, false if events should be dispatched further.
         */
        void SetConsumeEvent(bool consumeEvent);

        /**
         * Retrieves information about the event consumption. Consumed events are not dispatched any further.
         * @return True if the events are set to be consumed, false if they are set to be dispatched further
         */
        bool GetConsumeEvent() const
        {
            return m_consumeEvent;
        }

    protected:
        /**
         * Event handling method implementation with mechanisms to route events to other behaviors.
         * @param event The \event to handle.
         * @param dispatchResult The result.
         */
        virtual void OnEvent(const FeatStd::Event& event, EventDispatchResult& dispatchResult) override;

        virtual void OnAnimationTimeDispatcherChanged();

    private:
        /// @cond Doxygen ignore - start
        CdaBehaviorMixedDef(StateMachineBehavior, Behavior)
            CdaDescription("A state machine is able to implement the complex states of a node/scene/application (including sub state machines and history state machines). Transitions with conditions and actions are used to change between states.")
            CdaReadableName("State Machine")
            CdaCategory("State Machine")
            CdaProperties()
                CdaProperty(StateMachine, Candera::Internal::StateMachineBehaviorData*, GetStateMachine, SetStateMachine)
                    CdaDescription("")
                CdaPropertyEnd()
                CdaProperty(ConsumeEvent, bool, GetConsumeEvent, SetConsumeEvent)
                    CdaDescription("Set this flag to false if the event should not be consumed by the trigger. By default it is set to true and if the trigger evaluates to true the event will not be routed to any other behavior.")
                CdaPropertyEnd()
           CdaPropertiesEnd()
        CdaBehaviorDefEnd()
        /// @endcond Doxygen ignore - end
        
        FEATSTD_MAKE_CLASS_UNCOPYABLE(StateMachineBehavior);

        friend class DefaultAssetProvider;
        friend class Internal::StateMachineBehaviorActionCall;
        friend class Internal::StateMachineBehaviorBuilder;

        Candera::Internal::StateMachineBehaviorData* GetStateMachine() const { return m_data; }
        void SetStateMachine(Candera::Internal::StateMachineBehaviorData*);

        Candera::Internal::StateMachineBehaviorData* m_data;
        bool m_consumeEvent;
    };
/** @} */
} // namespace Candera

#endif
