//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "StateMachineBehavior.h"
#include <FeatStd/Util/StateMachine2.h>
#include <CanderaAssetLoader/AssetLoaderBase/StateMachineData.h>
#include <CanderaBehavior/BehaviorBase/ConditionBehavior.h>
#include <CanderaBehavior/BehaviorBase/ActionBehavior.h>
#ifdef CANDERA_TRANSITIONS_ENABLED
#include <CanderaTransitions/Request.h>
#include <CanderaTransitions/TransitionStrategy.h>
#endif
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Candera {

    CGI_BEHAVIOR_RTTI_DEFINITION(StateMachineBehavior)

    static StateMachineBehavior*& StateMachineBehaviorGlobalStateMachine()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(StateMachineBehavior*, s_globalStateMachine, 0);
        return s_globalStateMachine;
    }

    StateMachineBehavior* StateMachineBehavior::GetGlobalStateMachine()
    {
        return StateMachineBehaviorGlobalStateMachine();
    }

    void StateMachineBehavior::SetGlobalStateMachine(StateMachineBehavior* stateMachine)
    {
        StateMachineBehaviorGlobalStateMachine() = stateMachine;
    }

    StateMachineBehavior::StateMachineBehavior() :
        m_data(0),
        m_consumeEvent(true)
    {
    }

    StateMachineBehavior::~StateMachineBehavior()
    {
        static_cast<void>(SetStateMachine(0));
    }

    void StateMachineBehavior::Finalize()
    {
        if ((0 != m_data) && (m_data->m_initialized) && (m_data->m_states.Size() > 0) && (0 != m_data->m_states[0])) {
            static_cast<Internal::StateMachineBehaviorData::StateMachine*>(m_data->m_states[0])->OnFinalize(*this);
        }
    }

    void StateMachineBehavior::Update()
    {
        if (0 != m_data) {
#ifdef CANDERA_TRANSITIONS_ENABLED
            bool localTransition = false;
            if (!Transitions::Request::IsRequestStarted()) {
                localTransition = true;
                Transitions::Request::Begin();
            }
#endif
            if ((!m_data->m_initialized) && (m_data->m_states.Size() > 0) && (0 != m_data->m_states[0])) {
                static_cast<Internal::StateMachineBehaviorData::StateMachine*>(m_data->m_states[0])->OnInit(*this);
                m_data->m_initialized = true;
            }

            for (SizeType actionIndex = 0; actionIndex < m_data->m_actions.Size(); ++actionIndex) {
                if (0 != m_data->m_actions[actionIndex]) {
                    m_data->m_actions[actionIndex]->Update();
                }
            }

            for (SizeType conditionIndex = 0; conditionIndex < m_data->m_conditions.Size(); ++conditionIndex) {
                if (0 != m_data->m_conditions[conditionIndex]) {
                    m_data->m_conditions[conditionIndex]->Update();
                }
            }
#ifdef CANDERA_TRANSITIONS_ENABLED
            if (localTransition) {
                Transitions::Request::SharedPointer transitionRequest = Transitions::Request::End();
                if (!transitionRequest.PointsToNull()) {
                    Transitions::Rule::Set::SharedPointer transitionRuleSet = DefaultAssetProvider::GetInstance().GetTransitionRuleCollection();
                    if (!transitionRuleSet.PointsToNull()) {
                        Transitions::Transition::Set transitionSet;
                        static_cast<void>(Transitions::TransitionStrategy::GetInstance().Start(*transitionRequest, *transitionRuleSet, transitionSet));
                    }
                }
            }
#endif
        }
    }

    void StateMachineBehavior::SetConsumeEvent(bool consumeEvent)
    {
        if (m_consumeEvent != consumeEvent) {
            m_consumeEvent = consumeEvent;
            WakeUpAllRenderComponents();
        }
    }

    void StateMachineBehavior::OnEvent(const FeatStd::Event& event, EventDispatchResult& dispatchResult)
    {
        if ((0 != m_data) && (m_data->m_states.Size() > 0) && (0 != m_data->m_states[0])) {
#ifdef CANDERA_TRANSITIONS_ENABLED
            bool localTransition = false;
            if (!Transitions::Request::IsRequestStarted()) {
                localTransition = true;
                Transitions::Request::Begin();
            }
#endif
            if (m_data->m_states[0]->OnEvent(event, *this)) {
                dispatchResult.SetHandled();
                if (GetConsumeEvent()) {
                    dispatchResult.StopDispatchingImmediately();
                }
            }
#ifdef CANDERA_TRANSITIONS_ENABLED
            if (localTransition) {
                Transitions::Request::SharedPointer transitionRequest = Transitions::Request::End();
                if (!transitionRequest.PointsToNull()) {
                    Transitions::Rule::Set::SharedPointer transitionRuleSet = DefaultAssetProvider::GetInstance().GetTransitionRuleCollection();
                    if (!transitionRuleSet.PointsToNull()) {
                        Transitions::Transition::Set transitionSet;
                        static_cast<void>(Transitions::TransitionStrategy::GetInstance().Start(*transitionRequest, *transitionRuleSet, transitionSet));
                    }
                }
            }
#endif
        }
    }

    void StateMachineBehavior::OnAnimationTimeDispatcherChanged()
    {
        if (0 != m_data) {

            for (SizeType actionIndex = 0; actionIndex < m_data->m_actions.Size(); ++actionIndex) {
                if (0 != m_data->m_actions[actionIndex]) {
                    m_data->m_actions[actionIndex]->SetAnimationTimeDispatcher(GetAnimationTimeDispatcher());
                }
            }
            for (SizeType conditionIndex = 0; conditionIndex < m_data->m_conditions.Size(); ++conditionIndex) {
                if (0 != m_data->m_conditions[conditionIndex]) {
                    m_data->m_conditions[conditionIndex]->SetAnimationTimeDispatcher(GetAnimationTimeDispatcher());
                }
            }
        }
    }

    void StateMachineBehavior::SetStateMachine(Internal::StateMachineBehaviorData* data)
    {
        if (m_data != data) {
            if (m_data != 0) {
                m_data->Dispose();
            }
            m_data = data;
        }
    }
}
