//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_BEHAVIOR_EVENT_DISPATCH_STRATEGY_H
#define CANDERA_BEHAVIOR_EVENT_DISPATCH_STRATEGY_H

#include <FeatStd/Event/Event.h>
#include <Candera/EngineBase/Common/AbstractNodePointer.h>
#include <CanderaBehavior/BehaviorBase/EventDispatchResult.h>

namespace Candera {
/** @addtogroup BehaviorBase
  * @{
  */

    class Behavior;

    class EventDispatchStrategy
    {
    public:
        class Condition
        {
        public:
            /**
             * Accept method.
             * @param behavior
             * @param node
             * @param event
             * @param dispatchResult
             * @return True if successful, false otherwise.
             */
            virtual bool Accept(const Behavior* /*behavior*/, const AbstractNodePointer& /*node*/, const FeatStd::Event& /*event*/, const EventDispatchResult& /*dispatchResult*/) const
            {
                return true;
            }

        protected:
            /**
             * EventDispatchStrategy::Condition default constructor
             */
            Condition()
            {
            }

            /**
             * EventDispatchStrategy::Condition destructor
             */
            virtual ~Condition()
            {
            }

        private:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(Condition);
        };

        class ExcludeBehaviorInstanceCondition : public Condition
        {
        public:
            /**
             * ExcludeBehaviorInstanceCondition::Condition constructor
             * @param behavior
             */
            ExcludeBehaviorInstanceCondition(const Behavior* behavior) :
                m_behavior(behavior)
            {
            }

            /**
             * Accept method.
             * @param behavior
             * @param node
             * @param event
             * @param dispatchResult
             * @return True if successful, false otherwise.
             */
            virtual bool Accept(const Behavior* behavior, const AbstractNodePointer& /*node*/, const FeatStd::Event& /*event*/, const EventDispatchResult& /*dispatchResult*/) const
            {
                return (m_behavior != behavior);
            }

        private:
            const Behavior* m_behavior;
        };

        template<typename T>
        class IsOfTypeCondition : public Condition
        {
        public:
            /**
             * IsOfTypeCondition default constructor.
             */
            IsOfTypeCondition()
            {
            }

            /**
             * Accept method.
             * @param behavior
             * @param node
             * @param event
             * @param dispatchResult
             * @return True if successful, false otherwise.
             */
            virtual bool Accept(const Behavior* behavior, const AbstractNodePointer& /*node*/, const FeatStd::Event& /*event*/, const EventDispatchResult& /*dispatchResult*/) const
            {
                return 0 != Dynamic_Cast<const T*>(behavior);
            }
        };

        class ExcludeNodeCondition : public Condition
        {
        public:
            /**
             * ExcludeNodeCondition constructor.
             * @param nodePointer
             */
            ExcludeNodeCondition(const AbstractNodePointer nodePointer) :
                m_nodePointer(nodePointer)
            {
            }

            /**
             * Accept method.
             * @param behavior
             * @param node
             * @param event
             * @param dispatchResult
             * @return True if successful, false otherwise.
             */
            virtual bool Accept(const Behavior* /*behavior*/, const AbstractNodePointer& node, const FeatStd::Event& /*event*/, const EventDispatchResult& /*dispatchResult*/) const
            {
                return (m_nodePointer != node);
            }

        private:
            const AbstractNodePointer m_nodePointer;
        };

        class IsNot : public Condition
        {
        public:
            /**
             * IsNot constructor.
             * @param condition 
             */
            IsNot(const Condition& condition) :
                m_condition(condition)
            {
            }

            /**
             * Accept method.
             * @param behavior
             * @param node
             * @param event
             * @param dispatchResult
             * @return True if successful, false otherwise.
             */
            virtual bool Accept(const Behavior* behavior, const AbstractNodePointer& node, const FeatStd::Event& event, const EventDispatchResult& dispatchResult) const
            {
                return !m_condition.Accept(behavior, node, event, dispatchResult);
            }

        private:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(IsNot);
            FEATSTD_MAKE_CLASS_STATIC(IsNot);
            const Condition& m_condition;
        };

        /**
         * Dispatch the event.
         * @param node
         * @param event
         * @param dispatchResult
         * @param condition
         */
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Virtual function does not override a base-class function.")
        virtual void DispatchEvent(const AbstractNodePointer& node, const FeatStd::Event& event, EventDispatchResult& dispatchResult, const Condition* condition = 0) = 0;

    protected:
            /**
             * Check if the EventDispatchStrategy is a direct event receiver.
             * @param behavior
             * @return True if it is a direct event receiver, false otherwise.
             */
        static bool IsDirectEventReceiver(const Behavior* behavior);

        /**
         * Event handling method implementation with mechanisms to route events to other behaviors.
         * @param behavior
         * @param event The \event to handle.
         * @param dispatchResult The result.
         */
        static void OnEvent(Behavior* behavior, const FeatStd::Event& event, EventDispatchResult& dispatchResult);

        /**
         * Reset the flag for stopping dispatching immediately to false.
         * @param dispatchResult The result.
         */
        static void ResetStopDispatchingImmediately(EventDispatchResult& dispatchResult);

        /**
         * Reset the flag for stopping dispatching to false.
         * @param dispatchResult The result.
         */
        static void ResetStopDirectDispatching(EventDispatchResult& dispatchResult);

        /**
         * Reset the flag for stopping after direct dispatching to false.
         * @param dispatchResult The result.
         */
        static void ResetStopAfterDirectDispatching(EventDispatchResult& dispatchResult);

        /**
         * Reset the flag for stopping dispatching for descendants to false.
         * @param dispatchResult The result.
         */
        static void ResetStopDispatchingForDescendants(EventDispatchResult& dispatchResult);

        /**
         * Reset the flag for stopping dispatching for siblings to false.
         * @param dispatchResult The result.
         */
        static void ResetStopDispatchingForSiblings(EventDispatchResult& dispatchResult);
    };

/** @} */
}
#endif //CANDERA_BEHAVIOR_EVENT_DISPATCH_STRATEGY_H

