//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_CONDITION_BEHAVIOR_H
#define CANDERA_CONDITION_BEHAVIOR_H

#include <CanderaBehavior/BehaviorBase/Behavior.h>
#include <CanderaAssetLoader/AssetLoaderBase/ArrayProperty.h>
#include <CanderaAssetLoader/AssetLoaderBase/ArrayDataType.h>

namespace Candera {
/** @addtogroup BehaviorBase
  * @{
  */

  /**
 * @brief The ConditionBehavior is referenced by an TriggerBehavior. 
 * ConditionBehavior and ActionBehavior are base classes. Any input event is processed by a TriggerBehavior, 
 * which evaluates a condition (ConditionBehavior). 
 * If the condition evaluates to true, then the TriggerBehavior invokes one or more ActionBehavior. 
 * A filter on event type can be achieved by a specialized ConditionBehavior class for such an event.
 **/
class EvaluateConditionEvent : public FeatStd::Event
{
public:
    FEATSTD_RTTI_DECLARATION();

    /**
     * EvaluateConditionEvent constructor.
     * @param triggerEvent The trigger \event of this EvaluateConditionEvent.
     * @param evaluator The evaluator of this EvaluateConditionEvent.
     */
    EvaluateConditionEvent(const FeatStd::Event& triggerEvent, const Behavior* evaluator) :
        m_triggerEvent(triggerEvent),
        m_evaluator(evaluator)
    {
    }

     /**
     * EvaluateConditionEvent destructor.
     */
     virtual ~EvaluateConditionEvent()
    {
    }

    /**
     * Retrieve the trigger \event of this EvaluateConditionEvent.
     * @return The trigger \event of this EvaluateConditionEvent.
     */
    const FeatStd::Event& GetTriggerEvent() const {
        return m_triggerEvent;
    }

    /**
     * Retrieve the evaluator of this EvaluateConditionEvent.
     * @return The evaluator of this EvaluateConditionEvent.
     */
    const Behavior* GetEvaluator() const
    {
        return m_evaluator;
    }

private:
    FEATSTD_MAKE_CLASS_STATIC(EvaluateConditionEvent);
    FEATSTD_MAKE_CLASS_UNCOPYABLE(EvaluateConditionEvent);

    const FeatStd::Event& m_triggerEvent;
    const Behavior* m_evaluator;
};

class ConditionEvaluationResult : public EventDispatchResult
{
public:
    FEATSTD_RTTI_DECLARATION();
    /**
     * ConditionEvaluationResult default constructor.
     */
    ConditionEvaluationResult() :
        m_match(false)
    {
    }

    /**
     * ConditionEvaluationResult destructor.
     */
    virtual ~ConditionEvaluationResult()
    {
    }

    /**
     * Sets the condition's result to match, which means, the condition evaluates to true.
     */
    void Match()
    {
        m_match = true;
    }

    /**
     * Checks if the condition's result matches.
     * @return true if it matches and false otherwise.
     */
    bool IsMatch() const
    {
        return m_match;
    }

private:
    bool m_match;
};

class ConditionBehavior : public ::Candera::Behavior
{
public:
    /**
     * ConditionBehavior default constructor.
     */
    ConditionBehavior()
    {
    }

    /**
     * ConditionBehavior destructor.
     */
    virtual ~ConditionBehavior()
    {
    }

protected:
    /**
    * Check if this Behavior is a direct event receiver.
    * @return True if this Behavior is a direct event receiver.
    */
    virtual bool IsDirectEventReceiver() const
    {
        return false;
    }

private:
    /// @cond Doxygen ignore - start
    // @WidgetBaseClass
    CdaBehaviorMixedDef(ConditionBehavior, ::Candera::Behavior)
        CdaDescription("A condition is used for logical evaluations within other behaviors like the trigger or the state machine." \
                       " In case of a positive evaluation that behavior will react with a certain operation like the invokation of one or several actions.")
        CdaReadableName("Condition")
        CdaCategory("Logic")
        CdaProperties()
        CdaPropertiesEnd()
    CdaBehaviorDefEnd()
    /// @endcond Doxygen ignore - end
};
typedef ArrayProperty<ConditionBehavior*> ConditionBehaviorArrayProperty;

/** @} */
} // namespace Candera

CGI_BEHAVIOR_FORWARD_AssetLoaderDataTypeDef(::Candera::ConditionBehavior)

#endif
