//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_BEHAVIORMETAINFO_H
#define CANDERA_BEHAVIORMETAINFO_H

#include <CanderaWidget/WidgetBase/WidgetBase.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLoaderBaseDataTypes.h>
#include <FeatStd/Util/StaticObject.h>

/// @addtogroup BehaviorBase
/// @{

/**
*  Opens a behavior definition section
*     @param SpecificBehavior type of the Behavior to define meta information for
*     @param BaseBehavior behavior base class that widgetType is derived from
*/
#define CdaBehaviorDef(SpecificBehavior, BaseBehavior) \
    template <typename TOut, typename TIn> friend class ::FeatStd::Internal::DynamicCastDispatcher; \
    CGI_WIDGET_RTTI_DECLARATION(SpecificBehavior, BaseBehavior); \
    CdaWidgetDef(SpecificBehavior, BaseBehavior)
/**
*  Opens a behavior definition section for mixed behaviors (2D and 3D)
*     @param SpecificBehavior type of the Behavior to define meta information for
*     @param BaseBehavior behavior base class that widgetType is derived from
*/
#define CdaBehaviorMixedDef(SpecificBehavior, BaseBehavior) \
    template <typename TOut, typename TIn> friend class ::FeatStd::Internal::DynamicCastDispatcher; \
    CGI_WIDGET_RTTI_DECLARATION(SpecificBehavior, BaseBehavior); \
    _CdaWidgetDefInternal(SpecificBehavior, BaseBehavior, WidgetMixed, FEATSTD_NEW(CurrentType)())
/**
*  Opens a behavior definition section for 2D behaviors
*     @param SpecificBehavior type of the Behavior to define meta information for
*     @param BaseBehavior behavior base class that widgetType is derived from
*/
#define CdaBehavior2DDef(SpecificBehavior, BaseBehavior) \
    template <typename TOut, typename TIn> friend class ::FeatStd::Internal::DynamicCastDispatcher; \
    CGI_WIDGET_RTTI_DECLARATION(SpecificBehavior, BaseBehavior); \
    CdaWidget2DDef(SpecificBehavior, BaseBehavior)
/**
*  Closes a behavior definition section
*/
#define CdaBehaviorDefEnd() \
    CdaWidgetDefEnd()
/**
*  Opens a behavior class definition
*     @param SpecificBehavior type of the Behavior to define meta information for
*     @param BaseBehavior behavior base class that widgetType is derived from
*/
#define CGI_BEHAVIOR_RTTI_DEFINITION( CLASS ) \
    CGI_WIDGET_RTTI_DEFINITION(CLASS)
/** @} */
#ifdef COURIER_ENHANCED_ENABLED
#define CGI_BEHAVIOR_MESSAGE_HANDLER() \
        virtual bool OnMessage(const ::Courier::Message& message);
#define CGI_BEHAVIOR_IS_TOUCHABLE() \
        virtual bool IsTouchable() const { return true; }
#define CGI_BEHAVIOR_PROPERTY_MODIFIED_METHOD()
#define CGI_BEHAVIOR_WAKEUP_METHOD()
#define CGI_BEHAVIOR_INVALIDATE_METHOD()
#define CGI_BEHAVIOR_BINDABLE_PROPERTY()
#define CGI_BEHAVIOR_BINDABLE_GETTER()
#define CGI_BEHAVIOR_BINDABLE_SETTER()
#define CGI_BEHAVIOR_BINDABLE_MEMBER()
#else
/// @addtogroup BehaviorBase
/// @{
#ifndef CGI_BEHAVIOR_MESSAGE_HANDLER
#define CGI_BEHAVIOR_MESSAGE_HANDLER()
#endif
#ifndef CGI_BEHAVIOR_IS_TOUCHABLE
#define CGI_BEHAVIOR_IS_TOUCHABLE()
#endif
#ifndef CGI_BEHAVIOR_PROPERTY_MODIFIED_METHOD
#define CGI_BEHAVIOR_PROPERTY_MODIFIED_METHOD() \
    void PropertyModified(const FeatStd::Char*) const {}
#endif
#ifndef CGI_BEHAVIOR_WAKEUP_METHOD
#define CGI_BEHAVIOR_WAKEUP_METHOD() \
    void WakeUpAllRenderComponents() const {}
#endif
#ifndef CGI_BEHAVIOR_INVALIDATE_METHOD
#define CGI_BEHAVIOR_INVALIDATE_METHOD() \
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1735, CANDERA_LINT_REASON_DEFAULT_PARAMETER_REQUIRED) \
    virtual void Invalidate(const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>()) { \
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(196, CANDERA_LINT_REASON_PARAMETERRELEVANT) \
        FEATSTD_UNUSED(dirtyArea); \
    }
#endif
#ifndef CGI_BEHAVIOR_BINDABLE_PROPERTY
/**
*  Opens a bindable property definition section
*/
#define CGI_BEHAVIOR_BINDABLE_PROPERTY() \
    CdaProperty( BindableProperty , const FeatStd::Char*, GetBindableProperty, SetBindableProperty) \
    CdaDescription("") \
    CdaPropertyEnd()
#endif
#ifndef CdaBindableProperty
/**
*  Opens a  bindable property definition section
*     @param name name of the property (e.g. Position)
*     @param type type of the property (e.g. Point3D)
*     @param getter getter function of the behavior to get property value (e.g. GetPosition)
*     @param setter setter function of the behavior to set property value (e.g. SetPosition)
*/
#define CdaBindableProperty(name, type, getter, setter) \
        CdaProperty(name, type, getter, setter)
#endif
#ifndef CdaBindablePropertyEnd
/**
*  Closes a bindable property definition section
*/
#define CdaBindablePropertyEnd() \
    CdaPropertyEnd()
#endif
#ifndef CGI_BEHAVIOR_BINDABLE_GETTER
#define CGI_BEHAVIOR_BINDABLE_GETTER() \
    void SetBindableProperty(const FeatStd::Char*) const { }
#endif
#ifndef CGI_BEHAVIOR_BINDABLE_SETTER
#define CGI_BEHAVIOR_BINDABLE_SETTER() \
    const FeatStd::Char* GetBindableProperty() const { return 0; }
#endif
#ifndef CGI_BEHAVIOR_BINDABLE_MEMBER
#define CGI_BEHAVIOR_BINDABLE_MEMBER()
#endif
/** @} */
#endif // !1

/// @addtogroup BehaviorBase
/// @{
#ifdef CANDERA_META_DESCRIPTION

/**
* Get Editor of parameter type
* @param type Type to get the editor from.
* @param scPropertyEditor
*/
 #define CGI_BEHAVIOR_AssetLoaderDataTypeGetEditorDef(type, scPropertyEditor) \
     static const Candera::Char* GetEditor() { \
         static std::string s_editor = std::string("builtin://BehaviorEditor?Base=") + std::string(::Candera::Internal::MetaInfoExposer::GetName< type >()); \
         return s_editor.c_str(); \
     }
#else
/**
* Get Editor of parameter type
* @param type Type to get the editor from.
* @param scPropertyEditor
*/
#define CGI_BEHAVIOR_AssetLoaderDataTypeGetEditorDef(type, scPropertyEditor)
#endif

/**
 * Set Data Type of AssetLoader
 * @param type Type which gets an Asset Loader.
 */
#define CGI_BEHAVIOR_FORWARD_AssetLoaderDataTypeDef(type) \
    namespace Candera {\
        namespace MetaInfo { \
            template < > struct DataType< type* > { \
                static inline bool ConvertToString( type * const& ptr, Char *buf, UInt bufLen) { \
                    WidgetBase* widgetBase = ptr; \
                    typedef ::Candera::WidgetBase CanderaWidgetBase; \
                    return ::Candera::MetaInfo::Internal::DataType<CanderaWidgetBase*>::ConvertToString(widgetBase, buf, bufLen); \
                } \
                static bool ConvertFromString( type * &destination, const Char *buf, AssetProvider* provider = 0) { \
                    WidgetBase* widgetBase = 0; \
                    typedef ::Candera::WidgetBase CanderaWidgetBase; \
                    if (::Candera::MetaInfo::Internal::DataType<CanderaWidgetBase*>::ConvertFromString(widgetBase, buf, provider)) { \
                        destination = Dynamic_Cast< type *>(widgetBase); \
                        if (0 != destination) { \
                            return true; \
                        } \
                        if (0 != widgetBase) { \
                            widgetBase->Dispose(); \
                        } \
                    } \
                    return false; \
                } \
                CGI_BEHAVIOR_AssetLoaderDataTypeGetEditorDef(type, "builtin://WidgetEditor") \
            }; \
        } \
    }
/** @} */
#ifdef CANDERA_META_DESCRIPTION

namespace Candera {
    namespace Internal {
        class EventMetaInfoBase : public MetaInfo::MetaInfoBase
        {
        public:
            /**
             * Retrieves the events.
             * @return The events.
             */
            static const FeatStd::Internal::Vector<const EventMetaInfoBase*>& GetEvents()
            {
                if (SortEvents()) {
                    Comperator comperator;
                    Events().Sort(comperator);
                    SortEvents() = false;
                }
                return Events();
            }

           /**
             *  Constructs MetaInfoBase object with the given name
             *  @param typeName name of the object described by MetaInfo
             */
            EventMetaInfoBase(const Char *typeName) :
                MetaInfoBase(typeName),
                m_hash(Candera::MetaInfo::Internal::Fnv1aHash::Hash(typeName))
            {
                Events().Add(this);
                SortEvents() = true;
            }

            /**
             * EventMetaInfoBase destructor
             */
            virtual ~EventMetaInfoBase()
            {
                Events().Clear();
                // This only happens when the process is terminated by exit.
                // Hence, a simple Clear is better than removing each meta info on its own.
                // for (FeatStd::SizeType i = 0; i < Events(); ++i) {
                //     if (Events()[i] == this) {
                //         Events()[i].Remove(i);
                //         break;
                //     }
                // }
            }

            /**
             * Retrieves the sort number of the events.
             * @return The sort number.
             */
            virtual FeatStd::Int32 GetSortOrder() const
            {
                return FeatStd::Internal::Limits<FeatStd::Int32>::Max();
            }

            /**
             * Retrieve hash value associated with MetaInfo.
             * @return hash value
             */
            virtual FeatStd::UInt32 GetHash() const
            {
                return m_hash;
            }

        private:
            struct Comperator
            {
                bool operator()(const EventMetaInfoBase* left, const EventMetaInfoBase* right)
                {
                    if (left == right) {
                        return true;
                    }
                    if (0 == left) {
                        return false;
                    }
                    if (0 == right) {
                        return false;
                    }
                    return left->GetSortOrder() < right->GetSortOrder();
                }
            };

            static bool& SortEvents()
            {
                FEATSTD_UNSYNCED_STATIC_OBJECT(bool, s_sortEvents, false);
                return s_sortEvents;
            }

            static FeatStd::Internal::Vector<const EventMetaInfoBase*>& Events()
            {
                FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Internal::Vector<const EventMetaInfoBase*>, s_events);
                return s_events;
            }

            FEATSTD_MAKE_CLASS_STATIC(EventMetaInfoBase);
            FEATSTD_MAKE_CLASS_UNCOPYABLE(EventMetaInfoBase);
            FeatStd::UInt32 m_hash;
        };

        template<typename EventType> struct EventMetaInfoProvider
        {
            static const EventMetaInfoBase& GetMetaInfo();
        };

        class EventMetaInfoProxy : public MetaInfo::MetaInfoBase
        {
        public:
            typedef enum
            {
                Input, ///< Input type
                Output ///< Output type
            } Type;

            /**
             * EventMetaInfoProxy constructor.
             * @param usageDescription name of the object described by MetaInfo
             * @param type The type of the EventMetaInfoProxy.
             * @param target
             */
            EventMetaInfoProxy(const Char* usageDescription, Type type, const EventMetaInfoBase& target) :
                MetaInfo::MetaInfoBase(usageDescription), // the base name is not wasted. Instead it is reused to store the usageDescription.
                m_type(type),
                m_target(target)
            {
            }

            /**
             * EventMetaInfoProxy destructor.
             */
            virtual ~EventMetaInfoProxy()
            {
            }

            /**
             * Retrieves the EventMetaInfoProxy type.
             * @return The type.
             */
            Type GetType() const
            {
                return m_type;
            }

            /**
             * Retrieves the EventMetaInfoProxy target.
             * @return The target.
             */
            const EventMetaInfoBase& GetTarget() const
            {
                return m_target;
            }

            /**
             * Retrieves the EventMetaInfoProxy name.
             * @return The name.
             */
            virtual const Char* GetName() const override
            {
                return m_target.GetName();
            }

            /**
             * Retrieve hash value associated with MetaInfo.
             * @return The hash value.
             */
            virtual UInt32 GetHash() const override
            {
                return m_target.GetHash();
            }

#ifdef CANDERA_META_DESCRIPTION
            /**
             * Retrieve the usage description from the MetaInfoBase base class.
             * @return The usage description.
             */
            const Char* GetUsageDescription() const
            { // the base name is not wasted. Instead it is reused to store the usageDescription.
                return MetaInfo::MetaInfoBase::GetName();
            }

            /**
             * Retrieve the readable name of the target.
             * @return The target's readable name.
             */
            virtual const Char* GetReadableName() const override
            {
                return m_target.GetReadableName();
            }

            /**
             * Retrieve the description of the target.
             * @return The target's description.
             */
            virtual const Char *GetDescription() const override
            {
                return m_target.GetDescription();
            }

            /**
             * Retrieve the category of the target.
             * @return The target's category.
             */
            virtual const Char* GetCategory() const override
            {
                return m_target.GetCategory();
            }

            /**
             * Checks if the target is deprecated.
             * @return True if the target is deprecated, false otherwise.
             */
            virtual bool IsDeprecated() const override
            {
                return m_target.IsDeprecated();
            }
#endif

        private:
            FEATSTD_MAKE_CLASS_STATIC(EventMetaInfoProxy);
            FEATSTD_MAKE_CLASS_UNCOPYABLE(EventMetaInfoProxy);

            Type m_type;
            const EventMetaInfoBase& m_target;
        };

        class EventMetaInfoSetBase
        {
        public:
            /**
             * EventMetaInfoSetBase constructor
             * @param size
             * @param base
             */
            EventMetaInfoSetBase(FeatStd::SizeType size, const EventMetaInfoSetBase* base) :
                m_itemCount(size / sizeof(Candera::Internal::EventMetaInfoProxy)),
                m_base(base)
            {
            }

            /**
            *  Returns the meta information for the given name.
            *  @param name The name of the meta information.
            *  @return the meta information for the given name.
            */
            const EventMetaInfoProxy* LookupItem(const Char *name) const
            {
                FeatStd::UInt32 hash = Candera::MetaInfo::Internal::Fnv1aHash::Hash(name);
                const EventMetaInfoProxy* events = reinterpret_cast<const EventMetaInfoProxy*>(this + 1);
                for (FeatStd::SizeType i = 0; i < m_itemCount; ++i) {
                    const EventMetaInfoProxy* event = &events[i];
                    if ((hash == event->GetHash()) && (FeatStd::Internal::String::CompareStrings(event->GetName(), name) == 0)) {
                        return event;
                    }
                }
                return (0 != m_base) ? m_base->LookupItem(name) : 0;
            }

            /**
            *  Returns meta information at the given index
            *  @param index The index of the property.
            *  @return The pointer to the meta information.
            */
            const EventMetaInfoProxy* GetItem(FeatStd::SizeType index) const
            {
                FeatStd::SizeType totalItemCount = GetItemCount();
                if (index + 1 > totalItemCount) {
                    return 0;
                }
                if (index < totalItemCount - m_itemCount) {
                    return (0 != m_base) ? m_base->GetItem(index) : 0;
                }
                return &reinterpret_cast<const EventMetaInfoProxy*>(this + 1)[index - (totalItemCount - m_itemCount)];
            }

            /**
            *  Returns the number of meta information stored in this set.
            *  @return The number of meta information stored in this set.
            */
            FeatStd::SizeType GetItemCount() const
            {
                return m_itemCount + ((0 != m_base) ? m_base->GetItemCount() : 0);
            }

        private:
            FEATSTD_MAKE_CLASS_STATIC(EventMetaInfoSetBase);
            FEATSTD_MAKE_CLASS_UNCOPYABLE(EventMetaInfoSetBase);

            FeatStd::SizeType m_itemCount;
            const EventMetaInfoSetBase* m_base;
        };
    }
}

/// @addtogroup BehaviorBase
/// @{

/**
*  Opens an event set definition.
*/
#define CdaEvents() \
    class Events : public Candera::Internal::EventMetaInfoSetBase \
    { \
    public: \
        static const Candera::Internal::EventMetaInfoSetBase* GetInstance() \
        { \
            static Events s_events(Candera::Internal::Exposer<BaseMetaInfoProvider, true>::GetEvents<const Candera::Internal::EventMetaInfoSetBase*, Candera::MetaInfo::WidgetMetaInfo*>()); \
            return &s_events; \
        } \
    private: \
        FEATSTD_MAKE_CLASS_STATIC(Events); \
        FEATSTD_MAKE_CLASS_UNCOPYABLE(Events); \
        Events(const EventMetaInfoSetBase* base) : \
            Candera::Internal::EventMetaInfoSetBase(sizeof(Events), base) \
        { \
        }
/**
*  Opens an Event set definition.
*     @param EventCategory category of the event (Input, Output)
*     @param EventType type of the event
*     @param Description description of the event
*/
#define CdaEvent(EventCategory, EventType, Description) \
        FEATSTD_LINT_NEXT_EXPRESSION(1516, "Member declaration hides inherited member [MISRA C++ Rule 2-10-2] - false positive, the name is not required and collisions of the generated name are not relevant!") \
        class _CdaUniqueName(Event) : public Candera::Internal::EventMetaInfoProxy \
        { \
        public: \
            _CdaUniqueName(Event)() : \
                Candera::Internal::EventMetaInfoProxy(Description, EventCategory , Candera::Internal::EventMetaInfoProvider< EventType >::GetMetaInfo()) \
            { \
            } \
            FEATSTD_LINT_NEXT_EXPRESSION(1516, "Member declaration hides inherited member [MISRA C++ Rule 2-10-2] - false positive, the name is not required and collisions of the generated name are not relevant!") \
        } _CdaUniqueName(m_event);

/**
*  macro for an Output Event (Candera::Internal::EventMetaInfoProxy::Output)
*/
#define CdaOutputEvent Candera::Internal::EventMetaInfoProxy::Output
/**
*  macro for an Input Event (Candera::Internal::EventMetaInfoProxy::Input)
*/
#define CdaInputEvent Candera::Internal::EventMetaInfoProxy::Input
/**
*  Closes an Event set definition.
*/
#define CdaEventsEnd() \
    };
/** @} */
#else
/// @addtogroup BehaviorBase
/// @{
/**
*  Opens an event set definition.
*/
#define CdaEvents()
/**
*  Opens an Event set definition.
*     @param EventCategory category of the event (Input, Output)
*     @param EventType type of the event
*     @param Description description of the event
*/
#define CdaEvent(EventCategory, EventType, Description)
/**
*  macro for an Output Event (Candera::Internal::EventMetaInfoProxy::Output)
*/
#define CdaOutputEvent
/**
*  macro for an Input Event (Candera::Internal::EventMetaInfoProxy::Input)
*/
#define CdaInputEvent
/**
*  Closes an Event set definition.
*/
#define CdaEventsEnd()

/** @} */
#endif

#ifdef CANDERA_META_DESCRIPTION

#define CdaEventDef(EventType) \
namespace Candera { \
    namespace Internal { \
        static const EventMetaInfoBase& _CdaUniqueName(s_forcedInit) = EventMetaInfoProvider<EventType>::GetMetaInfo(); \
        template <> const EventMetaInfoBase& EventMetaInfoProvider<EventType>::GetMetaInfo() \
        { \
            class EventMetaInfo : public EventMetaInfoBase \
            { \
            public: \
                EventMetaInfo() : \
                    EventMetaInfoBase(EventType::GetTypeId()) \
                { \
                }

#define CdaEventSortOrder(order) \
                virtual FeatStd::Int32 GetSortOrder() const \
                { \
                    return (order); \
                }

#define CdaEventDefEnd() \
            }; \
            static EventMetaInfo s_metaInfo; \
            return s_metaInfo; \
        } \
    } \
}


#else
/// @addtogroup BehaviorBase
/// @{
/**
*  Opens an Event set definition.
*    @param Event which has to get defined
*/
#define CdaEventDef(Event)
/**
 * Set the sort number of the event (e.g. used in SC to order events for EventHandler)
 */
#define CdaEventSortOrder(order)
/**
 * Close an Event set definition.
 */
#define CdaEventDefEnd()
/** @} */
#endif
#ifdef CANDERA_META_DESCRIPTION
namespace Candera {
    namespace Internal {
        struct Legacy_3_5_0 {
            static bool& IsLegacyVersion() {
                static bool s_isLegacyVersion(false);
                return s_isLegacyVersion;
            }
        };
    }
}
#endif

#endif // CANDERA_BEHAVIORMETAINFO_H
