//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_BEHAVIOR_H
#define CANDERA_BEHAVIOR_H

#include <FeatStd/Event/Event.h>
#include <FeatStd/Event/EventListener.h>
#include <FeatStd/Util/Optional.h>
#include <FeatStd/Util/String.h>

#include <Candera/EngineBase/Common/AbstractNodePointer.h>
#include <CanderaBehavior/BehaviorBase/EventDispatchResult.h>


#ifdef CANDERA_2D_ENABLED
#include <CanderaWidget/Widget2D/Widget2DDataTypes.h>
#endif

#ifdef CANDERA_3D_ENABLED
#include <CanderaWidget/Widget3D/WidgetDataTypes.h>
#endif


#include <Candera/System/Container/Map.h>

#include <CanderaBehavior/BehaviorBase/BehaviorMetaInfo.h>

namespace Candera {
/** @addtogroup BehaviorBase
  * @{
  */

    class EventDispatchStrategy;

    class PreNodeChangeEvent : public FeatStd::Event
    {
    public:
        FEATSTD_TYPEDEF_BASE(FeatStd::Event);
        FEATSTD_RTTI_DECLARATION();

       /**
        * PreNodeChangeEvent default constructor.
        */
        PreNodeChangeEvent() { }
    private:
        FEATSTD_MAKE_CLASS_UNCOPYABLE(PreNodeChangeEvent);
    };

    class PostNodeChangeEvent : public FeatStd::Event
    {
    public:
        FEATSTD_TYPEDEF_BASE(FeatStd::Event);
        FEATSTD_RTTI_DECLARATION();

       /**
        * PostNodeChangeEvent default constructor.
        */
        PostNodeChangeEvent() { }
    private:
        FEATSTD_MAKE_CLASS_UNCOPYABLE(PostNodeChangeEvent);
    };

    class PreNodeDestroyEvent : public FeatStd::Event
    {
    public:
        FEATSTD_TYPEDEF_BASE(FeatStd::Event);
        FEATSTD_RTTI_DECLARATION();

       /**
        * PreNodeDestroyEvent constructor.
        * @param node The node used for this PreNodeDestroyEvent.
        */
        PreNodeDestroyEvent(const AbstractNodePointer node) : m_node(node) { }

       /**
        * Retrieves the Node for this PreNodeDestroyEvent.
        * @return The node used of this PreNodeDestroyEvent.
        */
        const AbstractNodePointer& GetNode() const { return m_node; }

    private:
        FEATSTD_MAKE_CLASS_STATIC(PreNodeDestroyEvent);
        FEATSTD_MAKE_CLASS_UNCOPYABLE(PreNodeDestroyEvent);
        AbstractNodePointer m_node;
    };

    /**
    * @brief Behaviors are small re-usable functional blocks that communicate via events.
    *       Behaviors are based on Widgets but extend their functionality.
    *       Compared to a Widget a Behavior is attached to a node by concept.
    *       This gives a clear hierarchy as well as access to the list of behaviors and makes event routing possible.
    *       Furthermore, with Behaviors, there is no early separation between 2D and 3D nodes.
    **/
    class Behavior : public WidgetBase
    {
    public:
        /**
         * Get the first behavior attached to a node.
         * @param node The Node, from which the first attached behavior is retrieved.
         * @return A pointer to the first attached behavior, or null, if no behaviors are attached.
         */
        static Behavior* GetFirstBehavior(const AbstractNodePointer& node);

       /**
        * Behavior default constructor.
        */
        Behavior();

       /**
        * Behavior destructor.
        */
        virtual ~Behavior();

        /**
         * Attach the behavior to a node.
         * @param node The node, where the behavior shall be attached to.
         * @return true, if successful.
         */
        bool AttachTo(const AbstractNodePointer& node);

        /**
         * Detach the behavior from the node it was previously attached to.
         * @return true, if successful.
         */
        bool Detach();

        /**
         * Dispatch the event directly.
         * @param event The event to dispatch.
         */
        void DispatchEventDirect(const FeatStd::Event& event) const;

        /**
         * Dispatch the event directly.
         * @param event The event to dispatch.
         * @param dispatchResult The result.
         */
        void DispatchEventDirect(const FeatStd::Event& event, EventDispatchResult& dispatchResult) const;

        /**
         * Dispatch the event.
         * @param eventDispatchStrategy The EventDispatchStrategy to dispatch the event.
         * @param event The event to dispatch.
         */
        void DispatchEvent(EventDispatchStrategy& eventDispatchStrategy, const FeatStd::Event& event) const;

        /**
         * Dispatch the event.
         * @param eventDispatchStrategy The EventDispatchStrategy to dispatch the event.
         * @param event The event to dispatch.
         * @param dispatchResult The result.
         */
        void DispatchEvent(EventDispatchStrategy& eventDispatchStrategy, const FeatStd::Event& event, EventDispatchResult& dispatchResult) const;

        /**
         * Dispatch the event locally.
         * @param event The event to dispatch.
         */
        void DispatchEventLocal(const FeatStd::Event& event);

        /**
         * Dispatch the event locally.
         * @param event The event to dispatch.
         * @param dispatchResult The result.
         */
        void DispatchEventLocal(const FeatStd::Event& event, EventDispatchResult& dispatchResult);

        /**
         * Retrieve the Behavior's node.
         * @return The Behavior's node.
         */
        AbstractNodePointer GetNode() const;

        /**
         * Set the Behavior's node.
         * @param node The node to set to the Behavior.
         */
        void SetNode(const AbstractNodePointer& node);

        /**
         * Retrieve the next Behavior.
         * @return The next Behavior.
         */
        Behavior* GetNextBehavior() const { return m_next; }

        CGI_BEHAVIOR_WAKEUP_METHOD();
        
        CGI_BEHAVIOR_INVALIDATE_METHOD()

    protected:
        enum PropertyChangeState
        {
            BeforeChange, ///< Used for the first call to OnChange, which happens before the change.
            AfterChange,  ///< Used for the second call to OnChange, which happens after the change.
            UpdateChange  ///< Used for the final call to OnChange for the update state (only if generated for the update phase).
        };

        /**
         * The OnChanged method will be called by the generated behavior base class code to notify the change of a property 
         * (only if the generated property is marked for notification).
         * The property id will also be generated in the base class. Use these property ids to check the modified property.
         * There is one call before the change, one after the change and one final call for the update state (only if generated for the update phase).
         * @param state The state of the property change.
         * @param propertyId The id of the changed property.
         */
        virtual void OnChanged(PropertyChangeState /*state*/, ::FeatStd::UInt32 /*propertyId*/) { }

        /**
         * Event handling method implementation with mechanisms to route events to other behaviors.
         * @param event The \event to handle.
         * @param dispatchResult The result.
         */
        virtual void OnEvent(const FeatStd::Event& event, EventDispatchResult& dispatchResult)
        {
            FEATSTD_UNUSED(event);
            FEATSTD_UNUSED(dispatchResult);
        }

        /**
         * Checks if the behavior is a direct \event receiver.
         * @return True if the behavior is a direct \event receiver.
         */
        virtual bool IsDirectEventReceiver() const
        {
            return true;
        }

        /**
         * Invalidates the layout of the behavior's node.
         */
        void InvalidateLayout() const;

        CGI_BEHAVIOR_PROPERTY_MODIFIED_METHOD()

    private:
        /// @cond Doxygen ignore - start
        CdaBehaviorMixedDef(Behavior, WidgetBase)
            CdaDescription("")
            CdaProperties()
            CGI_BEHAVIOR_BINDABLE_PROPERTY()
            CdaPropertiesEnd()
        CdaBehaviorDefEnd()
        /// @endcond Doxygen ignore - end

        CGI_BEHAVIOR_BINDABLE_GETTER()
        CGI_BEHAVIOR_BINDABLE_SETTER()
        CGI_BEHAVIOR_BINDABLE_MEMBER()

        friend class EventDispatchStrategy;

        //TODO: add notify for delete of node. Node and Node2D have to provide the EventSource interface and send the PreNodeDestroyEvent event.
        // The BehaviorContainer has to 
        class BehaviorContainer : public FeatStd::EventListener
        {
        public:

            /**
             * BehaviorContainer default constructor.
             */
            BehaviorContainer();

            /**
             * BehaviorContainer constructor.
             * @param behaviorContainer
             */
            BehaviorContainer(const BehaviorContainer& behaviorContainer);

            ~BehaviorContainer();

            /**
             * Retrieves the first Behavior of the BehaviorContainer.
             * @return The first element in the BehaviorContainer.
             */
            Behavior* GetFirst() const { return m_head; }

            /**
             * Adds a Behavior to the BehaviorContainer.
             * @param behavior The Behavior to add to the BehaviorContainer.
             */
            bool Add(Behavior* behavior);

            /**
             * Removes a Behavior from the BehaviorContainer.
             * @param behavior The Behavior to remove from the BehaviorContainer.
             */
            bool Remove(Behavior* behavior);

            /**
             * Every Behavior has to be attached to a node in the scenegraph. This method retrieves the node from the BehaviorContainer.
             * @return The BehaviorContainer's node.
             */
            AbstractNodePointer GetNode() const { return m_node; }

            /**
             * Every Behavior has to be attached to a node in the scenegraph. This method sets the node to the BehaviorContainer.
             * @param node The node to be set to the BehaviorContainer.
             */
            void SetNode(const AbstractNodePointer& node);

        protected:
            /**
             * Event handling method implementation with mechanisms to route events to other behaviors.
             * @param event The \event to handle.
             * @return Proceed if processing the event should be proceeded, and stop otherwise.
            */
            virtual FeatStd::EventResult::Enum OnEvent(const FeatStd::Event &event);

        private:
            friend class Behavior;
            AbstractNodePointer m_node;
            Behavior* m_head;
            Behavior* m_tail;
        };

        typedef Internal::Map<CanderaObject*, Behavior::BehaviorContainer*> NodeBehaviorContainerMap;
        static NodeBehaviorContainerMap& GetNodeBehaviorContainerMap();
        static BehaviorContainer* GetBehaviorContainer(AbstractNodePointer node, bool createIfRequired);

        BehaviorContainer* m_behaviorContainer;
        Behavior* m_next;
    };

#ifdef CANDERA_2D_ENABLED
    class Behavior2D
    {
    public:
        /**
         * Retrieves the node from the Behavior.
         * @param behavior
         * @return The Behavior's node.
         */
        static Node2D* GetNode(const Behavior& behavior);

        /**
         * Sets the node to the Behavior.
         * @param behavior
         * @param node The node to be set to the Behavior.
         */
        static void SetNode(Behavior& behavior, Node2D* node);
    };
#endif

#ifdef CANDERA_3D_ENABLED
    class Behavior3D
    {
    public:
        /**
         * Retrieves the node from the Behavior.
         * @param behavior
         * @return The Behavior's node.
         */
        static Node* GetNode(const Behavior& behavior);

        /**
         * Sets the node to the Behavior.
         * @param behavior
         * @param node The node to be set to the Behavior.
         */
        static void SetNode(Behavior& behavior, Node* node);
    };
#endif

}
/** @} */
#endif
