//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Behavior.h"

#include <CanderaBehavior/BehaviorBase/EventDispatchStrategy.h>
#include <CanderaBehavior/BehaviorBase/DirectEventDispatchStrategy.h>
#include <FeatStd/Util/StaticObject.h>

#if defined(CANDERA_2D_ENABLED)
    #include <Candera/Engine2D/Core/Scene2D.h>
#endif

namespace Candera {

    FEATSTD_RTTI_DEFINITION(PreNodeChangeEvent, FeatStd::Event)
    FEATSTD_RTTI_DEFINITION(PostNodeChangeEvent, FeatStd::Event)
    FEATSTD_RTTI_DEFINITION(PreNodeDestroyEvent, FeatStd::Event)
    CGI_BEHAVIOR_RTTI_DEFINITION(Behavior)

    
    Behavior::Behavior() :
        m_behaviorContainer(0),
        m_next(0)
    {
    }


    Behavior::~Behavior()
    {
        static_cast<void>(Detach());
    }

    Behavior* Behavior::GetFirstBehavior(const AbstractNodePointer& node)
    {
        BehaviorContainer* behaviorContainer = GetBehaviorContainer(node, false);
        return (0 != behaviorContainer) ? behaviorContainer->GetFirst() : 0;
    }

    bool Behavior::AttachTo(const AbstractNodePointer& node)
    {
        bool result = false;
        if (0 == m_behaviorContainer) {
            BehaviorContainer* behaviorContainer = GetBehaviorContainer(node, true);
            if ((0 != behaviorContainer) && (behaviorContainer->Add(this))) {
                m_behaviorContainer = behaviorContainer;
                result = true;
            }
        }
        else {
            // add fatal error log here: the behavior is already attached to a node. It has to be first detached.
        }
        PostNodeChangeEvent postNodeChangeEvent;
        DispatchEventLocal(postNodeChangeEvent);
        return result;
    }

    bool Behavior::Detach()
    {
        PreNodeChangeEvent preNodeChangeEvent;
        DispatchEventLocal(preNodeChangeEvent);
        if (0 == m_behaviorContainer) {
            return false;
        }
        bool result = m_behaviorContainer->Remove(this);
        if (0 == m_behaviorContainer->m_head) {
            static_cast<void>(GetNodeBehaviorContainerMap().Remove(m_behaviorContainer->m_node.ToCanderaObject()));
            FEATSTD_DELETE(m_behaviorContainer);
        }
        m_behaviorContainer = 0;
        return result;
    }

    void Behavior::DispatchEventDirect(const FeatStd::Event& event) const
    {
        EventDispatchResult dispatchResult;
        DirectEventDispatchStrategy directEventDispatchStrategy;
        DispatchEvent(directEventDispatchStrategy, event, dispatchResult);
    }

    void Behavior::DispatchEventDirect(const FeatStd::Event& event, EventDispatchResult& dispatchResult) const
    {
        DirectEventDispatchStrategy directEventDispatchStrategy;
        DispatchEvent(directEventDispatchStrategy, event, dispatchResult);
    }

    void Behavior::DispatchEvent(EventDispatchStrategy& eventDispatchStrategy, const FeatStd::Event& event) const
    {
        EventDispatchResult dispatchResult;
        eventDispatchStrategy.DispatchEvent(GetNode(), event, dispatchResult);
    }

    void Behavior::DispatchEvent(EventDispatchStrategy& eventDispatchStrategy, const FeatStd::Event& event, EventDispatchResult& dispatchResult) const
    {
        eventDispatchStrategy.DispatchEvent(GetNode(), event, dispatchResult);
    }

    void Behavior::DispatchEventLocal(const FeatStd::Event& event)
    {
        EventDispatchResult dispatchResult;
        OnEvent(event, dispatchResult);
    }

    void Behavior::DispatchEventLocal(const FeatStd::Event& event, EventDispatchResult& dispatchResult)
    {
        OnEvent(event, dispatchResult);
    }

    AbstractNodePointer Behavior::GetNode() const {
        return (0 != m_behaviorContainer) ? m_behaviorContainer->GetNode() : AbstractNodePointer();
    }

    void Behavior::SetNode(const AbstractNodePointer& node)
    {
        if (GetNode() != node) {
            static_cast<void>(Detach());
            static_cast<void>(AttachTo(node));
        }
    }


    Behavior::BehaviorContainer::BehaviorContainer() :
        m_head(0),
        m_tail(0)
    {
    }

    Behavior::BehaviorContainer::BehaviorContainer(const BehaviorContainer& behaviorContainer) :
        m_node(behaviorContainer.m_node),
        m_head(behaviorContainer.m_head),
        m_tail(behaviorContainer.m_tail)
    {
    }

    Behavior::BehaviorContainer::~BehaviorContainer()
    {
    }

    bool Behavior::BehaviorContainer::Add(Behavior* behavior)
    {
        if ((0 == behavior) || behavior->GetNode().IsValid()) {
            return false;
        }
        if (0 == m_head) {
            m_head = behavior;
        }
        if (0 != m_tail) {
            m_tail->m_next = behavior;
        }
        m_tail = behavior;
        return true;
    }


    bool Behavior::BehaviorContainer::Remove(Behavior* behavior)
    {
        if ((0 == behavior) || (0 == behavior->m_behaviorContainer)) {
            return false;
        }
        if (behavior == m_head) {
            m_head = m_head->m_next;
            if (0 == m_head) {
                m_tail = 0;
            }
        }
        Behavior* current = m_head;
        while (0 != current) {
            if (behavior == current->m_next) {
                current->m_next = behavior->m_next;
                behavior->m_next = 0;
                if (behavior == m_tail) {
                    m_tail = current;
                }
                return true;
            }
            current = current->m_next;
        }
        return false;
    }

    void Behavior::BehaviorContainer::SetNode(const AbstractNodePointer& node)
    {
        //TODO: reactivate the listener as soon as the Node and Node2D class implement the EventSource.
        //            if (m_node.IsValid()) {
        //                FeatStd::EventSource eventSource = m_node.GetEventSource();
        //                eventSource.Remove(this);
        //            }
        m_node = node;
        //            if (m_node.IsValid()) {
        //                FeatStd::EventSource eventSource = m_node.GetEventSource();
        //                eventSource.Remove(this);
        //            }
    }


    FeatStd::EventResult::Enum Behavior::BehaviorContainer::OnEvent(const FeatStd::Event &event)
    {
        const PreNodeDestroyEvent* preNodeDestroyEvent = Candera::Dynamic_Cast<const PreNodeDestroyEvent*>(&event);
        if ((0 != preNodeDestroyEvent) && (preNodeDestroyEvent->GetNode() == m_node)) {
            Behavior* current = m_head;
            m_head = 0;
            m_tail = 0;
            while (0 != current) {
                Behavior* next = current->m_next;
                current->m_next = 0;
                // TODO: as soon as the Behavior is no longer derived from WidgetBase or
                // the ownership of the behavior is moved from the asset provider to the node
                // the Dispose call has to be reactivated
                //current->Dispose();
                current = next;
            }
        }

        //TODO: Check event result.
        return FeatStd::EventResult::Proceed;
    }


    void Behavior::InvalidateLayout() const
    {
#if defined(CANDERA_2D_ENABLED) && defined(CANDERA_LAYOUT_ENABLED)
        AbstractNodePointer node = GetNode();
        if (node.IsValid()) {
            Layouter::InvalidateLayout(node);
        }
#endif
    }

    Behavior::NodeBehaviorContainerMap& Behavior::GetNodeBehaviorContainerMap()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(NodeBehaviorContainerMap, s_nodeBehaviorContainerMap);
        return s_nodeBehaviorContainerMap;
    }

    Behavior::BehaviorContainer* Behavior::GetBehaviorContainer(AbstractNodePointer node, bool createIfRequired)
    {
        if (node.IsValid()) {
            BehaviorContainer** nodeBehaviorMapEntry = GetNodeBehaviorContainerMap().Find(node.ToCanderaObject());
            BehaviorContainer* behaviorContainer = (0 != nodeBehaviorMapEntry) ? *nodeBehaviorMapEntry : 0;
            if ((0 == nodeBehaviorMapEntry) && createIfRequired) {
                behaviorContainer = FEATSTD_NEW(Behavior::BehaviorContainer);
                if (0 != behaviorContainer) {
                    if (!GetNodeBehaviorContainerMap().Insert(node.ToCanderaObject(), behaviorContainer)) {
                        FEATSTD_DELETE(behaviorContainer);
                        behaviorContainer = 0;
                    }
                }
            }
            if (0 != behaviorContainer) {
                behaviorContainer->m_node = node;
            }
            return behaviorContainer;
        }
        return 0;
    }

#ifdef CANDERA_2D_ENABLED
    Node2D* Behavior2D::GetNode(const Behavior& behavior)
    {
        return behavior.GetNode().ToNode2D();
    }

    void Behavior2D::SetNode(Behavior& behavior, Node2D* node)
    {
        behavior.SetNode(AbstractNodePointer(node));
    }
#endif

#ifdef CANDERA_3D_ENABLED
    Node* Behavior3D::GetNode(const Behavior& behavior)
    {
        return behavior.GetNode().ToNode();
    }

    void Behavior3D::SetNode(Behavior& behavior, Node* node)
    {
        behavior.SetNode(AbstractNodePointer(node));
    }
#endif
}
