//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_ACTION_BEHAVIOR_H
#define CANDERA_ACTION_BEHAVIOR_H

#include <CanderaBehavior/BehaviorBase/Behavior.h>
#include <CanderaAssetLoader/AssetLoaderBase/ArrayProperty.h>
#include <CanderaAssetLoader/AssetLoaderBase/ArrayDataType.h>

namespace Candera {
class ActionBehavior;
/** @addtogroup BehaviorBase
 *  @{
 */
typedef ArrayProperty<ActionBehavior*> ActionBehaviorArrayProperty;

/**
 * @brief Event to invoke an action.
 * An input event is processed by a TriggerBehavior, which evaluates a condition (Condition Behavior).
 * If the condition evaluates to true, then it invokes one or more ActionBehaviors using the InvokeActionEvent.
 */
class InvokeActionEvent : public FeatStd::Event
{
public:
    FEATSTD_RTTI_DECLARATION();

    /**
     * InvokeActionEvent constructor.
     * @param invoker The invoker of this InvokeActionEvent.
     * @param event The event of this InvokeActionEvent.
     */
    InvokeActionEvent(const Behavior* invoker, const Event* event) :
        m_invoker(invoker),
        m_event(event)
    {
    }

     /**
     * InvokeActionEvent destructor.
     */
   virtual ~InvokeActionEvent()
    {
    }

    /** 
     * Retrieves the invoker of this InvokeActionEvent.
     * @return The invoker of this InvokeActionEvent.
     */
    const Behavior* GetInvoker() const
    {
        return m_invoker;
    }

    /** 
     * Retrieves the Event of this InvokeActionEvent.
     * @return The event of this InvokeActionEvent.
     */
    const Event* GetEvent() const
    {
        return m_event;
    }

private:
    FEATSTD_MAKE_CLASS_STATIC(InvokeActionEvent);
    FEATSTD_MAKE_CLASS_UNCOPYABLE(InvokeActionEvent);

    const Behavior* m_invoker;
    const Event* m_event;
};

/**
 * @brief Result of InvokeActionEvent. The ActionBehavior may modify the result when reacting on the InvokeActionEvent.
 **/
class InvokeActionEventDispatchResult : public Candera::EventDispatchResult
{
public:
    FEATSTD_RTTI_DECLARATION();

    /**
     * InvokeActionEventDispatchResult default constructor.
     */
    InvokeActionEventDispatchResult()
    {
    }

    /**
     * InvokeActionEventDispatchResult destructor.
     */
    virtual ~InvokeActionEventDispatchResult()
    {
    }

    /**
    * Add an ActionBehavior to be invoked as postponed action.
    * @param action The action to be added.
    */
    void AddPostponedAction(ActionBehavior& action)
    {
        m_postponedActions.Add(&action);
    }

    /**
    * Add several ActionBehavior instances as an array to be invoked as postponed actions.
    * @param actions Array of ActionBehavior to be invoked postponed.
    */
    void AddPostponedActions(const ActionBehaviorArrayProperty& actions);

    /**
    * Get the ActionBehavior instances to be invoked as postponed actions.
    * @return An Array of ActionBehavior to be invoked postponed.
    */
    const ActionBehaviorArrayProperty& GetPostponedActions() const
    {
        return m_postponedActions;
    }

private:
    FEATSTD_MAKE_CLASS_UNCOPYABLE(InvokeActionEventDispatchResult);
    ActionBehaviorArrayProperty m_postponedActions;
};

/**
* @brief The Action Behavior is part of the Event Handling with Behaviors pattern.
*       It follows the ECA (Event-Condition-Action) design pattern.
*       @see https://en.wikipedia.org/wiki/Event_condition_action).
*       An input event is processed by a Trigger Behavior, which evaluates a condition (Condition Behavior).
*       If the condition evaluates to true, then it invokes one or more Action Behaviors.
*       The class ActionBehavior is a base class, derived behaviors are used in the described pattern.
**/
class ActionBehavior : public ::Candera::Behavior
{
public:
    /**
    * The Invocation of the ActionBehavior.
    * @param invoker
    * @param event
    * @param actions
    */
    static void Invoke(const Behavior* invoker, const FeatStd::Event* event, const ActionBehaviorArrayProperty& actions);

    /**
    * The Invocation of the ActionBehavior.
    * @param invoker
    * @param event
    * @param actions
    * @param invokeActionEventDispatchResult The result of this invoke method.
    */
    static void Invoke(const Behavior* invoker, const FeatStd::Event* event, const ActionBehaviorArrayProperty& actions, InvokeActionEventDispatchResult& invokeActionEventDispatchResult);

    /**
    * The Invocation of the ActionBehavior.
    * @param invoker
    * @param event
    * @param actions
    * @param begin
    * @param end
    */
    static void Invoke(const Behavior* invoker, const FeatStd::Event* event, const FeatStd::Internal::Vector< ActionBehavior* >& actions, FeatStd::SizeType begin, FeatStd::SizeType end);

    /**
     * ActionBehavior default constructor.
     */
    ActionBehavior()
    {
    }

    /**
     * ActionBehavior destructor.
     */
    virtual ~ActionBehavior()
    {
    }

protected:
    /**
    * Check if this Behavior is a direct event receiver.
    * @return True if this Behavior is a direct event receiver.
    */
    virtual bool IsDirectEventReceiver() const
    {
        return false;
    }

private:
    /// @cond Doxygen ignore - start
    // @WidgetBaseClass
    CdaBehaviorMixedDef(ActionBehavior, ::Candera::Behavior)
        CdaDescription("")
        CdaReadableName("Action")
        CdaCategory("Logic")
        CdaProperties()
        CdaPropertiesEnd()
    CdaBehaviorDefEnd()
   /// @endcond Doxygen ignore - end
};

/** @} */
} // namespace Candera

CGI_BEHAVIOR_FORWARD_AssetLoaderDataTypeDef(::Candera::ActionBehavior)

#endif
