//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if  !defined(CANDERA_SCENECONTEXTBASE_H)
#define CANDERA_SCENECONTEXTBASE_H

#include <Candera/Environment.h>

#include <Candera/System/Container/Vector.h>
#include <CanderaAssetLoader/AssetLoaderBase/ContentLoader.h>
#include <CanderaAssetLoader/AssetLoaderBase/ReloadEntry.h>
#include <Candera/System/Container/ForwardIterator.h>
#include <Candera/EngineBase/Animation/AnimationBlendedProperty.h>

#include <FeatStd/Util/NumericUtil.h>

FEATSTD_LINT_FILE(967, SceneContextBase.h, "false positive - file has valid include guards.")


namespace Candera {

    /** @addtogroup AssetLoaderBase
    *  @{
    */

    // Forward declarations
    class WidgetBase;
    class AssetProvider;
    namespace Internal {
        class SceneContextAnimationBlendedPropertyListAssetReader;
        template <typename T>
        class AssetBuilderBase;
        template <typename T>
        class AssetReaderBase;
    }
    namespace UnitTest {
        class ReloadListTest;
    }

    /**
    * @brief Base class for Scene2DContext and SceneContext to encapsulate main objects that define a scene context an application can operate on.
    */

    class SceneContextBase {
    private:
        typedef Candera::Internal::Vector<WidgetBase*> WidgetList;
    public:
        typedef ForwardIterator<WidgetList::ConstIterator> WidgetIterator;

        SceneContextBase();

        ~SceneContextBase();
        
        /**
         * Retrieve a ForwardIterator that points to the first widget.
         *
         * @return First widget ForwardIterator.
         */
        WidgetIterator GetWidgetIterator() const;

        /**
         *  Iteration helper: Returns the first widget in the list.
         *  @param widget [out] first widget.
         *  @param identifier [out] first widget identifier.
         *  @return false if reached the end of the list.
         */
        bool GetFirstWidget(WidgetBase*& widget, const Char*& identifier = dummyIdentifier) const;

        /**
         *  Iteration helper: Returns the next widget in the list.
         *  Assumes a previous call to GetFirstWidget().
         *  @param widget [out] next widget.
         *  @param identifier [out] next widget identifier.
         *  @return false if reached the end of the list.
         */
        bool GetNextWidget(WidgetBase*& widget, const Char*& identifier = dummyIdentifier) const;

        /**
        *  Return the widget with the specified identifier. This method is related to AddWidget. To
        *  retrieve an widget from the list, the same identifier shall be provided as the
        *  one used by AddWidget for the widget instance. If no identifier was provided, the name
        *  of the widget was considered instead.
        *  @param identifier Identifier specifying the widget.
        *  @return Pointer to the widget specified by the identifier.
        */
        WidgetBase* GetWidgetByIdentifer(const Char* identifier) const;

        /**
        *  Return the widget with the specified id.
        *  @param id If specifying the widget.
        *  @return Pointer to the widget specified by the id.
        */
        WidgetBase* GetWidget(Id id) const;

        /**
         *  Adds an widget to the context.
         *  @param widget Widget to be added.
         *  @return False if widget with same name already exists in the internal list, true otherwise.
         */
        bool AddWidget(WidgetBase* widget);

        /**
         *  Removes all the widgets from the scene context.
         */
        void ClearWidgetList() { m_widgetList.Clear(); }

        /**
         *  Returns the name of the theme that was used when loading this scene.
         *  @return name of the theme that was used when loading this scene.
         */
        const Char* GetLoadedThemeName() const;

        /**
         *  Returns the id of the theme that was used when loading this scene.
         *  @return id of the theme that was used when loading this scene.
         */
        Id GetLoadedThemeId() const { return m_loadedThemeId; }

        /**
         *  Returns the name of the default theme for this scene.
         *  @return Name of the default theme for this scene.
         */
        const Char* GetDefaultThemeName() const;

        /**
         *  Returns the id of the default theme for this scene.
         *  @return id of the default theme for this scene.
         */
        Id GetDefaultThemeId() const { return m_defaultThemeId; }

        /**
         *  Sets the upload policy that was used when loading this scene.
         *
         *  The UploadPolicy is set and used by the ContentLoader to specify the
         *   upload policy at SceneContext build time and to Unload the scene
         *   at release time if necessary.
         *
         *  If the application Uploads/Unloads the scene manually, it should update the
         *   SceneContext::UploadPolicy property before releasing the scene trough
         *   ContentLoader.
         *  @param uploadPolicy The upload policy that was used when loading the scene.
         */
        void SetUploadPolicy(ContentLoader::UploadPolicy uploadPolicy) { m_uploadPolicy = uploadPolicy; }

        /**
         *  Returns the upload policy that was used when loading this scene. @see SetUploadPolicy
         *  @return The upload policy that was used when loading the scene.
         */
        ContentLoader::UploadPolicy GetUploadPolicy() const { return m_uploadPolicy; }

        /**
         *  Retrieve number of attached animations.
         *
         *  An animation is attached to a SceneContext if at least one AnimationBlendedProperty attachment was added
         *   to the SceneContext along with the animation name as hostName.
         *
         *  @return number of attached animations.
         */
        UInt32 GetAttachedAnimationCount() const { return FeatStd::Internal::NumericConversion<UInt32>(m_animationBlendedPropertyLists.Size()); }

        /**
         *  Retrieve name of attached animation.
         *
         *  @param index Index of the attached animation (between 0 and GetAttachedAnimationCount()).
         *  @return name of attached animation.
         */
        const Char* GetAttachedAnimationName(UInt32 index);

        /**
         *  Retrieve asset id of attached animation.
         *
         *  @param index Index of the attached animation (between 0 and GetAttachedAnimationCount()).
         *  @return asset id of attached animation.
         */
        Id GetAttachedAnimationId(UInt32 index);

    protected:
        SceneContextBase(const SceneContextBase& other);

        void SetLoadedThemeId(Id themeId) { m_loadedThemeId = themeId; }
        void SetDefaultThemeId(Id themeId) { m_defaultThemeId = themeId; }
    private:
        friend class Candera::Internal::AssetBuilderBase<SceneContext*>;
        friend class Candera::Internal::AssetBuilderBase<Scene2DContext*>;
        friend class Candera::Internal::AssetReaderBase<SceneContextBase>;
        friend class Candera::Internal::SceneContextAnimationBlendedPropertyListAssetReader;
        friend class SceneContext;
        friend class Scene2DContext;
        friend class ContentLoader;
        friend class Candera::UnitTest::ReloadListTest;

        static const Char* dummyIdentifier;

        Id m_loadedThemeId;
        Id m_defaultThemeId;
        ContentLoader::UploadPolicy m_uploadPolicy;
        mutable WidgetIterator m_widgetListIterator;
        WidgetList m_widgetList;

        struct AnimationBlendedPropertyList {
            Id m_animationId;
            Candera::Internal::Vector<Animation::AnimationBlendedProperty::SharedPointer> m_properties;
        };
        Candera::Internal::Vector<AnimationBlendedPropertyList> m_animationBlendedPropertyLists;

        Candera::Internal::DependencyList m_reloadList;
        Candera::Internal::DependencyList::Iterator m_currentEntry;

        void AddAnimatedPropertyAttachment(Id hostId, MemoryManagement::SharedPointer<Animation::AnimationBlendedProperty> attachment);
        UInt32 GetAnimatedPropertyAttachmentCount(Id animationId);
        MemoryManagement::SharedPointer<Animation::AnimationBlendedProperty> GetAnimatedPropertyAttachment(Id assetId, UInt32 index);
    };

/** @} */ // end of AssetLoader3D

}

#endif //CANDERA_SCENECONTEXTBASE_H
