//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_OPTIONALDATATYPE_H)
#define CANDERA_OPTIONALDATATYPE_H

#include <CanderaAssetLoader/AssetLoaderBase/OptionalDataTypeBase.h>
#include <Candera/System/Mathematics/Optional.h>

#ifdef CANDERA_META_DESCRIPTION
#include <string>
#endif

namespace Candera {
    namespace MetaInfo {
        namespace Internal {
            
            template<typename T>
            class TypedOptionalDataTypeConverterToString: public OptionalDataTypeConverterToString {
                public:
                    TypedOptionalDataTypeConverterToString(const Optional<T>& optionalProperty) : m_optionalProperty(optionalProperty) {}

                    bool ConvertToString(Char *buf, UInt size) {
                        buf[0] = '0';
                        if (m_optionalProperty.IsSet())
                        {
                            buf[0] = '1';
                            buf[1] = ',';
                            return OptionalDataTypeConverterToString::ConvertToString(buf + 2, size - 2);
                        }
                        return true;
                    }
                protected:
                    virtual bool ConvertItemToString(Char *buf, UInt size);
                private:
                    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1725, CANDERA_LINT_REASON_ASSOCIATION)
                    const Optional<T>& m_optionalProperty;

                    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1704, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
                    FEATSTD_MAKE_CLASS_UNCOPYABLE(TypedOptionalDataTypeConverterToString<T>);
            };

            template<typename T>
            bool TypedOptionalDataTypeConverterToString<T>::ConvertItemToString(Char *buf, UInt size)
            {
                return Internal::DataType<T>::ConvertToString(*m_optionalProperty, buf, size);
            }

            template<typename T>
            class TypedOptionalDataTypeConverterFromString: public OptionalDataTypeConverterFromString {
                public:
                    TypedOptionalDataTypeConverterFromString(Optional<T>& optionalProperty) : m_optionalProperty(optionalProperty) {}
                protected:
                    virtual bool ConvertItemFromString(const Char* buf);
                private:
                    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1725, CANDERA_LINT_REASON_ASSOCIATION)
                        Optional<T>& m_optionalProperty;

                    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1704, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
                        FEATSTD_MAKE_CLASS_UNCOPYABLE(TypedOptionalDataTypeConverterFromString<T>);
            };

            template<typename T>
            bool TypedOptionalDataTypeConverterFromString<T>::ConvertItemFromString(const Char* buf)
            {
                T val;
                Float indicator;
                static_cast<void>(SerializationPlatform::ToBinary(buf, ",", &indicator, &indicator + 1));
                if (indicator == 1.0F)
                {
                    if (Candera::MetaInfo::Internal::DataType<T>::ConvertFromString(val, buf + 2))
                    {
                        m_optionalProperty = val;
                        return true;
                    }
                    return false;
                }
                return true;
            }

            /**
             * DataType specialization for optional properties.
             */
            template<typename T> struct DataType<Optional<T> > {
                /**
                 * Convert an optional property to string.
                 *
                 * @param val Optional to convert.
                 * @param buf Buffer to write the string representation.
                 * @param size Buffer length.
                 * @return true if conversion successful, false otherwise.
                 */
                template <typename P>
                static bool ConvertToString(const Optional<T>& val, Char *buf, UInt size) {
                    return Internal::TypedOptionalDataTypeConverterToString<T>(val).ConvertToString(buf, size);
                }
                static bool ConvertToString(const Optional<T>& val, Char *buf, UInt size) {
                    return Internal::TypedOptionalDataTypeConverterToString<T>(val).ConvertToString(buf, size);
                }

                /**
                 * Convert the string representation to an optional.
                 *
                 * @param val converted Optional.
                 * @param buf Buffer that contains the string representation.
                 * @return true if conversion successful, false otherwise.
                 */
                template <typename P>
                static bool ConvertFromString(Optional<T>& val, const Char *buf) {
                    return Internal::TypedOptionalDataTypeConverterFromString<T>(val).ConvertFromString(buf);
                }
                static bool ConvertFromString(Optional<T>& val, const Char *buf) {
                    return Internal::TypedOptionalDataTypeConverterFromString<T>(val).ConvertFromString(buf);
                }

                /**
                 * Get the PropertyEditor name as known by SceneComposer.
                 *
                 * @return string containing the SceneComposer property editor name.
                 */
                static const Char* GetEditor() {
#ifdef CANDERA_META_DESCRIPTION
                    static std::string editor = std::string("builtin://OptionalEditor?Type=(")
                        + Internal::DataType<T>::GetEditor() + ")";
                    return editor.c_str();
#else
                    return "builtin://OptionalEditor";
#endif
                }
            };
        }
    }
}
#endif // CANDERA_OPTIONALDATATYPE_H
