//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ITEMPATHINFO_H)
#define CANDERA_ITEMPATHINFO_H

#include <Candera/Environment.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLibEntities.h>

namespace Candera {
    /**
     * @brief Provides info on an item's path as used for referencing objects within the
     * Compact File Format asset lib.
     *
     * A path is a string that uniquely identifies an object within an hierarchy or context
     * (nodes, widgets, referenced templates, and other).
     *
     * It is constructed by successive pairs of :
     *     "/ObjectType:ObjectName"
     *     "/ObjectType:"            - to reference node Attachments (like Appearance, material and so on..)
     *     "/ObjectName"             - to reference nodes by name
     *
     * The number of pairs (structured as /ObjectType:ObjectName) defines the depth level
     * of the path (the depth level of the item looking for within a logical hierarchy)
     *
     * Some examples:
     *     - referencing a mesh: "/Scene:Tunnel/Group:car/Mesh:antenna" with depth = 3
     *                      or : "/Scene:Tunnel/car/antenna" with depth = 3
     *     - referencing an widget: "/Scene:Tunnel/Widget:speedometer" with depth = 2
     *
     * ItemPathInfo tokenizes the path and provides access to the token list (names and types)
     *
     *
     * A valid path contains GetDepth() number of pairs.
     * Pairs are numbered starting with 0.  0 <= pairIndex < GetDepth();
     */
    class ItemPathInfo
    {
    public:
        /**
         * Constructs ItemPathObject object with a given itemPath string to tokenize.
         * @param itemPath Item path string to parse and tokenize.
         */
        ItemPathInfo(const Char* itemPath);

        /**
         * Constructs ItemPathObject with a copy of other's path
         * @param other ItemPathInfo to copy from.
         */
        ItemPathInfo(const ItemPathInfo& other);

        /**
        * Constructs ItemPathObject with a copy of other's path, truncated to maxDepth elements.
        * @param other ItemPathInfo to copy from.
        * @param maxDepth If other's depth is > maxDepth, then the new constructed copy gets truncated to maxDepth.
        */
        ItemPathInfo(const ItemPathInfo& other, const UInt16 maxDepth);

        /**
         * @brief Copies the content of the specified item path into this
         * After the assignment, the two will be identical
         * If the other path is not valid, the result is undefined
         */
        ItemPathInfo& operator=(const ItemPathInfo& other);

        /**
         * @brief Returns true if this path equals the other
         */
        bool operator==(const ItemPathInfo& other);

        /**
         * @brief Returns true if this path is different than the other
         */
        bool operator!=(const ItemPathInfo& other);

        /**
         * Destructor
         */
        ~ItemPathInfo(void);

        /**
         * Validates the path
         * Should be used for checking before performing other operations
         * @return True if path is valid and has a depth > 0. Validness is checked when tokenizing in constructor.
         */
        bool IsPathValid() const  { return m_isPathValid && (m_depthLevel > 0); }

        /**
         * Returns the depth level of the path
         * (number of pairs of type /ObjectType:ObjectName that the path contains)
         * Returns 0 if the path is invalid
         * @return Depth level of the path.
         */
        UInt16 GetDepth() const { return m_depthLevel; }

        /**
         * Returns the type (as string) at the @see index position in the types list
         * @param index 0 <= index < GetDepth()
         * @return Type of the node as string at depth index. Returns 0 if the path is invalid.
         *
         */
        const Char* GetTypeStr(UInt16 index) const;

        /**
         * Returns the type (as string) at the @see index position in the types list
         * @param index 0 <= index < GetDepth()
         * @return Type of the node at depth index. Returns AssetObjectType::AotUnknown if the path is invalid or if the
         * type was not specified on the corresponding path node (for example in case of antenna object "/Scene:MyScene/Car/antenna")
         */
        AssetObjectType GetType(UInt16 index) const;

        /**
        * Returns the root type (as string)
        * @return Type of the node at index 0. Returns AssetObjectType::AotUnknown if the path is invalid or depth < 1
        */
        AssetObjectType GetRootType() const;

        /**
         * Returns the name at the @see index position in the names list
         * @param index 0 <= index < GetDepth()
         * @return Name of the node at depth index. Returns 0 if the path is invalid. Returns empty string if the name was not specified
         * in the corresponding path node (for example in case of the appearance, index = 3: "/Scene:MyScene/Car/antenna/Appearance:"
         */
        const Char* GetName(UInt16 index) const;

        /**
        * Returns the root name
        * @return Name of the node at depth 0. Returns 0 if the path is invalid;
        */
        const Char* GetRootName() const;

        /**
         * Returns the type of the item that the path references
         * @return Type of node referenced by this path (the last one referenced by GetDepth() - 1).
                   Returns AssetObjectType::AotUnknown if the path is invalid or the type name was not specified
                   for the corresponding path node (See GetTypeStr(index) )
         */
        AssetObjectType GetReferencedItemType() const;

        /**
         * Returns the name of the item that the path references
         * @return Name of the item referenced by this path (the last one referenced by GetDepth() - 1).
         Returns AssetObjectType::AotUnknown if the path is invalid; Returns empty string if the name was not specified (See GetName(index) )
         */
        const Char* GetReferencedItemName() const;

        /**
         * Converts the string representing a type within a path
         * to the corresponding AssetObjectType value.
         * @param  typeName String to convert to type.
         * @return Corresponding AssetObjectType. If typeName is not recognized then AssetObjectType::AotUnknown is returned.
         */
        static AssetObjectType ConvertToAssetObjectType(const Char* typeName);

        /**
         * Returns the original path that was passed at ItemPathInfo(const Char* itemPath) constructor.
         * @return Original path.
         */
        const Char* GetOriginalPath() const { return m_orignalPath; }

    protected:
        ItemPathInfo();

        const Char* GetPathCopy() const { return m_pathCopy; }
        UInt32 GetPathLength() const { return m_pathLength; }
        void CopyFrom(const ItemPathInfo& other, const UInt16 depth = 0);
        
    private:
        struct PathNode {
            AssetObjectType Type;
            UInt32 TypeOffset;
            UInt32 NameOffset;
        };

        struct PathNodeBoundaries
        {
            Int32 StartOffset;
            Int32 ColonOffset;
            Int32 EndOffset;
        };

        bool m_isPathValid;
        UInt16 m_depthLevel;
        UInt32 m_pathLength;
        Char* m_pathCopy;
        PathNode* m_nodes;
        const Char* m_orignalPath;

        // Copying not allowed for now:
        //ItemPathInfo(const ItemPathInfo& other);
        //ItemPathInfo& operator=(const ItemPathInfo& other);

        void Parse(const Char* itemPath);
        FEATSTD_LINT_CLEANUP_FUNCTION(Candera::ItemPathInfo::Release)
        void Release();
        static bool FindNextNode(const Char* path, PathNodeBoundaries& pathNodeBoundaries);
        static bool ExtractNode(const Char* path, const PathNodeBoundaries& pathNodeBoundaries, PathNode& pathNode);
        static void DelimitNode(Char* path, const PathNodeBoundaries& pathNodeBoundaries);
        static bool IsFirstCharValid(const Char* path);
        static void CountNodeDelimiters(const Char* path, UInt16& count, UInt32& pathLength);
        void CreateBuffers(const Char* path);
    };
    typedef ItemPathInfo AssetPath;
}

#endif // CANDERA_ITEMPATHINFO_H
