//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ItemPathInfo.h"
#include <Candera/Macros.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLoaderMemoryPool.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>
#include <CanderaPlatform/OS/StringPlatform.h>

namespace Candera {
    /**
     * @brief Creates a new buffer and copies <size> bytes into it
     */

    struct TypeNameToAssetObjectPair
    {
        const Char* name;
        AssetObjectType assetObjectType;
    };

    static const Int s_typeNameCount /* Array MUST BE SORTED! */ = 67;
    static const TypeNameToAssetObjectPair SortedTypeNamesToAssetObjectTypes[s_typeNameCount] =
    {
        { "Anchor", AotAnchor },
        { "Anchor2D", AotAnchor2D },
        { "Animation", AotAnimation },
        { "AnimationGroup", AotAnimationGroup },
        { "Appearance", AotAppearance },
        { "AppearanceCollection", AotAppearanceCollection },
        { "Behavior", AotBehavior },
        { "BehaviorCollection", AotBehaviorCollection },
        { "Billboard", AotBillboard },
        { "Bitmap", AotBitmap },
        { "Camera", AotCamera },
        { "Camera2D", AotCamera2D },
        { "CameraGroup", AotCameraGroup },
        { "Canvas", AotCanvas },
        { "CanvasGroup", AotCanvasGroup },
        { "CanvasSprite", AotCanvasSprite },
        { "CanvasText", AotCanvasText },
        { "ClearMode", AotClearMode },
        { "Composite", AotComposite },
        { "Composite2D", AotComposite2D },
        { "CompositeNode", AotCompositeNode },
        { "CompositeNode2D", AotCompositeNode2D },
        { "CompositeNodeAnimation", AotCompositeNodeAnimation },
        { "ContextResourcePool", AotContextResourcePool },
        { "DevicePackage", AotDevicePackage },
        { "Display", AotDisplay },
        { "Effect", AotEffect2D },
        { "EffectCollection", AotEffect2DCollection },
        { "Font", AotFont },
        { "Group", AotGroup },
        { "Group2D", AotGroup2D },
        { "Light", AotLight },
        { "LodNode", AotLodNode },
        { "Material", AotMaterial },
        { "Mesh", AotMesh },
        { "MorphingMesh", AotMorphingMesh },
        { "PlanarShadow", AotPlanarShadow },
        { "PointSprite", AotPointSprite },
        { "RawResource", AotRawResource },
        { "ReferencedTemplate", AotReferencedTemplate },
        { "ReflectionCamera", AotReflectionCamera },
        { "RenderMode", AotRenderMode },
        { "RenderNode2D", AotRenderNode },
        { "RenderTarget", AotGraphicDeviceUnit },
        { "Scene", AotScene },
        { "Scene2D", AotScene2D },
        { "Script", AotScript },
        { "Shader", AotShader },
        { "ShaderProgram", AotShaderProgram },
        { "SkyBox", AotSkyBoxMesh },
        { "StereoCamera", AotStereoCamera },
        { "TextStyle", AotTextStyle },
        { "Texture", AotTexture },
        { "TextureCollection", AotTextureCollection },
        { "Theme", AotTheme },
        { "ThemeEntry", AotThemeEntry },
        { "ThemeEntryCollection", AotThemeEntryCollection },
        { "Transition", AotTransition },
        { "TransitionCollection", AotTransitionCollection },
        { "TransitionConfiguration", AotTransitionConfiguration },
        { "TransitionHint", AotTransitionHint },
        { "TransitionRule", AotTransitionRule },
        { "TransitionRuleCollection", AotTransitionRuleCollection },
        { "UniformSetter", AotUniformSetter },
        { "VertexBuffer", AotVertexBuffer },
        { "Widget", AotWidget },
        { "WidgetCollection", AotWidgetCollection }
    };

    Int Search(const Char* typeName, Int low, Int high)
    {
        if (low > high) {
            return -1;
        }

        Int mid = (low + high) / 2;

        Int compareResult = StringPlatform::CompareStrings(typeName, SortedTypeNamesToAssetObjectTypes[mid].name);

        if (compareResult < 0) {
            return Search(typeName, low, mid - 1);
        }
        else if (compareResult > 0) {
            return Search(typeName, mid + 1, high);
        }
        else {
            return mid;
        }
    }

    bool AreAllTypeNamesSorted()
    {
        for (Int i = 1; i < s_typeNameCount; i++) {
            const Char* str1 = SortedTypeNamesToAssetObjectTypes[i - 1].name;
            const Char* str2 = SortedTypeNamesToAssetObjectTypes[i].name;
            if (!(StringPlatform::CompareStrings(str1, str2) <= 0)) {
                return false;
            }
        }
        return true;
    }

    static bool s_isTypeNamesOrderChecked = false;

    AssetObjectType TypeNameToAssetObjectType(const Char* typeName)
    {
        if (!s_isTypeNamesOrderChecked) {
            FEATSTD_DEBUG_ASSERT(AreAllTypeNamesSorted());
            s_isTypeNamesOrderChecked = true;
        }

        Int index = Search(typeName, 0, s_typeNameCount - 1);
        return index != -1 ? SortedTypeNamesToAssetObjectTypes[index].assetObjectType : AotUnknown;
    }

    Char* CloneBuffer(const Char* buffer, const UInt32 size)
    {
        Char* ptr = ASSETLOADER_TRANSIENT_NEW_ARRAY(Char, size);
        if (ptr != 0) {
            MemoryPlatform::Copy(ptr, buffer, size);
        }
        return ptr;
    }

    ItemPathInfo::~ItemPathInfo(void)
    {
        Release();
        // Following members are freed by Release()
        m_pathCopy = 0;
        m_nodes = 0;
    }

    ItemPathInfo::ItemPathInfo() :
        m_isPathValid(false),
        m_depthLevel(0),
        m_pathLength(0),
        m_pathCopy(0),
        m_nodes(0),
        m_orignalPath(0)
    {
    }

    ItemPathInfo::ItemPathInfo(const Char* itemPath) :
        m_isPathValid(false),
        m_depthLevel(0),
        m_pathLength(0),
        m_pathCopy(0),
        m_nodes(0),
        m_orignalPath(itemPath)
    {
        Parse(itemPath);
    }

    ItemPathInfo::ItemPathInfo(const ItemPathInfo& other) :
        m_isPathValid(false),
        m_depthLevel(0),
        m_pathLength(0),
        m_pathCopy(0),
        m_nodes(0),
        m_orignalPath(0)
    {
        CopyFrom(other, other.m_depthLevel);
    }

    ItemPathInfo::ItemPathInfo(const ItemPathInfo& other, const UInt16 maxDepth) :
        m_isPathValid(false),
        m_depthLevel(0),
        m_pathLength(0),
        m_pathCopy(0),
        m_nodes(0),
        m_orignalPath(0)
    {
        CopyFrom(other, maxDepth);
    }

    const Char* ItemPathInfo::GetName(UInt16 index) const
    {
        if ((m_nodes == 0) || (index >= m_depthLevel) || (!m_isPathValid)) {
            return 0;
        } else {
            return m_pathCopy + m_nodes[index].NameOffset;
        }
    }

    const Char* ItemPathInfo::GetRootName() const
    {
        return GetName(0);
    }

    const Char* ItemPathInfo::GetTypeStr(UInt16 index) const
    {
        if ((m_nodes == 0) || (index >= m_depthLevel) || (!m_isPathValid)) {
            return 0;
        } else {
            return m_pathCopy + m_nodes[index].TypeOffset;
        }
    }

    AssetObjectType ItemPathInfo::GetType(UInt16 index) const
    {
        if ((m_nodes == 0) || (index >= m_depthLevel) || (!m_isPathValid)) {
            return AotUnknown;
        } else {
            PathNode& pathNode = m_nodes[index];
            const Char* typeName = m_pathCopy + pathNode.TypeOffset;
            bool isTypeNameEmpty = (typeName[0] == '\0');
            if ((pathNode.Type == AotUnknown) && (!isTypeNameEmpty)) {
                pathNode.Type = ConvertToAssetObjectType(typeName);
            }
            return pathNode.Type;
        }
    }

    AssetObjectType ItemPathInfo::GetRootType() const
    {
        return GetType(0);
    }

    AssetObjectType ItemPathInfo::GetReferencedItemType() const
    {
        return GetType(m_depthLevel - 1);
    }

    const Char* ItemPathInfo::GetReferencedItemName() const
    {
        return GetName(m_depthLevel - 1);
    }

    AssetObjectType ItemPathInfo::ConvertToAssetObjectType(const Char* typeName)
    {
        return TypeNameToAssetObjectType(typeName);
    }

    void ItemPathInfo::Parse(const Char* itemPath)
    {
        Release();

        m_isPathValid = true;

        if ((itemPath == 0) || (!IsFirstCharValid(itemPath))) {
            m_isPathValid = false;
            return;
        }

        CountNodeDelimiters(itemPath, m_depthLevel, m_pathLength);
        if (m_depthLevel == 0) {
            m_isPathValid = false;
            return;
        }

        CreateBuffers(itemPath);
        if ((m_nodes == 0) || (m_pathCopy == 0)) {
            m_isPathValid = false;
            return;
        }

        UInt32 nodeCount = 0;
        UInt32 nextGroupOffset = 0;

        bool done = false;
        while (!done) {
            PathNodeBoundaries candidate;
            candidate.StartOffset = nextGroupOffset;
            if (!FindNextNode(m_pathCopy, candidate)) {
                done = true;
                break;
            }
            nextGroupOffset = candidate.EndOffset;
            PathNode& pathNode = m_nodes[nodeCount];
            if (!ExtractNode(m_pathCopy, candidate, pathNode)) {
                m_isPathValid = false;
                return;
            }
            DelimitNode(m_pathCopy, candidate);
            nodeCount++;
        }


        if (nodeCount != m_depthLevel) {
            m_isPathValid = false;
            return;
        }
        FEATSTD_LINT_SYMBOL(438, done, "value used in while loop condition")
    }

    void ItemPathInfo::CreateBuffers(const Char* path)
    {
        UInt32 nBytes = m_pathLength + UInt32(1);
        m_pathCopy = ASSETLOADER_TRANSIENT_NEW_ARRAY(Char, nBytes);
        if (m_pathCopy == 0) {
            return;
        }
        MemoryPlatform::Copy(m_pathCopy, path, nBytes);

        m_nodes = ASSETLOADER_TRANSIENT_NEW_ARRAY(PathNode, m_depthLevel);
    }

    void ItemPathInfo::CountNodeDelimiters(const Char* path, UInt16& count, UInt32& pathLength)
    {
        count = 0;
        pathLength = 0;
        while (path[pathLength] != '\0') {
            if ((path[pathLength] == '\\') || (path[pathLength] == '/')) {
                count++;
            }
            pathLength++;
        }
    }

    bool ItemPathInfo::IsFirstCharValid(const Char* path)
    {
        const Char& c = *path;
        // Fast check if path is not valid to speed up constructing ItemPathInfo objects on invalid paths
        if ((c != '\0') && (c != '/') && (c != '\\')) {
            return false;
        }

        if (StringPlatform::CompareStrings(path, "") == 0) {
            return false;
        }

        return true;
    }

    bool ItemPathInfo::FindNextNode(const Char* path, PathNodeBoundaries& pathNodeBoundaries)
    {
        if (path == 0) {
            return false;
        }
        pathNodeBoundaries.ColonOffset = -1;
        pathNodeBoundaries.EndOffset = -1;

        if (path[pathNodeBoundaries.StartOffset] == '\0') {
            return false;
        }
        Int32 i = pathNodeBoundaries.StartOffset + 1;
        for (; (path[i] != '/') && (path[i] != '\\') && (path[i] != '\0'); i++) {
            if (path[i] == ':') {
                pathNodeBoundaries.ColonOffset = i;
            }
        }
        pathNodeBoundaries.EndOffset = i;

        return true;
    }

    bool ItemPathInfo::ExtractNode(const Char* path, const PathNodeBoundaries& pathNodeBoundaries, PathNode& pathNode)
    {
        if (path == 0) {
            return false;
        }
        if ((pathNodeBoundaries.EndOffset == -1) || (pathNodeBoundaries.StartOffset == -1) || (pathNodeBoundaries.EndOffset <= pathNodeBoundaries.StartOffset)) {
            return false;
        }
        if (pathNodeBoundaries.EndOffset - pathNodeBoundaries.StartOffset == 1) {
            return false;
        }
        if (pathNodeBoundaries.ColonOffset != -1) {
            if (pathNodeBoundaries.ColonOffset <= pathNodeBoundaries.StartOffset) {
                return false;
            }
            if (pathNodeBoundaries.ColonOffset >= pathNodeBoundaries.EndOffset) {
                return false;
            }
            if (pathNodeBoundaries.ColonOffset - pathNodeBoundaries.StartOffset == 1) {
                return false;
            }
        }

        pathNode.Type = AotUnknown;
        pathNode.TypeOffset = static_cast<UInt32>(pathNodeBoundaries.ColonOffset != -1 ? pathNodeBoundaries.StartOffset + 1 : pathNodeBoundaries.StartOffset);
        pathNode.NameOffset = static_cast<UInt32>(pathNodeBoundaries.ColonOffset != -1 ? pathNodeBoundaries.ColonOffset + 1 : pathNodeBoundaries.StartOffset + 1);
        return true;
    }

    void ItemPathInfo::DelimitNode(Char* path, const PathNodeBoundaries& pathNodeBoundaries)
    {
        if (path == 0) {
            return;
        }
        path[pathNodeBoundaries.StartOffset] = '\0';
        if (pathNodeBoundaries.ColonOffset != -1) {
            path[pathNodeBoundaries.ColonOffset] = '\0';
        }
    }

    void ItemPathInfo::Release()
    {
        m_isPathValid = false;
        m_depthLevel = 0;
        if (m_nodes != 0) {
            ASSETLOADER_DELETE_ARRAY(m_nodes);
            m_nodes = 0;
        }
        if (m_pathCopy != 0) {
            ASSETLOADER_DELETE_ARRAY(m_pathCopy);
            m_pathCopy = 0;
        }
        m_pathLength = 0;
    }

    ItemPathInfo& ItemPathInfo::operator=(const ItemPathInfo& other)
    {
        CopyFrom(other, other.m_depthLevel);
        return *this;
    }

    bool ItemPathInfo::operator==(const ItemPathInfo& other)
    {
        if ((!m_isPathValid) || (!other.m_isPathValid)) {
            return false;
        }
        if (m_isPathValid != other.m_isPathValid) {
            return false;
        }
        if ((m_depthLevel == 0) || (other.m_depthLevel == 0)) {
            return false;
        }
        if (m_depthLevel != other.m_depthLevel) {
            return false;
        }

        if (0 != m_nodes) {
        // Iterate through nodes and compare
        for (Int i = 0; i < m_depthLevel; ++i) {
            PathNode& nThis = m_nodes[i];
            PathNode& nOther = other.m_nodes[i];
            // Compare type strings
            if (StringPlatform::CompareStrings(m_pathCopy + nThis.TypeOffset, other.m_pathCopy + nOther.TypeOffset) != 0) {
                return false;
            }
            // If types are different than AotUnknown it means we can compare them
            if ((nThis.Type != AotUnknown) && (nOther.Type != AotUnknown) && (nThis.Type != nOther.Type)) {
                return false;
            }
            // Compare names
            if (StringPlatform::CompareStrings(m_pathCopy + nThis.NameOffset, other.m_pathCopy + nOther.NameOffset) != 0) {
                return false;
            }
        }
        }
        return true;
    }

    bool ItemPathInfo::operator!=(const ItemPathInfo& other)
    {
        return !(*this == other);
    }

    void ItemPathInfo::CopyFrom(const ItemPathInfo& other, const UInt16 depth /*= 0*/)
    {
        Release();

        if ((!other.m_isPathValid) || (other.m_depthLevel == 0)) {
            return;
        }

        UInt16 effectiveDepth = depth;
        if (effectiveDepth > other.m_depthLevel) {
            m_depthLevel = other.m_depthLevel;
        }
        if (effectiveDepth == 0) {
            return;
        }

        m_nodes = ASSETLOADER_TRANSIENT_NEW_ARRAY(PathNode, effectiveDepth);

        m_pathCopy = CloneBuffer(other.m_pathCopy, other.m_pathLength + 1);
        if (m_nodes != 0) {
            for (UInt16 i = 0; i < effectiveDepth; i++) {
                m_nodes[i] = other.m_nodes[i];
            }
        }
        m_isPathValid = true;
        m_depthLevel = effectiveDepth;
        m_pathLength = other.m_pathLength;
        m_orignalPath = other.m_orignalPath;
    }
}
