//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ASSETBUILDER_H)
#define CANDERA_ASSETBUILDER_H

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/Generic/AssetReader.h>

namespace Candera {
    namespace Internal {
        /**
         * Template class used to dereference a generic pointer type.
         */
        template<typename T>
        class PointerType;

        /**
         * PointerType specialization for C style pointers.
         */
        template<typename T>
        class PointerType<T*> {
            public:
                typedef T Type;
                static Type& Dereference(T* object) { return *object; }
        };

        /**
         * PointerType specialization for shared pointers implemented with
         *  Candera::MemoryManagement::SharedPointer<T> class.
         */
        template<typename T>
        class PointerType<MemoryManagement::SharedPointer<T> > {
            public:
                typedef T Type;
                static Type& Dereference(MemoryManagement::SharedPointer<T> object) { return object.GetSharedInstance(); }
        };

        /**
         * Generic base class for AssetBuilder specializations.
         *
         * GenericAssetBuilderBase provides methods for creating, disposing and validating an object.
         */
        FEATSTD_LINT_SYMBOL(1577, Candera::Internal::AssetBuilder*<*>, "GenericAssetBuilderBase is specialized in each builder")
        FEATSTD_LINT_SYMBOL(1511, Candera::Internal::GenericAssetBuilderBase<*>::Dispose, "GenericAssetBuilderBase derived classes also provide Dispose functions")
        FEATSTD_LINT_SYMBOL(1511, Candera::Internal::GenericAssetBuilderBase<*>::IsValid, "GenericAssetBuilderBase derived classes also provide IsValid functions")
        template<typename T>
        class GenericAssetBuilderBase {
            protected:
                //LoaderContext is one of the specialized LoaderContext<> templates, based on the LoaderType proxy
                // defined with the AssetLoaderTypeInfo macro.
                typedef TypedLoaderContext<static_cast<AssetLoaderType>(AssetLoaderInfo<typename PointerType<T>::Type>::LoaderType)> LoaderContext;

            public:
                /**
                 * Dispose an object of type T.
                 *
                 * Properties that are created and set by the AssetReader, but are not owned by the object should
                 *  be disposed as well.
                 *
                 * @param object Object to be disposed.
                 */
                static void Dispose(T /*object*/) {}

                /**
                 * Validate an object of type T on the given context.
                 *
                 * For example a Scene might be loaded with a certain theme, but the theme
                 *  in the DefaultAssetProvider has changed meanwhile.
                 *
                 * @param object Object to be validated.
                 * @param context LoaderContext used to validate the object.
                 */
                static bool IsValid(T object, LoaderContext& context);

//             protected:
//                 /**
//                  * Create an object of type T.
//                  */
//                 static T Create(LoaderContext& context);
        };

        template<typename T>
        inline bool GenericAssetBuilderBase<T>::IsValid(T /*object*/, LoaderContext& /*context*/) { return true; }

        /**
         * Generic AssetBuilderBase that has to be specialized.
         *
         * Its specialization must hide the Create method in the base class and can hide
         *  the Dispose and IsValid ones with its own implementations.
         */
        template<typename T>
        class AssetBuilderBase: public GenericAssetBuilderBase<T> {};

        /**
         * Generic AssetBuilder.
         *
         * The generic AssetBuilder provides methods for creating and reading an object.
         *
         * The generic AssetBuilder can be specialized for complex objects in which case AssetBuilderBase
         *  can be ignored.
         */
        template<typename T>
        class AssetBuilder: public AssetBuilderBase<T> {
            typedef AssetBuilderBase<T> Base;

            protected:
                typedef AssetReader<typename PointerType<T>::Type> BaseReader;
                typedef TypedLoaderContext<static_cast<AssetLoaderType>(AssetLoaderInfo<typename PointerType<T>::Type>::LoaderType)> LoaderContext;

            public:
                /**
                 * Create the object and read its properties that have to be set before it is cached.
                 *
                 * @param context Loader context.
                 * @return object.
                 */
                static T CreateAndBuildFirstPass(LoaderContext& context);

                /**
                 * BuildSecondPass is the method that is used to finish the object initialization after it is cached by the
                 *  AssetProvider.
                 *
                 * @param object Object to be initialized.
                 * @param context Loader context.
                 * @return true if AssetReader specialization for type T returned true..
                 */
                static bool BuildSecondPass(T object, LoaderContext& context);

                static T CreateInOnePass(LoaderContext& context);
        };

        template<typename T>
        T Candera::Internal::AssetBuilder<T>::CreateAndBuildFirstPass(LoaderContext& context)
        {
            if (!context.IsValid()) {
                return T(0);
            }

            T result = Base::Create(context);
            if (result != T(0)) {
                if (!BaseReader::ReadFirstPass(PointerType<T>::Dereference(result), context)) {
                    Base::Dispose(result);
                    result = T(0);
                }
            }

            return result;
        }

        template<typename T>
        bool Candera::Internal::AssetBuilder<T>::BuildSecondPass(T object, LoaderContext& context)
        {
            return BaseReader::ReadSecondPass(PointerType<T>::Dereference(object), context);
        }

        template<typename T>
        T Candera::Internal::AssetBuilder<T>::CreateInOnePass(LoaderContext& context)
        {
            T result = CreateAndBuildFirstPass(context);
            if (result != T(0)) {
                if (!BuildSecondPass(result, context)) {
                    Base::Dispose(result);
                    result = T(0);
                }
            }

            return result;
        }
    }
}
#endif  // CANDERA_ASSETBUILDER_H
