//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DEFAULTASSETPROVIDER_H)
#define CANDERA_DEFAULTASSETPROVIDER_H

#include <Candera/Environment.h>
#include <Candera/System/Container/Vector.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetValidation.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDescriptor.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetGroup.h>

#ifdef CANDERA_TRANSITIONS_ENABLED
#include <CanderaTransitions/Rule.h>
#endif

CANDERA_UNIT_TEST_TESTCASE_DECLARATION(DefaultAssetProviderTest, InvalidAssetIdTest)
namespace Candera {

    class AssetConfig;
    class AssetRepository;
    class ContentLoader;
    class DefaultAssetProvider;
    class SceneContextBase;
    struct AssetData;
    class CanderaDefaultAssetProviderFriend;
    class StateMachineBehavior;

    namespace Internal {
        class AppearanceCollectionAssetBuilder;
        class BinaryTheme;
        class CompositeInstanceBuilder;
        class Composite2DInstanceBuilder;
        struct DefaultAssetProviderInternal;
        class ScriptComponentCollectionAssetReader;
        template <typename T, typename CachingStrategy, typename TBuilder> class TypedCffAssetProvider;
        template <typename T, typename CachingStrategy, typename TBuilder> class TypedAssetProvider;
        template<typename T> class AssetReaderBase;
        template<typename T> class AssetBuilderBase;

        class AssetProviderFunctions {
        public:
            static const Char* GetName(const DefaultAssetProvider* provider, UInt16 repositoryId,  Int32 offset);
            static Id GetIdByName(AssetProvider* provider, AssetLib assetLib, const Char* name);
        };
    }

    /** @addtogroup AssetLoaderBase
     *  @{
     */

    /**
     * @brief Implements objects handling mechanisms.
     * It implements the interface AssetProvider and holds an asset cache for storing
     * already-loaded objects
     *
     * It provides:
     *  - Scene and objects loading
     *  - Objects retrieval
     *
     * It's main purpose is to retrieve objects from the asset lib in a transparent way:
     * When an object is requested, it searches the cache. If is found in the cache, it
     * is retrieved from it, otherwise the asset provider loads the object from the asset
     * lib, places it in the cache and than returns it.
     */
    class DefaultAssetProvider: public AssetProvider
    {
        friend class Candera::Internal::AppearanceCollectionAssetBuilder;
        friend class Candera::Internal::AssetProviderFunctions;
        friend class Candera::Internal::AssetSet;
        friend class Candera::Internal::Composite2DInstanceBuilder;
        friend class Candera::Internal::CompositeInstanceBuilder;
        friend class Candera::Internal::ScriptComponentCollectionAssetReader;
        friend class ContentLoader;
        friend class SceneContextBase;
        template<typename T> friend class Candera::Internal::AssetReaderBase;
        template<typename T> friend class Candera::Internal::AssetBuilderBase;
        template <typename T, typename CachingStrategy, typename TBuilder> friend class Candera::Internal::TypedCffAssetProvider;
        template <typename T, typename CachingStrategy, typename TBuilder> friend class Candera::Internal::TypedAssetProvider;

        public:
        typedef AssetProvider Base;

        /**
         * @brief Constructor for DefaultAssetProvider.
         * The default repository is associated to it.
         */
        DefaultAssetProvider();

        /**
         * Destructor
         */
        virtual ~DefaultAssetProvider();

        /**
         * Load display properties from the asset.
         *
         * A display with the ID retrievable by GetDisplay(name) hast be be
         *  available (created using the DevicePackageInterface) for the
         *  properties to be loaded from the asset.
         *
         * @param id Id of the display in the asset.
         * @return true if display exists and properties are successfully set,
         *   false otherwise.
         */
        bool LoadDisplayProperties(Id id);

        /**
         * Load device package properties from the asset.
         *
         * @return true if  properties are successfully set, false otherwise.
         */
        bool LoadDevicePackageProperties();

        /**
         * Returns the default asset manager instance
         * @return The default asset manager instance.
         */
        static DefaultAssetProvider& GetInstance();

        /**
         * Initializes the asset manager
         * @param assetRepository Asset repository containing data to load.
         * @param validationFlags Flags defining various asset validation levels. See AssetSet::Initialize.
         * @return False if any needed member for initializing assetBuilders,
         *         including themselves, is 0 or couldn't be initialized. True otherwise.
         */
        bool Initialize(AssetRepository* assetRepository, UInt32 validationFlags = AssetValidation::DefaultValidationFlags());

        /**
         * Initializes the asset manager
         * @param assetConfig Asset configuration
         * @param validationFlags Flags defining various asset validation levels. See AssetSet::Initialize.
         * @return False if any needed member for initializing assetBuilders,
         *         including themselves, is 0 or couldn't be initialized. True otherwise.
         */
        bool Initialize(AssetConfig* assetConfig, UInt32 validationFlags = AssetValidation::DefaultValidationFlags());

        /**
         * Finalizes the asset manager
         */
        void Finalize();

        /**
         * @brief Enable iterative loading.
         *
         * If iterative asset loading is enabled, the scene build process is split into multiple
         *  parts. Retrieving a scene using the (Default)AssetProvider interface will result in a
         *  complete scene tree, but no or few device objects attached to it.
         * To fully control the build process, the ContentLoader interface has to be used.
         *
         * By default, this behavior is NOT enabled.
         *
         * @param iterativeLoading Enables(true)/Disables(false) the behavior described above.
         * @see ContentLoader::BuildSceneContext
         * @see ContentLoader::BuildScene2DContext
         */
        void SetIterativeLoadingEnabled(bool iterativeLoading) {
            m_isIterativeLoadingEnabled = iterativeLoading;
        }

        /**
         * Retrieves whether the iterative loading is enabled or not.
         * @return Boolean whether the iterative loading is enabled or not.
         * @see SetIterativeLoadingEnabled
         */
        bool IsIterativeLoadingEnabled() const {
            return m_isIterativeLoadingEnabled;
        }

        /**
         * @brief Retrieve the AssetDescriptor object associated to this DefaultAssetProvider instance.
         *
         * AssetDescriptor class lists asset names available for retrieval from this DefaultAsseProvider instance,
         *  version information of the current loaded asset file and other asset related information.
         *
         * @return AssetDescriptor reference.
         * @see AssetDescriptor
         */
        const Candera::AssetDescriptor& GetAssetDescriptor() const { return m_assetDescriptor; }

        // overrides AssetProvider::GetFontResourceById
        virtual ResourceDataHandle GetFontResourceById(Id id) override;

        // overrides AssetProvider::GetRawResourceById
        virtual ResourceDataHandle GetRawResourceById(Id id) override;

        // overrides AssetProvider::GetLanguagePack
        virtual Globalization::LanguagePack::SharedPointer GetLanguagePack(const Char* locale) override;

        Globalization::Culture::SharedPointer GetCulture(const Char* locale);

        Globalization::Culture::SharedPointer GetCulture(Id id);

        /**
         * @param assetLib Type of the library the object can be found.
         * @param libraryId Id of the library item.
         * @param nodeId Id of the node (0 in case of library item name is requested).
         * @return Object name.
         */
        const Char* GetNameById(AssetLib assetLib, Id libraryId, Id nodeId) const;

#ifdef CANDERA_3D_ENABLED
        // overrides AssetProvider::GetDefaultRenderMode
        virtual MemoryManagement::SharedPointer<RenderMode> GetDefaultRenderMode() override;

        // overrides AssetProvider::ReleaseCompositeGroup
        virtual void ReleaseCompositeGroup(CompositeGroup* compositeGroup) override;
#endif //CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
        // overrides AssetProvider::ReleaseCompositeGroup2D
        virtual void ReleaseCompositeGroup2D(CompositeGroup2D* compositeGroup) override;
#endif //CANDERA_2D_ENABLED

#ifdef CANDERA_TRANSITIONS_ENABLED
        /**
         * @return The Transitions::Rule::Set initialized from the current loaded asset.
         */
        Transitions::Rule::Set::SharedPointer GetTransitionRuleCollection() const { return m_ruleSet;}
#endif

        /**
         * Release all cached shared objects that are only referenced by the
         * the DefaultAssetProvider. The objects that are referenced from other
         * places are maintained, so that they may be retrieved at a later time.
         * It is recommended to call this function periodically, to remove
         * from the cache items that are no longer needed.
         * Depending on the available amount of memory, one of the following
         * scenarios may be used:
         * 1. For strict memory usage, it may be called after each Scene release.
         * 2. For increased sharing, it may be called after all active Scenes have
         *    been loaded, to remove objects from the old Scenes that are no longer
         *    needed.
         */
        void ReleaseOrphanedObjects();

        FEATSTD_RTTI_DECLARATION();
    private:
        AssetDescriptor m_assetDescriptor;
        Candera::Internal::AssetId m_themeId;
        Candera::Internal::AssetGroup::SharedPointer m_assetGroup;
        mutable FeatStd::Internal::CriticalSection m_themeChangeCriticalSection;
        FeatStd::Internal::CriticalSection m_initializeCriticalSection;

#ifdef CANDERA_TRANSITIONS_ENABLED
        Transitions::Rule::Set::SharedPointer m_ruleSet;
#endif
        Candera::Internal::DefaultAssetProviderInternal* m_internal;
        bool m_isIterativeLoadingEnabled;

        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1704, CANDERA_LINT_REASON_NONCOPYABLE)
            FEATSTD_MAKE_CLASS_UNCOPYABLE(DefaultAssetProvider);

        virtual ResourceHandle OpenRawResourceByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual ResourceHandle OpenFontResourceByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual Int32 GetResourceSizeInternal(ResourceHandle resHandle) override;
        virtual const void* GetResourceAddressInternal(ResourceHandle resHandle)override;
        virtual SizeType ReadResourceInternal(ResourceHandle resHandle, void* buffer, OffsetType firstByte, SizeType byteCount)override;
        virtual void CloseResourceInternal(ResourceHandle resHandle) override;
        virtual MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> GetAnimationByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual void ReleaseAnimationByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual FeatStd::MemoryManagement::SharedPointer<Bitmap> GetBitmapByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual void ReleaseBitmapByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual GraphicDeviceUnit* GetGraphicDeviceUnitByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual Int GetDisplayDataByAssetId(const Candera::Internal::AssetId& assetId, Int& width, Int& height) override;
        virtual void SetCurrentThemeByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual Candera::Internal::AssetId GetCurrentThemeAssetId() const override;
        virtual MemoryManagement::SharedPointer<TextRendering::SharedStyle> GetTextStyleByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual CameraGroup* GetCameraGroupByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual Candera::Internal::StateMachineBehaviorData* GetStateMachineBehaviorDataByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual MemoryManagement::SharedPointer<Globalization::LanguagePack> GetLanguagePackByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual MemoryManagement::SharedPointer<SharedClearMode> GetClearModeByAssetId(const Candera::Internal::AssetId& assetId);
        virtual WidgetBase* GetWidgetByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual Candera::Internal::AssetId GetAssetId(AssetLib assetLib, const Char* name) override;
        virtual const Char* GetNameByAssetId(const Candera::Internal::AssetId& assetId) const override;
        virtual WidgetBase* GetWidgetByItemPath(const ItemPathInfo& itemPathInfo) override;

#ifdef CANDERA_3D_ENABLED
        virtual MemoryManagement::SharedPointer<Shader> GetShaderByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual MemoryManagement::SharedPointer<VertexBuffer> GetVertexBufferByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual SceneContext* GetSceneByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual Node* GetReferencedTemplateByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual void ReleaseSceneByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual MemoryManagement::SharedPointer<Appearance> GetAppearanceByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual MemoryManagement::SharedPointer<Material> GetMaterialByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual MemoryManagement::SharedPointer<RenderMode> GetRenderModeByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual MemoryManagement::SharedPointer<AbstractShaderParamSetter> GetShaderParamSetterByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual MemoryManagement::SharedPointer<Texture> GetTextureByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual MemoryManagement::SharedPointer<BitmapTextureImage> GetBitmapTextureImageByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual CompositeGroup* CreateCompositeGroupByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual Node* GetNodeByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual Node* GetRootNode(AssetObjectType aoType, const Char* name) override;
        virtual bool GetCompositePropertyInfo(CompositeGroup* compositeGroup, const Char* propertyName, WidgetBase*& widget, const Char*& widgetPropertyName) override;
#endif //CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
        virtual Scene2DContext* GetScene2DByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual MemoryManagement::SharedPointer<BitmapImage2D> GetBitmapImage2DByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual void ReleaseScene2DByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual CompositeGroup2D* CreateCompositeGroup2DByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual Node2D* GetNode2DByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual MemoryManagement::SharedPointer<Effect2D> GetEffect2DByAssetId(const Candera::Internal::AssetId& assetId) override;
        virtual Node2D* GetRootNode2D(AssetObjectType aoType, const Char* name) override;
        virtual bool GetComposite2DPropertyInfo(CompositeGroup2D* compositeGroup, const Char* propertyName, WidgetBase*& widget, const Char*& widgetPropertyName) override;
#endif //CANDERA_2D_ENABLED

#ifdef CANDERA_SCRIPTING_ENABLED
        virtual MemoryManagement::SharedPointer<Scripting::Script> GetScriptByAssetId(const Candera::Internal::AssetId& assetId) override;
#endif //CANDERA_SCRIPTING_ENABLED


#ifdef CANDERA_TRANSITIONS_ENABLED
        Transitions::Rule::Set::SharedPointer InitializeTransitionRuleCollection();
#endif

        void InitializeCultures(AssetDescriptor::AssetIdIterator languagePackIds, Candera::Globalization::Culture::SharedPointer defaultCulture);
        void InitializeCultureList(AssetDescriptor::AssetIdIterator languagePackIds);
        void InitializeDefaultCulture(Candera::Globalization::Culture::SharedPointer defaultCulture) const;
        bool InitializeGlobalStateMachine();

        Candera::Internal::AssetId ResolveThemeId(const Candera::Internal::AssetId& assetId);
        FeatStd::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayer> CreateCompositeAnimation(const Candera::Internal::AssetId& assetId);
        Id GetId(AssetLib assetLib, const Char* name);
        Candera::Internal::BinaryTheme* GetThemeByAssetId(const Candera::Internal::AssetId& assetId);
        ResourceHandle OpenResource(AssetLib expectedLib, Candera::Internal::AssetId assetId);

        //Make DefaultAssetProviderFriend class a friend to be able to change the private m_keepCompositeAttachmentsAfterLoad flag.
        friend class CanderaDefaultAssetProviderFriend;
        //Please change this flag before the DefaultAssetProvider is initalized, to avoid any synchronization issue.
        bool m_keepCompositeAttachmentsAfterLoad;

        StateMachineBehavior* m_globalStateMachine;

        CANDERA_UNIT_TEST_TESTCASE_FRIEND(DefaultAssetProviderTest, InvalidAssetIdTest);
    };

    /** @} */ // end of AssetLoaderBase

}

#endif // CANDERA_DEFAULTASSETPROVIDER_H
