//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <FeatStd/Base.h>
#include "DefaultAssetProvider.h"
#include "DefaultAssetProviderInternal.h"
#include <FeatStd/Util/StaticObject.h>
#include <FeatStd/Util/PointerUtil.h>
#include <Candera/EngineBase/Common/CameraGroup.h>
#ifdef CANDERA_SCRIPTING_ENABLED
#include <CanderaScripting/Script.h>
#endif
#include <Candera/TextEngine/Style.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/GlobalizationBase/CultureDataType.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/System/MemoryManagement/CanderaHeap.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetConfig.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetIdMacros.h>
#include <CanderaWidget/WidgetBase/WidgetMetaInfo.h>
#include <CanderaPlatform/Device/Common/Base/SharedClearMode.h>
#include <CanderaBehavior/BehaviorBase/StateMachineBehavior.h>

#if defined (CANDERA_TRANSITIONS_ENABLED)
#include <CanderaTransitions/Internal/RuleAdder.h>
#include <CanderaTransitions/DefaultTransitionFactory.h>
#endif

#if defined (CANDERA_GLOBALIZATION_ENABLED)
    #include <CanderaGlobalization/GlobalizationCultureManager.h>
    #include <CanderaGlobalization/LanguagePack.h>
    #include <CanderaGlobalization/DefaultLocalizer.h>
#else
    #include <Candera/System/GlobalizationBase/CultureManager.h>
    FEATSTD_LINT_SYMBOL(1762, Candera::DefaultAssetProvider::InitializeCultureList, "'could be made const': Not done, as function cannot be const with CANDERA_GLOBALIZATION_ENABLED defined.")
    FEATSTD_LINT_SYMBOL(1762, Candera::DefaultAssetProvider::InitializeDefaultCulture, "'could be made const': Not done, as function cannot be const with CANDERA_GLOBALIZATION_ENABLED defined.")
    FEATSTD_LINT_SYMBOL(1762, Candera::DefaultAssetProvider::GetCulture, "'could be made const': Not done, as function cannot be const with CANDERA_GLOBALIZATION_ENABLED defined.")
#endif

#ifdef CANDERA_3D_ENABLED
    #include <Candera/Engine3D/Core/VertexBuffer.h>
    #include <Candera/Engine3D/Core/Shader.h>
    #include <Candera/Engine3D/Core/Scene.h>
    #include <Candera/Engine3D/Core/Billboard.h>
    #include <Candera/Engine3D/Core/Camera.h>
    #include <Candera/Engine3D/Core/PointSprite.h>
    #include <Candera/Engine3D/Core/ReflectionCamera.h>
    #include <Candera/Engine3D/Core/Renderer.h>
    #include <Candera/Engine3D/Core/CompositeGroup.h>
    #include <CanderaAssetLoader/AssetLoader3D/SceneContext.h>
    #include <Candera/Engine3D/Core/TreeTraverser.h>
#endif //CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    #include <Candera/Engine2D/Core/Camera2D.h>
    #include <Candera/Engine2D/Core/RenderNode.h>
    #include <Candera/Engine2D/Core/CompositeGroup2D.h>
    #include <Candera/Engine2D/Core/BitmapImage2D.h>
    #include <CanderaPlatform/Device/Common/Base/Effect2DMetaInformation.h>
    #include <CanderaAssetLoader/AssetLoader2D/Scene2DContext.h>
    #include <Candera/Engine2D/Core/TreeTraverser2D.h>
    #include <Candera/Engine2D/Core/Scene2D.h>
#endif //CANDERA_2D_ENABLED

#include <CanderaAssetLoader/AssetLoaderBase/FileAssetRepository.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDescriptor.h>
#include <CanderaAssetLoader/AssetLoaderBase/ContentLoader.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetSet.h>
#include <CanderaAssetLoader/AssetLoaderBase/Theme.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultResourceProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDecompression.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/MetaInfoAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/CffAssetReaderBase.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/StateMachineCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/RawResourceCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/TransitionRuleCollectionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/TransitionRuleCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/TransitionParamsCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/ItemHeaderCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/LibraryHeaderCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/LanguagePackCffReader.h>
#include <CanderaPlatform/Device/Common/Base/DeviceMetaInfo.h>
#include <CanderaPlatform/Device/Common/Base/DeviceMetaInfoHost.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageDescriptor.h>

namespace Candera {
    using namespace Candera::Internal;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);
    FEATSTD_RTTI_DEFINITION(DefaultAssetProvider, AssetProvider)

    static AssetDataHandle GetParent(const AssetDataHandle& handle)
    {
        if (!handle.IsValid()) {
            return handle;
        }

        Int32 parentIndex = CFFReader::GetItemParentIndex(handle);
        if (parentIndex == -1) {
            return AssetDataHandle();
        }

        return CFFReader::GetItem(CFFReader::GetItemListHandle(CFFReader::CffGetAssetLibHandle(handle)), parentIndex);
    }


#ifdef CANDERA_3D_ENABLED
    class CompositeAttachmentClearTraverser: public TreeTraverser
    {
    protected:
        virtual TraverserAction ProcessNode(Node& node)
        {
            CompositeGroup* compositeGroup = Dynamic_Cast<CompositeGroup*>(&node);
            if (compositeGroup != 0) {
                static_cast<void>(CompositePropertyList::ClearPropertyList(compositeGroup));
            }

            return ProceedTraversing;
        }
    };
#endif //CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    class Composite2DAttachmentClearTraverser: public TreeTraverser2D
    {
    protected:
        virtual TraverserAction ProcessNode(Node2D& node)
        {
            CompositeGroup2D* compositeGroup = Dynamic_Cast<CompositeGroup2D*>(&node);
            if (compositeGroup != 0) {
                static_cast<void>(Composite2DPropertyList::ClearPropertyList(compositeGroup));
            }

            return ProceedTraversing;
        }
    };
#endif //CANDERA_2D_ENABLED


    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    DefaultAssetProvider::DefaultAssetProvider() :
        m_themeId(),
        m_internal(0),
        m_isIterativeLoadingEnabled(false),
        m_keepCompositeAttachmentsAfterLoad(false),
        m_globalStateMachine(0)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    DefaultAssetProvider::~DefaultAssetProvider(void)
    {
        Finalize();
    }

    bool DefaultAssetProvider::LoadDisplayProperties(Id id)
    {
        return (Candera::Internal::CffNonCachingAssetProvider<Candera::Display*>(this).Get(AssetIdFunctions::CreateId(DisplayLib, AotUnknown, id, 0)) != 0);
    }

    bool DefaultAssetProvider::LoadDevicePackageProperties()
    {
        const AssetId& invalidAssetId = AssetId::InvalidId();
        CffLoaderContext devicePackageProperties = {invalidAssetId, this, AssetDataHandle(m_assetGroup->GetDevicePackageMetaInfoHandle()), 0, false, 0};
        
        DeviceMetaInfoHost* host = DevicePackageDescriptor::GetMetaInformationHost();
        if (host == 0) {
            return false;
        }
        const MetaInfo::DeviceMetaInfo* metaInfo = DevicePackageDescriptor::GetPackageMetaInformation();
        if (metaInfo == 0) {
            return false;
        }

        return (MetaInfoAssetBuilder<DeviceMetaInfoHost, MetaInfo::DevicePropertyMetaInfo, MetaInfo::DeviceMetaInfo>(*host, *metaInfo)).Read(devicePackageProperties);
    }

    /******************************************************************************
     *  GetInstance
     ******************************************************************************/
    DefaultAssetProvider& DefaultAssetProvider::GetInstance()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(DefaultAssetProvider, s_defaultAssetProvider);
        static bool initialized = false;

        if (!initialized) {
            ContentLoader::GetInstance().Init(&s_defaultAssetProvider);
            ResourceProvider::Instance() = &DefaultResourceProvider::GetResourceProviderInstance();
            initialized = true;
        }

        return s_defaultAssetProvider;
    }

    /******************************************************************************
     *  Initialize
     ******************************************************************************/
    bool DefaultAssetProvider::Initialize(AssetRepository* assetRepository, UInt32 validationFlags)
    {
        DefaultAssetConfig assetConfig;
        assetConfig.Initialize(&assetRepository, 1);

        return Initialize(&assetConfig, validationFlags);
    }

    bool DefaultAssetProvider::Initialize(AssetConfig* assetConfig, UInt32 validationFlags)
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker initLock(&m_initializeCriticalSection);
#endif

        if (m_internal != 0) {
            return false;
        }

        if (assetConfig == 0) {
            return false;
        }

        m_assetGroup = AssetGroup::Create();
        if (m_assetGroup == 0) {
            FEATSTD_LOG_ERROR("Initialization failed, asset group could not be created.");
            return false;
        }

        DefaultResourceProvider::GetResourceProviderInstance().m_assetGroup = m_assetGroup;

        if (!m_assetGroup->Initialize(*assetConfig, validationFlags)) {
            FEATSTD_LOG_ERROR("AssetGroup initialization failed.");
            Finalize();
            return false;
        }

        m_internal = FEATSTD_NEW(DefaultAssetProviderInternal)(this);
        if (m_internal == 0) {
            Finalize();
            return false;
        }

        m_assetDescriptor.m_assetGroup = m_assetGroup;

#ifndef CANDERA_CUSTOM_LOCALIZER_ENABLED
        AssetDescriptor::AssetIdIterator languagePackIds = m_assetDescriptor.GetAssetIdIterator(LanguagePackLib);
        Globalization::Culture::SharedPointer defaultCulture;
        if (!MetaInfo::DataType<Globalization::Culture::SharedPointer>::ConvertFromString(defaultCulture, m_assetDescriptor.GetDefaultCultureName())) {
            return false;
        }

        InitializeCultures(languagePackIds, defaultCulture);
#endif

        //set master theme as current theme
        SetCurrentThemeByAssetId(AssetIdFunctions::CreateId(ThemeLib, AotUnknown, m_assetDescriptor.GetMasterThemeAssetId(), 0));

#ifdef CANDERA_3D_ENABLED
        Renderer::SetDefaultRenderMode(GetDefaultRenderMode());
        AssetGroup::AssetSetIterator it = m_assetGroup->GetAssetSetIterator();
        if (it.IsValid()) {
            AssetDataHandle autoUniformsSemanticsHandle(m_assetGroup->GetAutoUniformsSemanticsHandle());

            if (autoUniformsSemanticsHandle.IsValid()) {
                for (Int index = 0; index < CFFReader::GetShaderParamsCount(autoUniformsSemanticsHandle); index++) {
                    ShaderParamNames::UniformSemantic semantic = static_cast<ShaderParamNames::UniformSemantic>(CFFReader::GetShaderParamSemantic(autoUniformsSemanticsHandle, index));
                    const Char* name = (*it)->GetName(CFFReader::GetShaderParamName(autoUniformsSemanticsHandle, index));
                    ShaderParamNames::SetUniformName(semantic, name);
                }
            }

            AssetDataHandle shaderAttributesSemanticsHandle(m_assetGroup->GetShaderAttributesSemanticsHandle());

            if (shaderAttributesSemanticsHandle.IsValid()) {
                for (Int index = 0; index < CFFReader::GetShaderParamsCount(shaderAttributesSemanticsHandle); index++) {
                    ShaderParamNames::AttributeSemantic semantic = static_cast<ShaderParamNames::AttributeSemantic>(CFFReader::GetShaderParamSemantic(shaderAttributesSemanticsHandle, index));
                    const Char* name = (*it)->GetName(CFFReader::GetShaderParamName(shaderAttributesSemanticsHandle, index));
                    ShaderParamNames::SetAttributeName(semantic, name);
                }
            }
        }
#endif //CANDERA_3D_ENABLED

#ifdef CANDERA_TRANSITIONS_ENABLED
        m_ruleSet = InitializeTransitionRuleCollection();
#endif //CANDERA_TRANSITIONS_ENABLED

        if (!InitializeGlobalStateMachine()) {
            FEATSTD_LOG_ERROR("Global StateMachine initialization failed");
        }

        return true;
    }

    bool DefaultAssetProvider::InitializeGlobalStateMachine()
    {
        m_globalStateMachine = FEATSTD_NEW(StateMachineBehavior);
        if (m_globalStateMachine == 0) {
            return false;
        }

        AssetDescriptor::AssetIdIterator stateMachineIds = m_assetDescriptor.GetAssetIdIterator(StateMachineLib);
        for (; stateMachineIds.IsValid(); ++stateMachineIds) {
            UInt16 repositoryId = 0;
            AssetId stateMachineAssetId = AssetIdFunctions::CreateId(StateMachineLib, AotUnknown, *stateMachineIds, 1);
            if (stateMachineAssetId.IsValid()) {
                AssetDataHandle stateMachineAssetDataHandle = m_assetGroup->GetItemDataHandle(stateMachineAssetId, &repositoryId);
                if (CFFReader::GetStateMachineIsGlobal(stateMachineAssetDataHandle)) {
                    m_globalStateMachine->SetStateMachine(GetStateMachineBehaviorDataByAssetId(stateMachineAssetId));
                }
            }
        }

        StateMachineBehavior::SetGlobalStateMachine(m_globalStateMachine);

        return true;
    }

    /******************************************************************************
     *  InitializeCultures
     ******************************************************************************/
    inline void DefaultAssetProvider::InitializeCultures(AssetDescriptor::AssetIdIterator languagePackIds, Globalization::Culture::SharedPointer defaultCulture)
    {
#if defined(CANDERA_GLOBALIZATION_ENABLED)
        // register DefaultLocalizer
        Globalization::GlobalizationCultureManager::GetInstance().SetLocalizer(Globalization::Localizer::SharedPointer(&Globalization::Internal::DefaultLocalizer::GetInstance()));

        // DefaultLocalizer needs a link to the asset provider to be able to load language packs
        Globalization::Internal::DefaultLocalizer::GetInstance().SetAssetProvider(this);
#endif
        InitializeCultureList(languagePackIds);
        InitializeDefaultCulture(defaultCulture);
    }

    /******************************************************************************
     *  InitializeCultureList
     ******************************************************************************/
    inline void DefaultAssetProvider::InitializeCultureList(AssetDescriptor::AssetIdIterator languagePackIds)
    {
#if defined(CANDERA_GLOBALIZATION_ENABLED)
        Globalization::GlobalizationCultureManager& cultureManager = Globalization::GlobalizationCultureManager::GetInstance();
        cultureManager.RemoveAllCultures();
        for (; languagePackIds.IsValid(); ++languagePackIds) {
            Globalization::Culture::SharedPointer result = GetCulture(*languagePackIds);
            if(result.PointsToNull())
            {
                FEATSTD_LOG_WARN("Initialized culture list contains nullpointer");
            }
            cultureManager.AddCulture(result);
        }
#else
        FEATSTD_UNUSED(languagePackIds);
#endif
    }

 
    /******************************************************************************
     *  InitializeDefaultCulture
     ******************************************************************************/
    inline void DefaultAssetProvider::InitializeDefaultCulture(Globalization::Culture::SharedPointer defaultCulture) const
    {
#if defined(CANDERA_GLOBALIZATION_ENABLED)
        Globalization::GlobalizationCultureManager& cultureManager = Globalization::GlobalizationCultureManager::GetInstance(); 

        Globalization::Culture::SharedPointer currentCulture;
        if (!defaultCulture.PointsToNull()) {
            currentCulture = cultureManager.GetCulture(defaultCulture->GetLocale()); // search for Scene Composer-defined default culture in available cultures
        }

        if (currentCulture.PointsToNull()) {
            FEATSTD_LOG_INFO("Scene Composer-defined default culture not found. Using first available culture."); 
            currentCulture = cultureManager.GetCulture(UInt16(0)); // if not found, use first available culture
        } 
        
        if (!currentCulture.PointsToNull()) {
            defaultCulture = currentCulture; // ensure that default culture references existing culture in culture manager
        } else {
            FEATSTD_LOG_INFO("No cultures available in asset."); 
        }

        static_cast<void>(cultureManager.SetCurrentCulture(currentCulture));
#else
        Globalization::CultureManager& cultureManager = Globalization::CultureManager::GetInstance();
#endif
        cultureManager.SetDefaultCulture(defaultCulture);
    }

    /******************************************************************************
     *  Finalize
     ******************************************************************************/
    void DefaultAssetProvider::Finalize()
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker initLock(&m_initializeCriticalSection);
#endif

        if (m_globalStateMachine != 0) {
            m_globalStateMachine->SetGlobalStateMachine(0);
            StateMachineBehavior::SetGlobalStateMachine(0);
            FEATSTD_DELETE(m_globalStateMachine);
            m_globalStateMachine = 0;
        }

#if defined(CANDERA_GLOBALIZATION_ENABLED)
        // unregister DefaultLocalizer
        Globalization::GlobalizationCultureManager::GetInstance().SetLocalizer(Globalization::Localizer::SharedPointer());
#endif

        m_assetGroup.Release();

        m_assetDescriptor.m_assetGroup.Release();

        if (m_internal != 0) {
            FEATSTD_DELETE(m_internal);
            m_internal = 0;
        }

        DefaultResourceProvider::GetResourceProviderInstance().m_assetGroup.Release();

#ifdef CANDERA_TRANSITIONS_ENABLED
        m_ruleSet.Release();
#endif //CANDERA_TRANSITIONS_ENABLED
    }

#define ASSIGN_PROVIDER(provider) typedAssetProvider = &provider; break

#if defined(CANDERA_2D_ENABLED)
#define ASSIGN_2D_PROVIDER(provider) ASSIGN_PROVIDER(provider)
#else
#define ASSIGN_2D_PROVIDER(provider) break
#endif

#if defined(CANDERA_3D_ENABLED)
#define ASSIGN_3D_PROVIDER(provider) ASSIGN_PROVIDER(provider)
#else
#define ASSIGN_3D_PROVIDER(provider) break
#endif


    /******************************************************************************
     *  GetCulture
     ******************************************************************************/
#if defined(CANDERA_GLOBALIZATION_ENABLED)
    Globalization::Culture::SharedPointer DefaultAssetProvider::GetCulture(const Char* locale)
    {
        return GetCulture(GetId(LanguagePackLib, locale));
    }
    Globalization::Culture::SharedPointer DefaultAssetProvider::GetCulture(Id id)
    {
        if (m_internal == 0) {
            return Globalization::Culture::SharedPointer();
        }
        return m_internal->m_cultureProvider.Get(AssetIdFunctions::CreateId(LanguagePackLib, AotUnknown, id, 0));
    }
#else
    Globalization::Culture::SharedPointer DefaultAssetProvider::GetCulture(const Char* /*locale*/)
    {
        return Globalization::Culture::SharedPointer();
    }
    Globalization::Culture::SharedPointer DefaultAssetProvider::GetCulture(Id /*id*/)
    {
        return Globalization::Culture::SharedPointer();
    }
#endif
    
    Animation::AnimationPlayer::SharedPointer DefaultAssetProvider::CreateCompositeAnimation(const AssetId& assetId)
    {
        if ((m_internal == 0) || (assetId.m_libraryType != AnimationLib)) {
            return Animation::AnimationPlayer::SharedPointer();
        }
        return m_internal->m_compositeAnimationPlayerProvider.Get(assetId);
    }


    const Char* AssetProviderFunctions::GetName(const DefaultAssetProvider* provider, UInt16 repositoryId, Int32 offset)
    {
        if (provider->m_assetGroup == 0) {
            return 0;
        }

        for (AssetGroup::AssetSetIterator it = provider->m_assetGroup->GetAssetSetIterator(); it.IsValid(); ++it) {
            if ((*it)->GetId() == repositoryId) {
                return (*it)->GetName(offset);
            }
        }
        return 0;
    }

    Id AssetProviderFunctions::GetIdByName(AssetProvider* provider, AssetLib assetLib, const Char* name)
    {
        return AssetIdFunctions::GetLibraryId(provider->GetAssetId(assetLib, name));
    }

    AssetId DefaultAssetProvider::ResolveThemeId(const AssetId& assetId)
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        //block theme change during theme resolval
        FeatStd::Internal::CriticalSectionLocker lock(&m_themeChangeCriticalSection);
#endif
        if (assetId.m_libraryType != ThemeLib) {
            return assetId;
        }

        AssetDataHandle entryHandle = m_assetGroup->GetLibItemHeader(assetId);
        if (!entryHandle.IsValid()) {
            return AssetId::InvalidId();
        }

        Id entryId = CFFReader::GetItemId(entryHandle);
        AssetDataHandle collectionHandle = GetParent(entryHandle);
        if (!collectionHandle.IsValid()) {
            return AssetId::InvalidId();
        }

        Id collectionId = CFFReader::GetItemId(collectionHandle);
        BinaryTheme* theme = GetThemeByAssetId(GetCurrentThemeAssetId());
        if (theme == 0) {
            theme = GetThemeByAssetId(assetId);
        }
        if (theme == 0) {
            FEATSTD_LOG_WARN("GetThemeByAssetId returns 0.");
            return AssetId::InvalidId();
        }

        BinaryTheme::AssetIdEntryCollection* collection = theme->GetAssetIdEntryCollection(collectionId);
        if (0 == collection) {
            return AssetId::InvalidId();
        }
        AssetId* result = collection->Find(entryId);
        
        if (0 == result) {
            return AssetId::InvalidId();
        }

        if (!result->IsValid()) {
            // Search in master theme
            AssetId masterThemeAssetId = AssetIdFunctions::CreateId(ThemeLib, AotUnknown, GetAssetDescriptor().GetMasterThemeAssetId(), 1);
            BinaryTheme* masterTheme = GetThemeByAssetId(masterThemeAssetId);
            if ((0 != masterTheme) && (theme != masterTheme)) {
                collection = masterTheme->GetAssetIdEntryCollection(collectionId);
                if (0 == collection) {
                    return AssetId::InvalidId();
                }
                result = collection->Find(entryId);
            }
        }
        
        if (0 == result) {
            return AssetId::InvalidId();
        }

        result->m_instanceId = assetId.m_instanceId;
        return *result;
    }

    ResourceDataHandle DefaultAssetProvider::GetFontResourceById(Id id)
    {
        if (m_assetGroup == 0) {
            return ResourceDataHandle::InvalidHandle();
        }
        const AssetDataHandle& dataHandle = m_assetGroup->GetItemDataHandle(AssetIdFunctions::CreateId(FontResourceLib, AotUnknown, id, 1), 0);
        if (!dataHandle.IsValid()) {
            return ResourceDataHandle::InvalidHandle();
        }
        return DefaultResourceProvider::GetResourceProviderInstance().CreateRegionDataHandle(FontResourceLib, id, static_cast<UInt16>(CFFReader::GetRawResourceRawDataRegion(dataHandle)), static_cast<UInt32>(CFFReader::GetRawResourceRawDataLength(dataHandle)));
    }

    ResourceDataHandle DefaultAssetProvider::GetRawResourceById(Id id)
    {
        if (m_assetGroup == 0) {
            return ResourceDataHandle::InvalidHandle();
        }
        const AssetDataHandle& dataHandle = m_assetGroup->GetItemDataHandle(AssetIdFunctions::CreateId(RawResourceLib, AotUnknown, id, 1), 0);
        if (!dataHandle.IsValid()) {
            return ResourceDataHandle::InvalidHandle();
        }
        return DefaultResourceProvider::GetResourceProviderInstance().CreateRegionDataHandle(RawResourceLib, id, static_cast<UInt16>(CFFReader::GetRawResourceRawDataRegion(dataHandle)), static_cast<UInt32>(CFFReader::GetRawResourceRawDataLength(dataHandle)));
    }

    ResourceHandle DefaultAssetProvider::OpenRawResourceByAssetId(const AssetId& assetId)
    {
        ResourceDataHandle* resource = ASSETLOADER_TRANSIENT_NEW(ResourceDataHandle);
        if (resource != 0) {
            *resource = GetRawResourceById(AssetIdFunctions::GetLibraryId(assetId));
            if (!resource->m_accessor.IsValid()) {
                CloseResourceInternal(resource);
                resource = 0;
            }
        }

        return resource;
    }

    ResourceHandle DefaultAssetProvider::OpenFontResourceByAssetId(const AssetId& assetId)
    {
        ResourceDataHandle* resource = ASSETLOADER_TRANSIENT_NEW(ResourceDataHandle);
        if (resource != 0) {
            *resource = GetFontResourceById(AssetIdFunctions::GetLibraryId(assetId));
            if (!resource->m_accessor.IsValid()) {
                CloseResourceInternal(resource);
                resource = 0;
            }
        }

        return resource;
    }

    Int32 DefaultAssetProvider::GetResourceSizeInternal(ResourceHandle resHandle)
    {
        ResourceDataHandle* resource = FeatStd::Internal::PointerToPointer< ResourceDataHandle*>(resHandle);
        if (resource == 0) {
            return 0;
        }

        return resource->m_size;
    }

    const void* DefaultAssetProvider::GetResourceAddressInternal(ResourceHandle resHandle)
    {
        ResourceDataHandle* resource = FeatStd::Internal::PointerToPointer< ResourceDataHandle*>(resHandle);
        if (resource == 0) {
            return 0;
        }

        return ResourceObject<const void>::GetAddress(*resource);
    }

    SizeType DefaultAssetProvider::ReadResourceInternal(ResourceHandle resHandle, void* buffer, OffsetType firstByte, SizeType byteCount)
    {
        ResourceDataHandle* resource = FeatStd::Internal::PointerToPointer< ResourceDataHandle*>(resHandle);
        if (resource == 0) {
            return 0;
        }

        return ResourceObject<const void>::CopyData(buffer, *resource, firstByte, byteCount);
    }

    void DefaultAssetProvider::CloseResourceInternal(ResourceHandle resHandle)
    {
        if (resHandle != 0) {
            ASSETLOADER_DELETE(FeatStd::Internal::PointerToPointer<ResourceDataHandle*>(resHandle));
        }
    }

    SharedPointer<Animation::AnimationPlayerBase> DefaultAssetProvider::GetAnimationByAssetId(const AssetId& assetId)
    {
        Animation::AnimationPlayerBase::SharedPointer result;
        if ((m_internal == 0) || (!assetId.IsValid())) {
            return result;
        }
        const AssetId& id = ResolveThemeId(assetId);
        
#ifdef CANDERA_3D_ENABLED
        if (m_internal->m_currentCompositeGroupBuild != 0) {
            result = m_internal->m_currentCompositeGroupBuild->GetAnimation(AssetIdFunctions::GetLibraryId(id));
            if (!result.PointsToNull()) {
                return result;
            }
        }
#endif //CANDERA_3D_ENABLED
#ifdef CANDERA_2D_ENABLED
        if (m_internal->m_currentCompositeGroup2DBuild != 0) {
            result = m_internal->m_currentCompositeGroup2DBuild->GetAnimation(AssetIdFunctions::GetLibraryId(id));
            if (!result.PointsToNull()) {
                return result;
            }
        }
#endif //CANDERA_2D_ENABLED
        switch (id.m_libraryType)
        {
            case AnimationLib: result = m_internal->m_animationPlayerProvider.Get(id); break;
            case AnimationGroupLib: result = m_internal->m_animationGroupPlayerProvider.Get(id); break;
            default: break;
        }
        return result;
    }

    void DefaultAssetProvider::ReleaseAnimationByAssetId(const AssetId& assetId)
    {
        if (m_internal == 0) {
            return;
        }
        switch (assetId.m_libraryType)
        {
        case AnimationLib: m_internal->m_animationPlayerProvider.Release(assetId, true); break;
        case AnimationGroupLib: m_internal->m_animationGroupPlayerProvider.Release(assetId, true); break;
        default: break;
        }
    }

    SharedPointer<Bitmap> DefaultAssetProvider::GetBitmapByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != BitmapLib)) {
            return SharedPointer<Bitmap>(0);
        }
        return m_internal->m_bitmapProvider.Get(id);
    }

    void DefaultAssetProvider::ReleaseBitmapByAssetId(const AssetId& assetId)
    {
        if ((m_internal == 0) || (assetId.m_libraryType != BitmapLib)) {
            return;
        }
        m_internal->m_bitmapProvider.Release(assetId, true);
    }

    GraphicDeviceUnit* DefaultAssetProvider::GetGraphicDeviceUnitByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != RenderTargetLib)) {
            return 0;
        }
        return m_internal->m_gduProvider.Get(id);
    }

    Int DefaultAssetProvider::GetDisplayDataByAssetId(const AssetId& assetId, Int& width, Int& height)
    {
        if ((m_internal == 0) || (assetId.m_libraryType != DisplayLib)) {
            return -1;
        }
        DisplayData* displayData = m_internal->m_displayProvider.Get(assetId);

        if (displayData == 0) {
            return -1;
        }

        width = displayData->m_width;
        height = displayData->m_height;
        return displayData->m_id;
    }

    void DefaultAssetProvider::SetCurrentThemeByAssetId(const AssetId& assetId)
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker lock(&m_themeChangeCriticalSection);
#endif
        m_themeId = assetId;
    }

    AssetId DefaultAssetProvider::GetCurrentThemeAssetId() const
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker lock(&m_themeChangeCriticalSection);
#endif
        return m_themeId;
    }

    BinaryTheme* DefaultAssetProvider::GetThemeByAssetId(const AssetId& assetId)
    {
        if ((m_internal == 0) || (assetId.m_libraryType != ThemeLib)) {
            return 0;
        }
        return m_internal->m_binaryThemeProvider.Get(assetId);
    }

    SharedPointer<TextRendering::SharedStyle> DefaultAssetProvider::GetTextStyleByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != TextStyleLib)) {
            return SharedPointer<TextRendering::SharedStyle>();
        }
        return m_internal->m_styleProvider.Get(id);
    }

    Candera::Internal::StateMachineBehaviorData* DefaultAssetProvider::GetStateMachineBehaviorDataByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != StateMachineLib)) {
            return 0;
        }
        return m_internal->m_stateMachineProvider.Get(id);
    }

    CameraGroup* DefaultAssetProvider::GetCameraGroupByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != CameraGroupLib)) {
            return 0;
        }
        return m_internal->m_cameraGroupProvider.Get(id);
    }

    Globalization::LanguagePack::SharedPointer DefaultAssetProvider::GetLanguagePack(const Char* locale)
    {
        return GetLanguagePackByAssetId(GetAssetId(LanguagePackLib, locale));
    }

#if defined(CANDERA_GLOBALIZATION_ENABLED)
    SharedPointer<Globalization::LanguagePack> DefaultAssetProvider::GetLanguagePackByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != LanguagePackLib)) {
            return Globalization::LanguagePack::SharedPointer();
        }
        if (m_internal->m_currentLanguagePackId == id) {
            return m_internal->m_currentLanguagePack;
        }
        m_internal->m_currentLanguagePackAssetDataHandle = m_assetGroup->GetItemDataHandle(id, 0);
        if (m_internal->m_currentLanguagePackAssetDataHandle.IsValid()) {
            m_internal->m_currentLanguagePack = Globalization::LanguagePack::Create(reinterpret_cast<Globalization::Handle>(CFFReader::GetLanguagePackData(m_internal->m_currentLanguagePackAssetDataHandle)));
            m_internal->m_currentLanguagePackId = id;
        }
        else {
            m_internal->m_currentLanguagePackId = AssetId::InvalidId();
        }
        
        return m_internal->m_currentLanguagePack;
    }
#else
    SharedPointer<Globalization::LanguagePack> DefaultAssetProvider::GetLanguagePackByAssetId(const AssetId& /*id*/)
    {
        return Globalization::LanguagePack::SharedPointer();
    }
#endif

    WidgetBase* DefaultAssetProvider::GetWidgetByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if (!id.IsValid()) {
            FEATSTD_LOG_WARN("ResolveThemeId is not valid");
        }
        if (m_internal == 0) {
            return 0;
        }

        WidgetBase* result = 0;
        switch (id.m_libraryType) {
#ifdef CANDERA_3D_ENABLED
            case CompositeLib:
                if (m_internal->m_currentCompositeGroupBuild != 0) {
                    return m_internal->m_currentCompositeGroupBuild->GetWidget(AssetIdFunctions::GetNodeId(id));
                }
                break;
            case SceneLib: {
                SceneContext* sceneContext = GetSceneByAssetId(id);
                if (sceneContext != 0) {
                    return sceneContext->GetWidget(AssetIdFunctions::GetNodeId(id));
                }
                else {
                    FEATSTD_LOG_WARN("GetSceneByAssetId returns 0.");
                }
                break;
                           }
#endif
#ifdef CANDERA_2D_ENABLED
            case Composite2DLib:
                if (m_internal->m_currentCompositeGroup2DBuild != 0) {
                    return m_internal->m_currentCompositeGroup2DBuild->GetWidget(AssetIdFunctions::GetNodeId(id));
                }
                break;
            case Scene2DLib: {
                Scene2DContext* sceneContext = GetScene2DByAssetId(id);
                if (sceneContext != 0) {
                    return sceneContext->GetWidget(AssetIdFunctions::GetNodeId(id));
                }
                else{
                    FEATSTD_LOG_WARN("GetScene2DByAssetId returns 0.");
                }
                break;
                             }
#endif
            default:
                break;
        }

        return result;
    }

#if defined (CANDERA_TRANSITIONS_ENABLED)
    namespace TransitionIdentifier {
        enum Type
        {
            None = 0,  ///< Matches only itself
            SpecificScene2D,   ///< Identifies a specific 2D Scene
            SpecificScene3D,   ///< Identifies a specific 3D Scene
            SpecificNode2D,    ///< Identifies a specific 2D Node
            SpecificNode3D,    ///< Identifies a specific 3D Node
            AnyScene2D,///< Identifier for any 2D Scene
            AnyScene3D,///< Identifier for any 3D Scene
            AnyNode2D, ///< Identifier for any 2D Node
            AnyNode3D, ///< Identifier for any 3D Node
            ExternCustomIdentifier      ///< Custom Identifier implementation
        };
    }

    Transitions::Identifier GetIdentifier(TransitionIdentifier::Type type, const AssetId& assetId)
    {
        Id sceneId = AssetIdFunctions::GetLibraryId(assetId);
        Id nodeId = AssetIdFunctions::GetNodeId(assetId);
        Transitions::Identifier returnId = Transitions::Identifier();

        switch (type) {
            case TransitionIdentifier::None:
            case TransitionIdentifier::ExternCustomIdentifier:
                break;
            case TransitionIdentifier::SpecificScene2D:
                returnId = Transitions::Identifier(Transitions::Identifier::Scene2DType, sceneId);
                break;
            case TransitionIdentifier::SpecificScene3D:
                returnId = Transitions::Identifier(Transitions::Identifier::Scene3DType, sceneId);
                break;
            case TransitionIdentifier::SpecificNode2D:
                returnId = Transitions::Identifier(Transitions::Identifier::Scene2DType, sceneId, nodeId);
                break;
            case TransitionIdentifier::SpecificNode3D:
                returnId = Transitions::Identifier(Transitions::Identifier::Scene3DType, sceneId, nodeId);
                break;
            case TransitionIdentifier::AnyScene2D:
                returnId = Transitions::Identifier(Transitions::Identifier::AnyScene2D);
                break;
            case TransitionIdentifier::AnyScene3D:
                returnId = Transitions::Identifier(Transitions::Identifier::AnyScene3D);
                break;
            case TransitionIdentifier::AnyNode2D:
                returnId = Transitions::Identifier(Transitions::Identifier::AnyNode2D);
                break;
            case TransitionIdentifier::AnyNode3D:
                returnId = Transitions::Identifier(Transitions::Identifier::AnyNode3D);
                break;
            default:
                FEATSTD_LOG_WARN("Invalid TransitionRule Identifier in asset.");
                break;
        }

        return returnId;
    }

    Transitions::Rule::Set::SharedPointer DefaultAssetProvider::InitializeTransitionRuleCollection()
    {
        AssetDataHandle trcAssetHandle(m_assetGroup->GetTransitionRuleCollectionHandle());
        if (!trcAssetHandle.IsValid()) {
            FEATSTD_LOG_WARN("Invalid TransitionRuleCollection in asset.");
            return Transitions::Rule::Set::SharedPointer(0);
        }

        Candera::Internal::RuleAdder::BeginRuleSet();

        for (Int trIndex = 0; trIndex < CFFReader::GetTransitionRuleCollectionChildrenCount(trcAssetHandle); ++trIndex) {
            const AssetDataHandle& trAssetHandle = CFFReader::GetTransitionRuleCollectionChildrenElementAt(trcAssetHandle, trIndex);
            if (!trAssetHandle.IsValid()) {
                FEATSTD_LOG_WARN("Invalid TransitionRule is asset.");
                break;
            }

            TransitionIdentifier::Type sourceType = static_cast<TransitionIdentifier::Type>(CFFReader::GetTransitionRuleSourceIdentifier(trAssetHandle));
            const AssetId& sourceAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetTransitionRuleSourceItem(trAssetHandle));
            if (!sourceAssetId.IsValid()) {
                FEATSTD_LOG_DEBUG("Invalid asset ID! Transition is missing a source item!");
            }
            const Transitions::Identifier& sourceIdentifier = GetIdentifier(sourceType, sourceAssetId);

            TransitionIdentifier::Type destinationType = static_cast<TransitionIdentifier::Type>(CFFReader::GetTransitionRuleDestinationIdentifier(trAssetHandle));
            const AssetId& destinationAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetTransitionRuleDestinationItem(trAssetHandle));
            if (!destinationAssetId.IsValid()) {
                FEATSTD_LOG_DEBUG("Invalid asset ID! Transition is missing a destination item!");
            }
            const Transitions::Identifier& destinationIdentifier = GetIdentifier(destinationType, destinationAssetId);

            Transitions::TransitionFragmentFactory::SharedPointer factory;
            if (CFFReader::GetTransitionRuleFactory(trAssetHandle) == 0) {
                factory = Transitions::TransitionFragmentFactory::SharedPointer(ASSETLOADER_TRANSIENT_NEW(Transitions::DefaultTransitionFactory));
            }
            if (factory.PointsToNull()) {
                FEATSTD_LOG_WARN("Invalid TransitionFragmentFactory is asset.");
            }

            bool isAllowedReverse = CFFReader::GetTransitionRuleIsReversible(trAssetHandle);
            bool isBidirectional = CFFReader::GetTransitionRuleIsBidirectional(trAssetHandle);

            const AssetDataHandle& trpAssetHandle = CFFReader::GetTransitionRuleParams(trAssetHandle);
            if (!trpAssetHandle.IsValid()) {
                FEATSTD_LOG_WARN("Invalid TransitionRuleParams is asset.");
                break;
            }

            const Char* trvName = m_assetGroup->GetAssetSetIterator()->GetSharedInstance().GetName(CFFReader::GetTransitionRuleVariant(trAssetHandle));
            FeatStd::Optional<Float> fade;
            FeatStd::Optional<Vector2> fadeTimeline;
            FeatStd::Optional<Vector3> slide;
            FeatStd::Optional<Vector2> slideTimeline;
            FeatStd::Optional<Vector3> scale;
            FeatStd::Optional<Vector2> scaleTimeline;
            FeatStd::Optional<Candera::Id> animationId;
            FeatStd::Optional<Vector2> animationTimeline;
            FeatStd::Optional<Candera::Transitions::Hint::FragmentStrategy> activationStrategy;
            FeatStd::Optional<Candera::Transitions::Hint::FragmentStrategy> deactivationStrategy;
            FeatStd::Optional<Float> activationDelay;
            FeatStd::Optional<Float> deactivationDelay;
            Transitions::CustomHint::SharedPointer customHint;
            Float timelineValues[2];
            Float values[3];

            if (CFFReader::GetTransitionParamsIsFadeEnabled(trpAssetHandle)) {
                fade = CFFReader::GetTransitionParamsFade(trpAssetHandle);
            }

            if (CFFReader::GetTransitionParamsIsFadeTimelineEnabled(trpAssetHandle)) {
                CFFReader::GetTransitionParamsFadeTimeline(trpAssetHandle, timelineValues[0], timelineValues[1]);
                fadeTimeline = Vector2(timelineValues[0], timelineValues[1]);
            }

            if (CFFReader::GetTransitionParamsIsSlideEnabled(trpAssetHandle)) {
                CFFReader::GetTransitionParamsSlide(trpAssetHandle, values[0], values[1], values[2]);
                slide = Vector3(values[0], values[1], values[2]);
            }

            if (CFFReader::GetTransitionParamsIsSlideTimelineEnabled(trpAssetHandle)) {
                CFFReader::GetTransitionParamsSlideTimeline(trpAssetHandle, timelineValues[0], timelineValues[1]);
                slideTimeline = Vector2(timelineValues[0], timelineValues[1]);
            }

            if (CFFReader::GetTransitionParamsIsScaleEnabled(trpAssetHandle)) {
                CFFReader::GetTransitionParamsScale(trpAssetHandle, values[0], values[1], values[2]);
                scale = Vector3(values[0], values[1], values[2]);
            }

            if (CFFReader::GetTransitionParamsIsScaleTimelineEnabled(trpAssetHandle)) {
                CFFReader::GetTransitionParamsScaleTimeline(trpAssetHandle, timelineValues[0], timelineValues[1]);
                scaleTimeline = Vector2(timelineValues[0], timelineValues[1]);
            }


            if (CFFReader::GetTransitionParamsIsAnimationEnabled(trpAssetHandle)) {
                AssetId animationAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetTransitionParamsAnimationItem(trpAssetHandle));
                if (!animationAssetId.IsValid()) {
                    FEATSTD_LOG_DEBUG("Animation AssetId is not valid");
                }
                animationId = AssetIdFunctions::GetLibraryId(animationAssetId);
            }

            if (CFFReader::GetTransitionParamsIsAnimationTimelineEnabled(trpAssetHandle)) {
                CFFReader::GetTransitionParamsAnimationTimeline(trpAssetHandle, timelineValues[0], timelineValues[1]);
                animationTimeline = Vector2(timelineValues[0], timelineValues[1]);
            }
            
            activationStrategy = static_cast<Candera::Transitions::Hint::FragmentStrategy>(CFFReader::GetTransitionParamsActivationStrategy(trpAssetHandle));
            activationDelay = CFFReader::GetTransitionParamsActivationStrategyDelay(trpAssetHandle);

            deactivationStrategy = static_cast<Candera::Transitions::Hint::FragmentStrategy>(CFFReader::GetTransitionParamsDeactivationStrategy(trpAssetHandle));
            deactivationDelay = CFFReader::GetTransitionParamsDeactivationStrategyDelay(trpAssetHandle);

            Transitions::Hint hint(trvName, fade, fadeTimeline, slide, slideTimeline, scale, scaleTimeline, animationId, animationTimeline, activationStrategy, deactivationStrategy, activationDelay, deactivationDelay, customHint, 
                CFFReader::GetTransitionParamsIsSlideRelative(trpAssetHandle),
                CFFReader::GetTransitionParamsIsScaleRelative(trpAssetHandle),
                CFFReader::GetTransitionParamsIsAnimationTimeRelative(trpAssetHandle));

            Candera::Internal::RuleAdder::AddRule(sourceIdentifier, destinationIdentifier, hint, factory, isAllowedReverse, isBidirectional);
        }

        return Candera::Internal::RuleAdder::EndRuleSet();
    }
#endif

#ifdef CANDERA_3D_ENABLED

    SharedPointer<Shader> DefaultAssetProvider::GetShaderByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if (!id.IsValid()) {
            FEATSTD_LOG_DEBUG("ResolveThemeId is not valid");
        }
        if ((m_internal == 0) || (id.m_libraryType != ShaderLib)) {
            return SharedPointer<Shader>();
        }
        return m_internal->m_shaderProvider.Get(id);
    }

    SharedPointer<VertexBuffer> DefaultAssetProvider::GetVertexBufferByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if (!id.IsValid()) {
            FEATSTD_LOG_DEBUG("ResolveThemeId is not valid");
        }
        if ((m_internal == 0) || (id.m_libraryType != VertexBufferLib)) {
            return SharedPointer<VertexBuffer>();
        }
        return m_internal->m_vertexBufferProvider.Get(id);
    }

    SceneContext* DefaultAssetProvider::GetSceneByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if (!id.IsValid()) {
            FEATSTD_LOG_DEBUG("ResolveThemeId is not valid");
        }
        if ((m_internal == 0) || (id.m_libraryType != SceneLib)) {
            return 0;
        }

        // Retrieve asset's address from cache.
        SceneContext* sceneContext = m_internal->m_sceneProvider.GetFromCache(id);

        if (sceneContext == 0) {
            // Asset was not cached before, so cache it now.
            sceneContext = m_internal->m_sceneProvider.Get(id);

            //CgiApp wants to have access to the composite attachments after the scene is loaded
            if (!m_keepCompositeAttachmentsAfterLoad) {
                if (sceneContext != 0) {
                    if (sceneContext->GetScene() != 0) {
                        CompositeAttachmentClearTraverser().Traverse(*sceneContext->GetScene());
                    }
                    else {
                        // Scene pointer is null.
                    }
                }
                else {
                    // No asset with given ID was found.
                }
            }
            else {
                // Composite attachments are kept.
            }
        }
        else {
            // Asset was already cached. Don't traverse scene again.
        }

        return sceneContext;
    }

    Node* DefaultAssetProvider::GetReferencedTemplateByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != ReferencedTemplateLib)) {
            return 0;
        }
        ReferencedTemplate* referenceTemplate = m_internal->m_referenceTemplateProvider.Get(id);
        if (referenceTemplate != 0) { 
            return referenceTemplate->m_node;
        }

        return 0;
    }

    void DefaultAssetProvider::ReleaseSceneByAssetId(const AssetId& assetId)
    {
        if ((m_internal == 0) || (assetId.m_libraryType != SceneLib)) {
            return;
        }
        m_internal->m_sceneProvider.Release(assetId, true);
    }

    static Node* GetNodeRecursive(Node& root, const AssetDataHandle& nodeHandle)
    {
        Id nodeId = CDA_NODE_ASSETID(CFFReader::GetItemId(nodeHandle));
        AssetDataHandle parentHandle = GetParent(nodeHandle);
        if (parentHandle.IsValid()) {
            Node* parent = GetNodeRecursive(root, parentHandle);
            if (parent != 0) {
                AssetObjectType nodeType = static_cast<AssetObjectType>(CFFReader::GetItemType(nodeHandle));
                if (nodeType == AotAnchor) {
                    CompositeGroup* compositeGroup = Dynamic_Cast<CompositeGroup*>(parent);
                    if (compositeGroup != 0) {
                        return compositeGroup->GetAnchorPointNode(nodeId);
                    }
                }
                else {
                    return parent->GetChild(nodeId);
                }
            }
        }
        else {
            return root.GetChild(nodeId);
        }

        return 0;
    }

    SharedPointer<Appearance> DefaultAssetProvider::GetAppearanceByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if (!id.IsValid()) {
            FEATSTD_LOG_DEBUG("ResolveThemeId is not valid");
        }
        Appearance::SharedPointer result(0);
        if (m_internal == 0) {
            return result;
        }

        switch (id.m_libraryType) {
            case AppearanceLib: result =  m_internal->m_appearanceProvider.Get(id); break;
            case AppearanceCollectionLib: result = m_internal->m_appearanceCollectionProvider.Get(id); break;
            case SceneLib:
            case CompositeLib:
                {
                    AssetId nodeIdentifier = id;
                    nodeIdentifier.m_nodeId = CFFReader::GetItemId(GetParent(GetParent(m_assetGroup->GetLibItemHeader(id))));
                    Node* node = GetNodeByAssetId(nodeIdentifier);
                    if (node != 0) {
                        result = node->GetAppearance();
                        while ((result != 0) && (result->GetId() != AssetIdFunctions::GetNodeId(id))) {
                            result = result->GetNextPass();
                        }
                    } 
                    else{
                        FEATSTD_LOG_WARN("GetNodeByAssetId returns 0.");
                    }
                }
                break;
            default:
                break;
        }

        return result;
    }

    SharedPointer<Material> DefaultAssetProvider::GetMaterialByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if (!id.IsValid()) {
            FEATSTD_LOG_DEBUG("ResolveThemeId is not valid");
            return SharedPointer<Material>();
        }
        if (m_internal == 0) {
            return SharedPointer<Material>();
        }
        if (id.m_libraryType == MaterialLib) {
            return m_internal->m_materialProvider.Get(id);
        }

        AssetId appearanceIdentifier = id;
        appearanceIdentifier.m_nodeId = CFFReader::GetItemId(GetParent(m_assetGroup->GetLibItemHeader(id)));
        Appearance::SharedPointer appearance = GetAppearanceByAssetId(appearanceIdentifier);
        if (appearance != 0) {
            return appearance->GetMaterial();
        }
        else{
            FEATSTD_LOG_WARN("GetAppearanceByAssetId returns a null pointer.");
        }

        return Material::SharedPointer();
    }

    SharedPointer<RenderMode> DefaultAssetProvider::GetRenderModeByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if (!id.IsValid()) {
            FEATSTD_LOG_DEBUG("ResolveThemeId is not valid");
            return SharedPointer<RenderMode>();
        }
        if (m_internal == 0) {
            return SharedPointer<RenderMode>();
        }
        if (id.m_libraryType == RenderModeLib) {
            return m_internal->m_renderModeProvider.Get(id);
        }

        AssetId appearanceIdentifier = id;
        appearanceIdentifier.m_nodeId = CFFReader::GetItemId(GetParent(m_assetGroup->GetLibItemHeader(id)));
        Appearance::SharedPointer appearance = GetAppearanceByAssetId(appearanceIdentifier);
        if (appearance != 0) {
            return appearance->GetRenderMode();
        }
        else{
            FEATSTD_LOG_WARN("GetAppearanceByAssetId returns a null pointer.");
        }

        return RenderMode::SharedPointer();
    }

    SharedPointer<AbstractShaderParamSetter> DefaultAssetProvider::GetShaderParamSetterByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if (!id.IsValid()) {
            FEATSTD_LOG_DEBUG("ResolveThemeId is not valid");
            return SharedPointer<AbstractShaderParamSetter>();
        }
        if (m_internal == 0) {
            return SharedPointer<AbstractShaderParamSetter>();
        }
        if (id.m_libraryType == ShaderParamSetterLib) {
            return m_internal->m_shaderParamSetterProvider.Get(id);
        }

        AssetId appearanceIdentifier = id;
        appearanceIdentifier.m_nodeId = CFFReader::GetItemId(GetParent(m_assetGroup->GetLibItemHeader(id)));
        Appearance::SharedPointer appearance = GetAppearanceByAssetId(appearanceIdentifier);
        if (appearance != 0) {
            return appearance->GetShaderParamSetter();
        }
        else{
            FEATSTD_LOG_WARN("GetAppearanceByAssetId returns a null pointer.");
        }

        return SharedPointer<AbstractShaderParamSetter>();
    }

    SharedPointer<Texture> DefaultAssetProvider::GetTextureByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != TextureLib)) {
            return SharedPointer<Texture>();
        }
        return m_internal->m_textureProvider.Get(id);
    }

    SharedPointer<BitmapTextureImage> DefaultAssetProvider::GetBitmapTextureImageByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != BitmapLib)) {
            return SharedPointer<BitmapTextureImage>();
        }
        return m_internal->m_bitmapTextureImageProvider.Get(id);
    }

    SharedPointer<RenderMode> DefaultAssetProvider::GetDefaultRenderMode()
    {
        const AssetId& invalidAssetId = AssetId::InvalidId();
        CffLoaderContext renderModeContext = {invalidAssetId, this, AssetDataHandle(m_assetGroup->GetDefaultRenderModeHandle()), 0, false, 0};

        SharedPointer<RenderMode> renderMode = AssetBuilder<SharedPointer<RenderMode> >::CreateAndBuildFirstPass(renderModeContext);
        if (!AssetBuilder<SharedPointer<RenderMode> >::BuildSecondPass(renderMode, renderModeContext)) {
            FEATSTD_LOG_WARN("RenderMode BuildSecondPass failed.");
        }

        return renderMode;
    }

    CompositeGroup* DefaultAssetProvider::CreateCompositeGroupByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != CompositeLib)) {
            return 0;
        }
        return m_internal->m_compositeGroupProvider.Get(id);
    }

    void DefaultAssetProvider::ReleaseCompositeGroup(CompositeGroup* compositeGroup)
    {
        CompositeInstanceBuilder::Dispose(compositeGroup);
    }

    Node* DefaultAssetProvider::GetNodeByAssetId(const AssetId& assetId)
    {
        if (m_internal == 0) {
            return 0;
        }
        Node* root = 0;

        switch (assetId.m_libraryType) {
        case SceneLib:
            {
                SceneContext* sceneContext = GetSceneByAssetId(assetId);
                if (sceneContext != 0) {
                    root = sceneContext->GetScene();
                }
                else {
                    FEATSTD_LOG_WARN("GetSceneByAssetId returns 0.");
                }
                break;
            }
        case CompositeLib:
            root = m_internal->m_currentCompositeGroupBuild;
            break;
        default:
            break;
        }

        if (root == 0) {
            return 0;
        }

        if (assetId.m_nodeId == 0) {
            return root;
        }

        AssetDataHandle itemHandle = m_assetGroup->GetLibItemHeader(assetId);
        if (!itemHandle.IsValid()) {
            return 0;
        }

        return GetNodeRecursive(*root, itemHandle);
    }

    Node* DefaultAssetProvider::GetRootNode(AssetObjectType aoType, const Char* name)
    {
        Node* result = 0;
        if (m_internal == 0) {
            return 0;
        }

        switch (aoType) {
        case AotScene:
            {
            Candera::Internal::AssetId resultAssetId = GetAssetId(SceneLib, name);
            if (!resultAssetId.IsValid()) {
                FEATSTD_LOG_DEBUG("GetAssetId is not valid");
            }
            SceneContext* sceneContext = GetSceneByAssetId(resultAssetId);
                if (sceneContext != 0) {
                    result = sceneContext->GetScene();
                }
                else {
                    FEATSTD_LOG_DEBUG("GetSceneByAssetId returns 0.");
                }
            }
            break;
        case AotComposite:
            result = m_internal->m_currentCompositeGroupBuild;
            break;
        case AotReferencedTemplate:
            {
            Candera::Internal::AssetId resultAssetId = GetAssetId(ReferencedTemplateLib, name);
            if (!resultAssetId.IsValid()) {
                FEATSTD_LOG_DEBUG("GetAssetId is not valid");
            }
            result = GetReferencedTemplateByAssetId(resultAssetId);
            if (result == 0){
                FEATSTD_LOG_DEBUG("GetReferencedTemplateByAssetId returns 0.");
            }
            }
            break;
        default:
            break;
        }
        return result;
    }

    bool DefaultAssetProvider::GetCompositePropertyInfo(CompositeGroup* compositeGroup, const Char* propertyName, WidgetBase*& widget, const Char*& widgetPropertyName)
    {
        if (compositeGroup != 0) {
            const CompositePropertyList::Property* property = CompositePropertyList::GetPropertyList(compositeGroup).GetProperty(propertyName);
            if (property != 0) {
                widget = property->widget;
                widgetPropertyName = property->property;
                return true;
            }
        }
        return false;
    }

    SharedClearMode::SharedPointer DefaultAssetProvider::GetClearModeByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != ClearModeLib)) {
            return SharedClearMode::SharedPointer();
        }
        return m_internal->m_clearModeProvider.Get(id);
    }
#else
    SharedClearMode::SharedPointer DefaultAssetProvider::GetClearModeByAssetId(const AssetId& assetId)
    {
        FEATSTD_UNUSED(assetId);
        return SharedClearMode::SharedPointer();
    }
#endif

#ifdef CANDERA_2D_ENABLED

    Scene2DContext* DefaultAssetProvider::GetScene2DByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != Scene2DLib)) {
            return 0;
        }
        // Retrieve asset's address from cache.
        Scene2DContext* sceneContext = m_internal->m_scene2DProvider.GetFromCache(id);

       if (sceneContext == 0) {
            // Asset was not cached before, so cache it now.
            sceneContext = m_internal->m_scene2DProvider.Get(id);

            //CgiApp wants to have access to the composite properties after the scene is loaded
            if (!m_keepCompositeAttachmentsAfterLoad) {
                if (sceneContext != 0) {
                    if (sceneContext->GetScene() != 0) {
                        Composite2DAttachmentClearTraverser().Traverse(*sceneContext->GetScene());
                    }
                    else {
                        // Scene pointer is null.
                    }
                }
                else {
                    // No asset with given ID was found.
                }
            }
            else {
                // Control properties are kept.
            }
        }
        else {
            // Asset was already cached. Don't traverse scene again.
        }

        return sceneContext;
    }

    SharedPointer<BitmapImage2D> DefaultAssetProvider::GetBitmapImage2DByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != BitmapLib)) {
            return SharedPointer<BitmapImage2D>();
        }
        return m_internal->m_bitmapImage2DProvider.Get(id);
    }

    void DefaultAssetProvider::ReleaseScene2DByAssetId(const AssetId& assetId)
    {
        if ((m_internal == 0) || (assetId.m_libraryType != Scene2DLib)) {
            return;
        }
        m_internal->m_scene2DProvider.Release(assetId, true);
    }

    CompositeGroup2D* DefaultAssetProvider::CreateCompositeGroup2DByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != Composite2DLib)) {
            return 0;
        }
        return m_internal->m_compositeGroup2DProvider.Get(id);
    }

    void DefaultAssetProvider::ReleaseCompositeGroup2D(CompositeGroup2D* compositeGroup)
    {
        Composite2DInstanceBuilder::Dispose(compositeGroup);
    }

    static Node2D* GetNode2DRecursive(Node2D& root, const AssetDataHandle& nodeHandle)
    {
        Id nodeId = CDA_NODE_ASSETID(CFFReader::GetItemId(nodeHandle));
        AssetDataHandle parentHandle = GetParent(nodeHandle);
        if (parentHandle.IsValid()) {
            Node2D* parent = GetNode2DRecursive(root, parentHandle);
            if (parent != 0) {
                AssetObjectType nodeType = static_cast<AssetObjectType>(CFFReader::GetItemType(nodeHandle));
                if (nodeType == AotAnchor2D) {
                    CompositeGroup2D* compositeGroup = Dynamic_Cast<CompositeGroup2D*>(parent);
                    if (compositeGroup != 0) {
                        return compositeGroup->GetAnchorPointNode(nodeId);
                    }
                }
                else {
                    return parent->GetChild(nodeId);
                }
            }
        }
        else {
            return root.GetChild(nodeId);
        }

        return 0;
    }

    Node2D* DefaultAssetProvider::GetNode2DByAssetId(const AssetId& assetId)
    {
        if (!assetId.IsValid()) {
            return 0;
        }
        Node2D* root = 0;
        if (m_internal == 0) {
            return 0;
        }

        switch (assetId.m_libraryType) {
        case Scene2DLib:
            {
                Scene2DContext* sceneContext = GetScene2DByAssetId(assetId);
                if (sceneContext != 0) {
                    root = sceneContext->GetScene();
                }
                else{
                    FEATSTD_LOG_WARN("GetScene2DByAssetId returns 0.");
                }
                break;
            }
        case Composite2DLib:
            root = m_internal->m_currentCompositeGroup2DBuild;
            break;
        default:
            break;
        }

        if (root == 0) {
            return 0;
        }

        if (assetId.m_nodeId == 0) {
            return root;
        }

        AssetDataHandle itemHandle = m_assetGroup->GetLibItemHeader(assetId);
        if (!itemHandle.IsValid()) {
            return 0;
        }

        return GetNode2DRecursive(*root, itemHandle);
    }

    SharedPointer<Effect2D> DefaultAssetProvider::GetEffect2DByAssetId(const AssetId& assetId)
    {
        Effect2D* result = 0;
        AssetId nodeIdentifier = assetId;
        AssetDataHandle itemHandle = GetParent(GetParent(m_assetGroup->GetLibItemHeader(assetId)));
        if (itemHandle.IsValid()) {
            nodeIdentifier.m_nodeId = CFFReader::GetItemId(itemHandle);
            Node2D* resultNode = GetNode2DByAssetId(nodeIdentifier);
            if (resultNode == 0) {
                FEATSTD_LOG_DEBUG("GetNode2DByAssetId returns 0.");
            }
            RenderNode* node = Dynamic_Cast<RenderNode*>(resultNode);
            if (node != 0) {
                result = node->GetEffect(0);
                for (UInt8 index = 0; result != 0; result = node->GetEffect(++index)) {
                    if (result->GetId() == AssetIdFunctions::GetNodeId(assetId)) {
                        break;
                    }
                }
            }
        }

        return Effect2D::SharedPointer(result);
    }

    Node2D* DefaultAssetProvider::GetRootNode2D(AssetObjectType aoType, const Char* name)
    {
        Node2D* result = 0;
        if (m_internal == 0) {
            return 0;
        }

        switch (aoType) {
        case AotScene2D:
            {
                Candera::Internal::AssetId resultAssetId = GetAssetId(Scene2DLib, name);
                if (!resultAssetId.IsValid()) {
                    FEATSTD_LOG_DEBUG("GetAssetId is not valid");
                }
                Scene2DContext* sceneContext = GetScene2DByAssetId(resultAssetId);
                if (sceneContext != 0) {
                    result = sceneContext->GetScene();
                }
                else{
                    FEATSTD_LOG_DEBUG("GetScene2DByAssetId returns 0.");
                }
            }
            break;
        case AotComposite2D:
            result = m_internal->m_currentCompositeGroup2DBuild;
            break;
        default:
            break;
        }

        return result;
    }

    bool DefaultAssetProvider::GetComposite2DPropertyInfo(CompositeGroup2D* compositeGroup, const Char* propertyName, WidgetBase*& widget, const Char*& widgetPropertyName)
    {
        if (compositeGroup != 0) {
            const Composite2DPropertyList::Property* property = Composite2DPropertyList::GetPropertyList(compositeGroup).GetProperty(propertyName);
            if (property != 0) {
                widget = property->widget;
                widgetPropertyName = property->property;
                return true;
            }
        }
        return false;
    }
#endif

#ifdef CANDERA_SCRIPTING_ENABLED
    MemoryManagement::SharedPointer<Scripting::Script> DefaultAssetProvider::GetScriptByAssetId(const AssetId& assetId)
    {
        const AssetId& id = ResolveThemeId(assetId);
        if ((m_internal == 0) || (id.m_libraryType != ScriptLib)) {
            return SharedPointer<Scripting::Script>();
        }
        return m_internal->m_scriptProvider.Get(id);
    }
#endif //CANDERA_SCRIPTING_ENABLED

    WidgetBase* DefaultAssetProvider::GetWidgetByItemPath(const ItemPathInfo& itemPathInfo)
    {
        if (m_internal == 0) {
            return 0;
        }
        if (!itemPathInfo.IsPathValid()) {
            return 0;
        }
        if (itemPathInfo.GetReferencedItemType() != AotWidget) {
            return 0;
        }

        AssetObjectType rootType = itemPathInfo.GetRootType();
        const Char* rootName = itemPathInfo.GetRootName();
        const Char* widgetIdentifier = itemPathInfo.GetOriginalPath();

        WidgetBase* result = 0;

        switch (rootType) {
#ifdef CANDERA_3D_ENABLED
        case AotScene: {
            Candera::Internal::AssetId resultAssetId = GetAssetId(SceneLib, rootName);
            if (!resultAssetId.IsValid()) {
                FEATSTD_LOG_DEBUG("GetAssetId is not valid");
            }
            SceneContext* sceneContext = GetSceneByAssetId(resultAssetId);
            if (sceneContext != 0) {
                result = sceneContext->GetWidgetByIdentifer(widgetIdentifier);
            }
            else {
                FEATSTD_LOG_DEBUG("GetSceneByAssetId returns 0.");
            }
            break;
                       }
        case AotComposite: {
            if (m_internal->m_currentCompositeGroupBuild != 0) {
                result = m_internal->m_currentCompositeGroupBuild->GetWidgetByName(widgetIdentifier);
            }
            break;
        }
#endif //CANDERA_3D_ENABLED
#ifdef CANDERA_2D_ENABLED
        case AotScene2D: {
            Candera::Internal::AssetId resultAssetId = GetAssetId(Scene2DLib, rootName);
            if (!resultAssetId.IsValid()) {
                FEATSTD_LOG_DEBUG("GetAssetId is not valid");
            }
            Scene2DContext* sceneContext = GetScene2DByAssetId(resultAssetId);
            if (sceneContext != 0) {
                result = sceneContext->GetWidgetByIdentifer(widgetIdentifier);
            }
            else{
                FEATSTD_LOG_DEBUG("GetScene2DByAssetId returns 0.");
            }
            break;
                         }
        case AotComposite2D: {
            if (m_internal->m_currentCompositeGroup2DBuild != 0) {
                result = m_internal->m_currentCompositeGroup2DBuild->GetWidgetByName(widgetIdentifier);
            }
            break;
        }
#endif //CANDERA_2D_ENABLED
        default: result = 0;
        }

        return result;
    }

    AssetId DefaultAssetProvider::GetAssetId(AssetLib assetLib, const Char* name)
    {
        for (AssetGroup::AssetSetIterator it = m_assetGroup->GetAssetSetIterator(); it.IsValid(); ++it) {
            AssetDataHandle headerHandle((*it)->GetAssetLibHeaderHandle(assetLib));
            if (headerHandle.IsValid()) {
                Int32 elementCount = CFFReader::GetLibElementCount(headerHandle);
                for (Int32 index = 0; index < elementCount; ++index) {
                    AssetDataHandle elementHandle = CFFReader::GetLibElement(headerHandle, index);
                    UInt32 elementNameOffset = CFFReader::GetLibElementName(elementHandle);
                    const Char* elementName = (*it)->GetName(elementNameOffset);
                    if (elementName != 0) {
                        if (StringPlatform::CompareStrings(elementName, name) == 0) {
                            const AssetId& assetId = AssetIdFunctions::GetAssetId(CFFReader::GetLibElementId(elementHandle));
                            FEATSTD_LOG_DEBUG("Id requested for asset library type %d with name %s. Library object found in AssetSet with id:%d, name:(%d)%s, id " AssetIdLogStr, assetLib, name, (*it)->GetId(), elementNameOffset, elementName, AssetIdLogArgs(assetId));
                            return assetId;
                        }
                    }
                }
            }
        }

        return AssetId::InvalidId();
    }

    Id DefaultAssetProvider::GetId(AssetLib assetLib, const Char* name)
    {
        return AssetIdFunctions::GetLibraryId(GetAssetId(assetLib, name));
    }

    const Char* DefaultAssetProvider::GetNameByAssetId(const AssetId& assetId) const
    {
        if (assetId.m_libraryType >= c_libCount) {
            return 0;
        }

        if (assetId.m_nodeId != 0) {
            ResourceDataHandle resourceHandle = m_assetGroup->GetItemHeaderHandle(assetId);
            UInt16 repositoryId = resourceHandle.m_accessor.m_asset.m_repositoryId;
            AssetDataHandle elementHandle = m_assetGroup->GetLibItemHeader(assetId);
            if (elementHandle.IsValid()) {
                for (AssetGroup::AssetSetIterator it = m_assetGroup->GetAssetSetIterator(); it.IsValid(); ++it) {
                    if ((*it)->GetId() == repositoryId) {
                        return (*it)->GetName(CFFReader::GetItemName(elementHandle));
                    }
                }
            }
        }
        else {
            if (m_assetGroup != 0) {
                for (AssetGroup::AssetSetIterator it = m_assetGroup->GetAssetSetIterator(); it.IsValid(); ++it) {
                    AssetDataHandle elementHandle = (*it)->GetAssetLibElementHandle(assetId);
                    if (elementHandle.IsValid()) {
                        return (*it)->GetName(CFFReader::GetLibElementName(elementHandle));
                    }
                }
            }
        }

        return 0;
    }

    const Char* DefaultAssetProvider::GetNameById(AssetLib assetLib, Id libraryId, Id nodeId) const
    {
        return GetNameByAssetId(AssetIdFunctions::CreateId(assetLib, AotUnknown, libraryId, nodeId));
    }

namespace {
    template<typename SharedPointer, typename AssetProvider>
    class OrphanFinder
    {
        struct Visitor : public AssetProvider::Visitor
        {
            virtual bool Visit(SharedPointer& value) override
            {
                return (!FeatStd::MemoryManagement::Internal::IsSoleReference(value));
            }
        };
    public:
        static void RemoveAll(AssetProvider& provider)
        {
            Visitor visitor;
            provider.Filter(visitor);
        }
    };

    template <template <typename T> class AssetProvider, typename SharedPointer>
    void RemoveAllOrphans(AssetProvider<SharedPointer>& assetProvider)
    {
        OrphanFinder<SharedPointer, AssetProvider<SharedPointer> >::RemoveAll(assetProvider);
    }

    template <template <typename T, typename B> class AssetProvider, typename SharedPointer, typename AssetBuilder>
    void RemoveAllOrphans(AssetProvider<SharedPointer, AssetBuilder>& assetProvider)
    {
        OrphanFinder<SharedPointer, AssetProvider<SharedPointer, AssetBuilder> >::RemoveAll(assetProvider);
    }
}

    void DefaultAssetProvider::ReleaseOrphanedObjects()
    {
        if (m_internal == 0) {
            return;
        }
#ifdef CANDERA_2D_ENABLED
        RemoveAllOrphans(m_internal->m_bitmapImage2DProvider);
#endif

#ifdef CANDERA_3D_ENABLED
        RemoveAllOrphans(m_internal->m_shaderProvider);
        RemoveAllOrphans(m_internal->m_appearanceProvider);
        RemoveAllOrphans(m_internal->m_appearanceCollectionProvider);
        RemoveAllOrphans(m_internal->m_renderModeProvider);
        RemoveAllOrphans(m_internal->m_materialProvider);
        RemoveAllOrphans(m_internal->m_textureProvider);
        RemoveAllOrphans(m_internal->m_shaderParamSetterProvider);
        RemoveAllOrphans(m_internal->m_vertexBufferProvider);
        RemoveAllOrphans(m_internal->m_bitmapTextureImageProvider);
        RemoveAllOrphans(m_internal->m_clearModeProvider);
#endif //CANDERA_3D_ENABLED

#ifdef CANDERA_SCRIPTING_ENABLED
        RemoveAllOrphans(m_internal->m_scriptProvider);
#endif
        RemoveAllOrphans(m_internal->m_bitmapProvider);
        RemoveAllOrphans(m_internal->m_animationPlayerProvider);
        RemoveAllOrphans(m_internal->m_animationGroupPlayerProvider);
        RemoveAllOrphans(m_internal->m_styleProvider);
        RemoveAllOrphans(m_internal->m_cultureProvider);
    }

}
