//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if  !defined(CANDERA_CONTENTLOADER_H)
#define CANDERA_CONTENTLOADER_H

#include <Candera/Environment.h>
#include <FeatStd/Async/AsyncRequest.h>
#include <Candera/System/Container/Map.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetProvider.h>

namespace Candera {

/** @addtogroup AssetLoaderBase
 *  @{
 */
    // Forward declarations
    class Scene2DContext;
    class SceneContext;
    class SceneContextBase;

    /**
     * @brief Encapsulates routines for scene content handling
     * 
     * Usage example:
     *
     *  // Declarations
     *  DefaultAssetProvider* m_assetProvider = &DefaultAssetProvider::GetInstance();
     *  ContentLoader* m_contentLoader = &ContentLoader::GetInstance();
     *  
     *  // Initializations: 
     *  const Char* sceneName = "scene";
     *  const Char* filePath = "D:\\compactAssetLib.bin"; // place the desired filePath
     *  // Initialize the asset provider with the specified asset lib compact file
     *  m_assetProvider->Initialize(filePath);
     *  // Associate an asset provider to the content loader:
     *  m_contentLoader->Init(m_assetProvider);
     *  
     *  // Incremental context build:
     *  ContentLoader::BuildState buildState = m_contentLoader->BuildSceneContext(sceneName, ContentLoader::DefaultUploadPolicy);
     *  while (buildState == ContentLoader::MorePackets){
     *       buildState = m_contentLoader->BuildSceneContext(sceneName, ContentLoader::DefaultUploadPolicy);
     *  }
     *  
     *  if (buildState == ContentLoader::Completed) {
     *      // We can retrieve the scene context in order to operate on it
     *      SceneContext* m_sceneContext = m_contentLoader->GetSceneContext(sceneName);
     *       // ...
     *  } else {
     *      // ERROR
     *  }
     * 
     */
    class ContentLoader {
        public:
            ~ContentLoader();

            /**
             * Whether scene should be created in System- and VRAM or only in System RAM.
             * DefaultUploadPolicy - creates scene data in System RAM and uploads the asset data to VRAM
             * NoVramUploadPolicy - creates scene data in System RAM but does not upload anything to VRAM
             */
            enum UploadPolicy {
                DefaultUploadPolicy, ///< creates scene data in System RAM and uploads the asset data to VRAM
                NoVramUploadPolicy  ///< creates scene data in System RAM but does not upload anything to VRAM
            };

            /**
             * The Build State (MorePackets, Completed, Error)
             */
            enum BuildState {
                MorePackets,    ///< MorePackets
                Incomplete,     ///< Incomplete
                Completed,      ///< Completed
                Error           ///< Error
            };

            typedef FeatStd::AsyncRequest<BuildState> AsyncBuildState;

            /**
             * Returns the static (global) instance
             * @return The static (global) instance of ContentLoader.
             */
            static ContentLoader& GetInstance(); 

            /**
             * Associates an asset provider to the content loader and performs
             * initialization operations
             * @param provider AssetProvider to be associated to the content loader.
             */
            void Init(AssetProvider* provider);

            /**
             * Returns the asset provider associated to the content loader
             * @return Pointer to the asset provider
             */
            AssetProvider* GetAssetProvider() { return m_assetProvider; }

#ifdef CANDERA_3D_ENABLED
            /**
             * Builds a scene context on a certain upload policy
             * Depending on the upload policy, the build might take more than
             * one step to succeed ( incremental build step )
             * @param sceneId The id (or symbolic name) of the scene to build.
             * @param policy The upload policy to use during building
             * @return One of the BuildState values, indicating whether the build
             * failed, the build completed or more packets are needed.
             */
            BuildState BuildSceneContextById(Id sceneId, UploadPolicy policy = DefaultUploadPolicy);

            /**
             * Schedules an asynchronous scene build.
             *
             * The scene will be loaded when the returned request will be dispatched. In contrast to the
             * synchronuous method, loading the scene asynchronously will not involve the upload of the
             * DeviceObjects into VRAM by default. The default behavior can be changed by using the UploadPolicy
             * parameter, but it has to be taken into consideration that if a worker thread for asset loading
             * exits, then Candera will ignore the flag and throw a warning. Uploading the DeviceObjects
             * must be done in the render thread.
             *
             * @param sceneId The id (or symbolic name) of the scene of which the build is scheduled.
             * @param policy The upload policy to use during building.
             * @return An AsyncRequest that can be dispatched and queried for its status and result.
             * @see BuildSceneContextById
             */
            AsyncBuildState::SharedPointer BuildSceneContextAsync(Id sceneId, UploadPolicy policy = NoVramUploadPolicy);

            /**
             * Returns a scene context with the specified id
             * @param id The Id of the scene to retrieve
             * @return The scene context for this scene.
             */
            SceneContext* GetSceneContextById(Id id);

            /**
             * Releases the specified scene context, i.e. remove it 
             * from the internal list and delete the associated scene from the cache.
             * @param sceneContext The scene context to release
             */
            void ReleaseSceneContext(SceneContext*& sceneContext);

            /**
             * Releases all the scene context
             */
            void ReleaseAllSceneContexts();
            
#endif //CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
            /**
             * Builds a 2D scene context on a certain upload policy
             * Depending on the upload policy, the build might take more than
             * one step to succeed ( incremental build step )
             * @param sceneId The id (or symbolic name) of the scene to build
             * @param policy The upload policy to use during building
             * @return One of the BuildState values, indicating whether the build
             * failed, the build completed or more packets are needed.
             */
            BuildState BuildScene2DContextById(Id sceneId, UploadPolicy policy = DefaultUploadPolicy);

            /**
             * Schedules an asynchronous 2D scene build.
             *
             * The scene will be loaded when the returned request will be dispatched. In contrast to the
             * synchronuous method, loading the scene asynchronously will not involve the upload of the
             * DeviceObjects into VRAM by default. The default behavior can be changed by using the UploadPolicy
             * parameter, but it has to be taken into consideration that if a worker thread for asset loading
             * exits, then Candera will ignore the flag and throw a warning. Uploading the DeviceObjects
             * must be done in the render thread.
             *
             * @param sceneId The id (or symbolic name) of the scene of which the build is scheduled.
             * @param policy The upload policy to use during building.
             * @return An AsyncRequest that can be dispatched and queried for its status and result.
             * @see BuildSceneContextById
             */
            AsyncBuildState::SharedPointer BuildScene2DContextAsync(Id sceneId, UploadPolicy policy = NoVramUploadPolicy);

            /**
             * Returns a 2D scene context with the specified id
             * @param id The id of the scene to retrieve
             * @return The scene context for this scene.
             */
            Scene2DContext* GetScene2DContextById(Id id);

            /**
             * Releases the specified 2D scene context, i.e. remove it 
             * from the internal list and delete the associated scene from the cache.
             * @param sceneContext The scene context to release
             */
            void ReleaseScene2DContext(Scene2DContext*& sceneContext);

            /**
             * Releases all the 2D scene contexts
             */
            void ReleaseAllScene2DContexts();
#endif //

    private:

        BuildState LoadChunk(SceneContextBase& sceneContext) const;

        ContentLoader();
#ifdef CANDERA_3D_ENABLED
        typedef Candera::Internal::Vector<SceneContext*> SceneContextList;
        SceneContextList m_sceneContexts;
#endif //CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
        typedef Candera::Internal::Vector<Scene2DContext*> Scene2DContextList;
        Scene2DContextList m_scene2DContexts;
#endif //CANDERA_2D_ENABLED

        AssetProvider* m_assetProvider;

    };
 
 /** @} */ // end of AssetLoaderBase
 
}

#endif //CANDERA_CONTENTLOADER_H

