//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ContentLoader.h"
#include <FeatStd/Util/StaticObject.h>
#include <FeatStd/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/ReloadEntry.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/SceneContextBase.h>
#include <CanderaWidget/WidgetBase/WidgetBase.h>

#ifdef CANDERA_3D_ENABLED
#include <CanderaAssetLoader/AssetLoader3D/SceneContext.h>
#include <Candera/Engine3D/Core/Scene.h>
#endif //CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
#include <CanderaAssetLoader/AssetLoader2D/Scene2DContext.h>
#include <Candera/Engine2D/Core/Scene2D.h>
#endif

namespace Candera {
    using namespace Internal;
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

    ContentLoader::ContentLoader(void) :
        m_assetProvider(0)
    {
    }

    ContentLoader::~ContentLoader(void)
    {
        m_assetProvider = 0;
    }

    void ContentLoader::Init(AssetProvider* provider)
    {
        m_assetProvider = provider;
    }

    ContentLoader& ContentLoader::GetInstance()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(ContentLoader, contentLoader);
        return contentLoader;
    }
#ifdef CANDERA_3D_ENABLED

    ContentLoader::BuildState ContentLoader::BuildSceneContextById(Id sceneId, UploadPolicy policy)
    {
        if (m_assetProvider == 0) {
            return Error;
        }

        BuildState buildState = Completed;
        SceneContext* sceneContext = GetSceneContextById(sceneId);
        if (sceneContext != 0) {
            buildState = LoadChunk(*sceneContext);
            if (buildState == MorePackets) {
                return MorePackets;
            }
        }
        else {
            sceneContext = m_assetProvider->GetSceneById(sceneId);

            if (sceneContext == 0) {
                return Error;
            }

            static_cast<void>(m_sceneContexts.Add(sceneContext));
            sceneContext->SetUploadPolicy(policy);
            return MorePackets;
        }
        if (policy == DefaultUploadPolicy) {
            if ((sceneContext->GetScene() == 0) || (!sceneContext->GetScene()->UploadAll(ScopeMask(), true))) {
                return Error;
            }
        }

        return buildState;
    }

    SceneContext* ContentLoader::GetSceneContextById(Id id)
    {
        for (SizeType i = 0; i < m_sceneContexts.Size(); ++i) {
            SceneContext* sceneContext = m_sceneContexts[i];
            if ((sceneContext != 0) && (sceneContext->GetScene() != 0) && (sceneContext->GetScene()->GetId() == id)) {
                return sceneContext;
            }
        }

        return 0;
    }

    void ContentLoader::ReleaseAllSceneContexts()
    {
        if (m_assetProvider == 0) {
            return;
        }

        for (SizeType i = 0; i < m_sceneContexts.Size(); ++i) {
            SceneContext* sceneContext = m_sceneContexts[i];
            if ((sceneContext != 0) && (sceneContext->GetScene() != 0)) {
                if (sceneContext->GetUploadPolicy() == DefaultUploadPolicy) {
                    static_cast<void>(sceneContext->GetScene()->UnloadAll());
                }

                m_assetProvider->ReleaseSceneById(sceneContext->GetScene()->GetId());
            }
        }

        m_sceneContexts.Free();
    }

    class SceneBuilderAsyncRequest : public ContentLoader::AsyncBuildState
    {
    public:
        SceneBuilderAsyncRequest(Id id, ContentLoader::UploadPolicy policy): m_id(id), m_policy(policy)
        {
        }
    private:
        virtual void Execute() override
        {
            SetResult(ContentLoader::GetInstance().BuildSceneContextById(m_id, m_policy));
        }

        Id m_id;
        ContentLoader::UploadPolicy m_policy;
    };

    ContentLoader::AsyncBuildState::SharedPointer ContentLoader::BuildSceneContextAsync(Id sceneId, ContentLoader::UploadPolicy policy)
    {
        ContentLoader::AsyncBuildState::SharedPointer result(FEATSTD_NEW(SceneBuilderAsyncRequest)(sceneId, policy));
        if (result != 0) {
            GetAssetProvider()->AsyncProxy().m_dispatcher.GetRequestDispatcher().Schedule(result);
        }
        return result;
    }

    void ContentLoader::ReleaseSceneContext(SceneContext*& sceneContext)
    {
        if ((m_assetProvider == 0) || (sceneContext == 0) || (sceneContext->GetScene() == 0))
        {
            return;
        }

        for (SizeType index = 0; index < m_sceneContexts.Size(); ++index) {
            if (m_sceneContexts[index] == sceneContext) {
                static_cast<void>(m_sceneContexts.Remove(index));
                if (0 == m_sceneContexts.Size()) {
                    m_sceneContexts.Free();
                }
                break;
            }
        }

        if (sceneContext->GetUploadPolicy() == DefaultUploadPolicy) {
            static_cast<void>(sceneContext->GetScene()->UnloadAll());
        }

        m_assetProvider->ReleaseSceneById(sceneContext->GetScene()->GetId());

        sceneContext = 0;
    }

#endif //CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED

    ContentLoader::BuildState ContentLoader::BuildScene2DContextById(Id sceneId, UploadPolicy policy)
    {
        if (m_assetProvider == 0) {
            return Error;
        }

        BuildState buildState = Completed;
        Scene2DContext* sceneContext = GetScene2DContextById(sceneId);
        if (sceneContext != 0) {
            buildState = LoadChunk(*sceneContext);
            if (buildState == MorePackets) {
                return MorePackets;
            }
        }
        else {
            sceneContext = m_assetProvider->GetScene2DById(sceneId);

            if (sceneContext == 0) {
                return Error;
            }

            static_cast<void>(m_scene2DContexts.Add(sceneContext));
            sceneContext->SetUploadPolicy(policy);
            return MorePackets;
        }
        if (policy == DefaultUploadPolicy) {
            if ((sceneContext->GetScene() == 0) || (!sceneContext->GetScene()->Upload(ScopeMask(), Node2D::Deep))) {
                return Error;
            }
        }

        return buildState;
    }
    
    class Scene2DBuilderAsyncRequest : public ContentLoader::AsyncBuildState
    {
    public:
        Scene2DBuilderAsyncRequest(Id id, ContentLoader::UploadPolicy policy): m_id(id), m_policy(policy)
        {
        }
    private:
        virtual void Execute() override
        {
            SetResult(ContentLoader::GetInstance().BuildScene2DContextById(m_id, m_policy));
        }

        Id m_id;
        ContentLoader::UploadPolicy m_policy;
    };

    ContentLoader::AsyncBuildState::SharedPointer ContentLoader::BuildScene2DContextAsync(Id sceneId, ContentLoader::UploadPolicy policy)
    {
        ContentLoader::AsyncBuildState::SharedPointer result(FEATSTD_NEW(Scene2DBuilderAsyncRequest)(sceneId, policy));
        if (result != 0) {
            GetAssetProvider()->AsyncProxy().m_dispatcher.GetRequestDispatcher().Schedule(result);
        }
        return result;
    }

    Scene2DContext* ContentLoader::GetScene2DContextById(Id id)
    {
        for (SizeType i = 0; i < m_scene2DContexts.Size(); ++i) {
            Scene2DContext* sceneContext = m_scene2DContexts[i];
            if ((sceneContext != 0) && (sceneContext->GetScene() != 0) && (sceneContext->GetScene()->GetId() == id)) {
                return sceneContext;
            }
        }

        return 0;
    }

    void ContentLoader::ReleaseAllScene2DContexts()
    {
        if (m_assetProvider == 0) {
            return;
        }

        for (SizeType i = 0; i < m_scene2DContexts.Size(); ++i) {
            Scene2DContext* sceneContext = m_scene2DContexts[i];
            if ((sceneContext != 0) && (sceneContext->GetScene() != 0)) {
                if (sceneContext->GetUploadPolicy() == DefaultUploadPolicy) {
                    static_cast<void>(sceneContext->GetScene()->Unload(ScopeMask(), Node2D::Deep));
                }

                m_assetProvider->ReleaseScene2DById(sceneContext->GetScene()->GetId());
            }
        }

        m_scene2DContexts.Free();
    }

    void ContentLoader::ReleaseScene2DContext(Scene2DContext*& sceneContext)
    {
        if ((m_assetProvider == 0) || (sceneContext == 0) || (sceneContext->GetScene() == 0)) {
            return;
        }


        for (SizeType index = 0; index < m_scene2DContexts.Size(); ++index) {
            if (m_scene2DContexts[index] == sceneContext) {
                static_cast<void>(m_scene2DContexts.Remove(index));
                if (0 == m_scene2DContexts.Size()) {
                    m_scene2DContexts.Free();
                }
                break;
            }
        }

        if (sceneContext->GetUploadPolicy() == DefaultUploadPolicy) {
            static_cast<void>(sceneContext->GetScene()->Unload(ScopeMask(), Node2D::Deep));
        }

        m_assetProvider->ReleaseScene2DById(sceneContext->GetScene()->GetId());

        sceneContext = 0;
    }

#endif //CANDERA_2D_ENABLED

    ContentLoader::BuildState ContentLoader::LoadChunk(SceneContextBase& sceneContext) const
    {
        //retrieve reload list form SceneContext
        DependencyList& reloadList = sceneContext.m_reloadList;

        //last unresolved reload entry
        DependencyList::Iterator& firstEntry = sceneContext.m_currentEntry;

        //start over if end of reload entry list is reached
        if (firstEntry == reloadList.End()) {
            firstEntry = reloadList.Begin();
        }

        //if first unresolved reload entry is still the end of the list, it 
        // means that it reload is complete
        if (firstEntry == reloadList.End()) {
            return Completed;
        }

        if (m_assetProvider == 0) {
            return Error;
        }

        //advance entry iterator while current reload entry cannot be loaded
        DependencyList::Iterator currentEntry = firstEntry;
        while (!(*currentEntry)->Load(&reloadList)) {
            currentEntry++;
            if (currentEntry == reloadList.End()) {
                currentEntry = reloadList.Begin();
            }

            //if current reload entry is the one that started the iteration
            // it means none of the reload entries could be loaded
            if (currentEntry == firstEntry) {
                return Incomplete;
            }
        }

        //remove current entry (loaded) from the dependency list.
        firstEntry = currentEntry;
        firstEntry++;
        if (!reloadList.Remove(*currentEntry)) {
            return Error;
        }

        return MorePackets;
    }
}

