//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AssetValidation.h"

#include <Candera/CanderaVersion.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageDescriptor.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetRepository.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetSet.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AssetHeaderCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/CffAssetReaderVersion.h>
#include <CanderaWidget/WidgetBase/WidgetMetaInfo.h>
#include <Candera/EngineBase/Common/ResourceObject.h>

namespace Candera {
    using namespace Internal;
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

    template<AssetValidationAttributes attribute>
    bool AssetValidation::ValidateAssetData(const AssetData& assetData, UInt32 validationFlags)
    {
        //if item validation is not ignored
        if ((validationFlags & AssetValidationFlag<attribute, IgnoreLevel>::Flag) == 0) {
            //if validation fail
            if (!Validate<attribute>(assetData)) {
                //if failed validation must trigger an error
                if ((validationFlags & AssetValidationFlag<attribute, ErrorLevel>::Flag) != 0) {
                    return false;
                }
            }
        }

        return true;
    }

    FEATSTD_LINT_CURRENT_SCOPE(1576, "Violates MISRA C++ 2008 Required Rule 14-7-3: implementation in cpp to use logging")
    template<>
    bool AssetValidation::Validate<WidgetSetHashAttribute>(const AssetData& assetData)
    {
        if ((GetWidgetSet() == 0) || (GetWidgetSet()->GetHash() != assetData.m_versionInfo.m_widgetHash)) {
            FEATSTD_LOG_WARN("Asset generated with different WidgetSet (hash comparison failed).");
            return false;
        }

        return true;
    }

    template<>
    bool AssetValidation::Validate<CanderaVersionAttribute>(const AssetData& assetData)
    {
        if ((assetData.m_versionInfo.m_canderaVersion[0] != CANDERA_VERSION_MAJOR) ||
            (assetData.m_versionInfo.m_canderaVersion[1] != CANDERA_VERSION_MINOR) ||
            (assetData.m_versionInfo.m_canderaVersion[2] != CANDERA_VERSION_PATCH) ||
            (assetData.m_versionInfo.m_canderaVersion[3] != CANDERA_VERSION_TWEAK)) {
                FEATSTD_LOG_WARN("Asset generated for different Candera version - %d.%d.%d.%d", assetData.m_versionInfo.m_canderaVersion[0], assetData.m_versionInfo.m_canderaVersion[1], assetData.m_versionInfo.m_canderaVersion[2], assetData.m_versionInfo.m_canderaVersion[3]);
                return false;
        }

        return true;
    }

    template<>
    bool AssetValidation::Validate<WidgetSetVersionAttribute>(const AssetData& assetData)
    {
        if (GetWidgetSet() == 0) {
            FEATSTD_LOG_WARN("There is no widget set.");
            return false;
        }
        else if (GetWidgetSet()->GetVersion() != assetData.m_versionInfo.m_widgetSetVersion) {
            FEATSTD_LOG_WARN("Asset generated with different WidgetSet version - %d. Current WidgetSet version - %d", assetData.m_versionInfo.m_widgetSetVersion, GetWidgetSet()->GetVersion());
            return false;
        }
        else {
            //do nothing
        }

        return true;
    }

    template<>
    bool AssetValidation::Validate<FractionalNumberRepresentationAttribute>(const AssetData& assetData)
    {
        bool wrongFractionalNumberType =
            (assetData.m_fractionalNumberRepresentation == 0) ||
            (StringPlatform::CompareStrings(assetData.m_fractionalNumberRepresentation, FEATSTD_FRACTIONAL_NUMBER_TYPE_NAME) != 0);
        if (wrongFractionalNumberType) {
            FEATSTD_LOG_WARN(
                "CFF fractional number format (%s) not supported. Current supported version: %s.\n",
                assetData.m_fractionalNumberRepresentation,
                FEATSTD_FRACTIONAL_NUMBER_TYPE_NAME);
            return false;
        }

        return true;
    }

    template<>
    bool AssetValidation::Validate<PlatformNameAttribute>(const AssetData& assetData)
    {
        bool differentPlatform =
            (assetData.m_platformName == 0) ||
            (StringPlatform::CompareStrings(assetData.m_platformName, DevicePackageDescriptor::GetGraphicPackageName()) != 0);
        if (differentPlatform) {
            FEATSTD_LOG_WARN("Asset generated for different platform (%s)", assetData.m_platformName);
            return false;
        }

        return true;
    }

    template<>
    bool AssetValidation::Validate<PlatformInstanceNameAttribute>(const AssetData& assetData)
    {
        bool differentPlatformInstance =
            (assetData.m_platformInstanceName == 0) ||
            (StringPlatform::CompareStrings(assetData.m_platformInstanceName, DevicePackageDescriptor::GetGraphicPackageInstanceName()) != 0);
        if (differentPlatformInstance) {
            FEATSTD_LOG_WARN("Asset generated for different platform instance (%s)", assetData.m_platformInstanceName);
            return false;
        }

        return true;
    }

    template<>
    bool AssetValidation::Validate<CffVersionAttribute>(const AssetData& assetData)
    {
        if (assetData.m_versionInfo.m_fileVersion != CURRENT_VERSION) {
            FEATSTD_LOG_ERROR("CFF file version (%d) not supported. Current supported version: %d.", assetData.m_versionInfo.m_fileVersion, CURRENT_VERSION);
            return false;
        }

        return true;
    }


    bool AssetValidation::GetAssetVersionInfo(AssetRepository* assetRepository, AssetData::AssetVersionInfo& assetVersionInfo)
    {
        if (assetRepository == 0) {
            return false;
        }

        if (!assetRepository->Initialize()) {
            return false;
        }

        bool result = false;
        CFF_BYTE versionBuffer[CFFReader::CFF_ASSET_VERSION_SIZE];
        if (static_cast<Int32>(CFFReader::CFF_ASSET_VERSION_SIZE) == assetRepository->AssetRead(versionBuffer, SizeType(1), CFFReader::CFF_ASSET_VERSION_SIZE)) {
            result = AssetSet::InitializeAssetVersionInfo(AssetDataHandle(ResourceObject<const void>::CreateHandle(versionBuffer, 0, CFFReader::CFF_ASSET_VERSION_SIZE)), assetVersionInfo);
        }
        static_cast<void>(assetRepository->Finalize());
        return result;
    }

    bool AssetValidation::ValidateAssetVersion(const AssetData& assetData, UInt32 validationFlags)
    {
        FEATSTD_COMPILETIME_ASSERT(AssetValidation::CURRENT_VERSION == CFFReader::s_version);
        bool result = ValidateAssetData<CffVersionAttribute>(assetData, validationFlags);

        if (assetData.m_versionInfo.m_canderaVersion[0] != 0) {
            result = ValidateAssetData<WidgetSetVersionAttribute>(assetData, validationFlags) && result;
            result = ValidateAssetData<WidgetSetHashAttribute>(assetData, validationFlags) && result;
            result = ValidateAssetData<CanderaVersionAttribute>(assetData, validationFlags) && result;
        }
        else {
            FEATSTD_LOG_INFO("File does not contain AssetValidationInfo");
        }

        return result;
    }

    bool AssetValidation::ValidateAssetPlatform(const AssetData& assetData, UInt32 validationFlags)
    {
        bool result = ValidateAssetData<FractionalNumberRepresentationAttribute>(assetData, validationFlags);
        result = ValidateAssetData<PlatformNameAttribute>(assetData, validationFlags) && result;
        result = ValidateAssetData<PlatformInstanceNameAttribute>(assetData, validationFlags) && result;

        return result;
    }
}
