//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ASSET_PARTITION_H)
    #define CANDERA_ASSET_PARTITION_H

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDataHandle.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetId.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetRepository.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLoaderMemoryPool.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLibHeadInformation.h>
#include <Candera/EngineBase/Common/ResourceDataHandle.h>

namespace Candera {
    namespace Internal {

        /**
         * @brief Wraps an AssetRepository to provide asset data access abstraction.
         */
        class AssetSet {
        public:
            typedef MemoryManagement::SharedPointer<AssetSet> SharedPointer;
            /**
             * @brief Create an AssetSet object.
             */
            static FeatStd::MemoryManagement::SharedPointer<AssetSet> Create(AssetRepository& assetRepository);

            /**
             * @brief Destructor.
             */
            ~AssetSet();

            /**
             * @brief Initializer.
             *
             * Prepare the AssetSet for future asset libraries accesses.
             * @param reservedIds contains the list of asset set ids (partition ids) which are in use so far
             * and which cannot be subsequently used. Initialize will assign a different id if the one from the
             * asset is found in reservedIds
             */
            bool Initialize(UInt32 validationFlags, FeatStd::Internal::Vector<UInt16>& reservedIds);

            /**
             * @brief Load asset data.
             *
             * @param libHandle Asset handle to be initialized.
             * @param offset Offset in asset from where to get the data.
             *   Use -1 to specify current position.
             * @param size Size of data to be retrieved from the asset.
             *   Use -1 to specify that the first bytes in asset represent
             *   data size.
             * @return true if load was successful, false otherwise.
             */
            ResourceDataHandle CreateAssetHandle(OffsetType offset = FeatStd::OFFSETTYPE_MAX, Int32 count = -1, Int32 elementSize = 1);

            ResourceDataHandle CreateAssetItemDataHandle(const AssetId& id);
            ResourceDataHandle CreateAssetItemHeaderHandle(const AssetId& id);
            ResourceDataHandle CreateAssetItemHeaderHandle(OffsetType offset);
            ResourceDataHandle GetAssetLibHeaderHandle(AssetLib assetLib) const { return m_libHeaderResourceHandles[assetLib]; }
            AssetDataHandle GetAssetLibElementHandle(const AssetId& id) const;

            SizeType ReadData(void* buffer, OffsetType offset, SizeType count);
            const void* GetAddress(OffsetType offset, SizeType count) const;

            const Char* GetName(UInt32 offset) const;
            bool IsWrapping(const AssetRepository* repository) const { return &m_assetRepository == repository; }
            static bool InitializeAssetVersionInfo(const AssetDataHandle& headerDataHandle, AssetData::AssetVersionInfo& versionInfo);
            UInt16 GetId() const { return m_id; }

            private:
            Candera::AssetRepository& m_assetRepository;
            UInt32 m_dataOffset;
            ResourceDataHandle m_libHeaderResourceHandles[c_libCount];
            UInt16 m_id;
            AssetDataHandle m_nameTableHandle;

            AssetSet(AssetRepository& assetRepository);
            bool IsPersistent() { return m_assetRepository.GetAssetAccessMode() == AssetRepository::ReadOnlyAddressable; }
            CANDERA_SHARED_POINTER_DECLARATION();
            FEATSTD_MAKE_CLASS_UNCOPYABLE(AssetSet);
        };
    }
}   // namespace Candera

#endif  // CANDERA_ASSET_PARTITION_H
