//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AssetSet.h"

#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Mathematics/Math.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLibEntities.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetRepository.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetValidation.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/LibraryCollectionInfoCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/ItemHeaderCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/LibraryHeaderCffReader.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>
#include <FeatStd/Platform/Memory.h>
#include <FeatStd/Util/NumericUtil.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace FeatStd::MemoryManagement;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        static AssetLib GetAssetLibType(AssetObjectType objectType) {
            AssetLib returnVal = c_libCount;
            switch (objectType){
                case AotAnimation              :returnVal = AnimationLib; break;
                case AotAnimationGroup         :returnVal = AnimationGroupLib; break;
                case AotBitmap                 :returnVal = BitmapLib; break;
                case AotCameraGroup            :returnVal = CameraGroupLib; break;
                case AotClearMode              :returnVal = ClearModeLib; break;
                case AotDisplay                :returnVal = DisplayLib; break;
                case AotFont                   :returnVal = FontResourceLib; break;
                case AotRawResource            :returnVal = RawResourceLib; break;
                case AotReferencedTemplate     :returnVal = ReferencedTemplateLib; break;
                case AotGraphicDeviceUnit      :returnVal = RenderTargetLib; break;
                case AotScene2D                :returnVal = Scene2DLib; break;
                case AotScene                  :returnVal = SceneLib; break;
                case AotShaderProgram          :returnVal = ShaderLib; break;
                case AotComposite2D            :returnVal = Composite2DLib; break;
                case AotStateMachine           :returnVal = StateMachineLib; break;
                case AotComposite              :returnVal = CompositeLib; break;
                case AotTextStyle              :returnVal = TextStyleLib; break;
                case AotTheme                  :returnVal = ThemeLib; break;
                case AotTransition             :returnVal = TransitionLib; break;
                case AotVertexBuffer           :returnVal = VertexBufferLib; break;
                case AotAppearanceCollection   :returnVal = AppearanceCollectionLib; break;
                case AotAppearance             :returnVal = AppearanceLib; break;
                case AotMaterial               :returnVal = MaterialLib; break;
                case AotRenderMode             :returnVal = RenderModeLib; break;
                case AotUniformSetter          :returnVal = ShaderParamSetterLib; break;
                case AotTexture                :returnVal = TextureLib; break;
                case AotLanguage               :returnVal = LanguagePackLib; break;
                case AotScript                 :returnVal = ScriptLib; break;
                default                        :returnVal = c_libCount; break;
            }
            return returnVal;
        }

        AssetSet::AssetSet(AssetRepository& assetRepository)
            :m_assetRepository(assetRepository),
            m_dataOffset(0U),
            m_libHeaderResourceHandles(),
            m_id(FeatStd::Internal::Limits<UInt16>::Max())
            FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4351, All handles will be zero initialized)
        {

        }

        SharedPointer<AssetSet> AssetSet::Create(AssetRepository& assetRepository)
        {
            return SharedPointer(ASSETLOADER_TRANSIENT_NEW(AssetSet)(assetRepository));
        }

        AssetSet::~AssetSet()
        {
            static_cast<void>(m_assetRepository.Finalize());
        }

        inline UInt16 FindUniqueId(UInt16 candidateId, FeatStd::Internal::Vector<UInt16>& reservedIds)
        {
            UInt16 result = candidateId;
            while (reservedIds.Contains(result)) {
                if (0xFFFFU == result) {
                    result = 0U; // Overflow is intended, continue id lookup from 0
                }
                else {
                    result++;
                }
            }
            return result;
        }

        bool AssetSet::Initialize(UInt32 validationFlags, FeatStd::Internal::Vector<UInt16>& reservedIds)
        {
            if (!m_assetRepository.Initialize()) {
                FEATSTD_LOG_ERROR("Initialization failed, asset repository could not be opened.");
                return false;
            }
            UInt8 header[CFFReader::CFF_ASSET_HEADER_SIZE];
            if (!m_assetRepository.AssetSeek(0, AssetRepository::Begin)) {
                FEATSTD_LOG_ERROR("Initialization failed, asset header could not be read.");
                return false;
            }

            if (1 != m_assetRepository.AssetRead(header, CFFReader::CFF_ASSET_HEADER_SIZE, 1)) {
                FEATSTD_LOG_ERROR("Initialization failed, asset header could not be read.");
                return false;
            }

            ResourceDataHandle resourceHandle = { { { header, 0 } }, CFFReader::CFF_ASSET_HEADER_SIZE, false, false };

            AssetDataHandle headerDataHandle(resourceHandle);
            if (!headerDataHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Initialization failed, asset header could not be read.");
                return false;
            }

            UInt16 candidateId = static_cast<UInt16>(CFFReader::GetAssetPartitionIndex(headerDataHandle));
            m_id = FindUniqueId(candidateId, reservedIds);

            AssetData assetData;
            if (!InitializeAssetVersionInfo(headerDataHandle, assetData.m_versionInfo)) {
                FEATSTD_LOG_ERROR("InitializeAssetVersionInfo failed.");
                return false;
            }
            if (!AssetValidation::ValidateAssetVersion(assetData, validationFlags)) {
                FEATSTD_LOG_ERROR("AssetSet validation failed.");
                return false;
            }

            m_nameTableHandle = AssetDataHandle(CreateAssetHandle(CFFReader::GetAssetNametableOffset(headerDataHandle)));
            if (!m_nameTableHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to create AssetSet.");
                return false;
            }

            AssetDataHandle librariesHeaderHandle(CreateAssetHandle(CFFReader::CFF_ASSET_LIBINFO_COUNT_OFFSET, -1, CFFReader::CFF_ASSET_LIBINFO_SIZE));
            if (!librariesHeaderHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Initialization failed, asset header could not be read.");
                return false;
            }

            Int libCount = CFFReader::GetAssetLibInfoCount(librariesHeaderHandle);
            if (libCount == 0) {
                return true;
            }
            m_dataOffset = CFFReader::GetAssetLibInfoDataOffset(librariesHeaderHandle, 0U);

            for (Int index = 0; index < libCount; index++) {
                AssetLib assetLibType = GetAssetLibType(static_cast<AssetObjectType>(CFFReader::GetAssetLibInfoType(librariesHeaderHandle, index)));
                if (assetLibType >= c_libCount) {
                    return false;
                }
                m_libHeaderResourceHandles[assetLibType] = CreateAssetHandle(CFFReader::GetAssetLibInfoHeaderOffset(librariesHeaderHandle, index));
            }

            return true;
        }

        ResourceDataHandle AssetSet::CreateAssetHandle(OffsetType offset /*= Candera::Internal::InvalidVal*/, Int32 count, Int32 elementSize)
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker criticalSectionLocker(&m_assetRepository.m_seekAndReadCriticalSection);
#endif
            ResourceDataHandle result(ResourceDataHandle::InvalidHandle());
            if (FeatStd::Internal::NumericConversion<UInt32>(offset) != CFFReader::InvalidVal) {
                if (!m_assetRepository.AssetSeek(static_cast<Int32>(offset), AssetRepository::Begin)) {
                    FEATSTD_LOG_ERROR("Asset seek error.");
                    return result;
                }
            }
            else {
                offset = m_assetRepository.AssetTell();
            }
            if (count == -1) {
                SizeType bytesRead = m_assetRepository.AssetRead(&count, SizeType(1), CFF_INT_SIZE);
                if (bytesRead != CFF_INT_SIZE) {
                    FEATSTD_LOG_ERROR("Asset read error.");
                    return result;
                }
                if (!m_assetRepository.AssetSeek(- static_cast<Int32>(CFF_INT_SIZE), AssetRepository::Current)) {
                    FEATSTD_LOG_ERROR("Asset seek error.");
                    return result;
                }

                if (elementSize != 1) {
                    count = count * elementSize + 1;
                }
            }

            ResourceDataHandle resourceHandle = {{{0, 0}}, static_cast<UInt32>(count), false, true};
            resourceHandle.m_accessor.m_asset.m_offset = offset;
            resourceHandle.m_accessor.m_asset.m_regionIndex = NOT_A_REGION;
            resourceHandle.m_accessor.m_asset.m_repositoryId = m_id;

            return resourceHandle;
        }

        SizeType AssetSet::ReadData(void* buffer, OffsetType offset, SizeType count)
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker criticalSectionLocker(&m_assetRepository.m_seekAndReadCriticalSection);
#endif

            if (buffer == 0) {
                return 0;
            }

            if (!m_assetRepository.AssetSeek(offset, AssetRepository::Begin)) {
                return 0;
            }

            return m_assetRepository.AssetRead(buffer, 1ULL, count);
        }

        const void* AssetSet::GetAddress(OffsetType offset, SizeType count) const
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker criticalSectionLocker(&m_assetRepository.m_seekAndReadCriticalSection);
#endif
            const void* result = 0;

            if (!m_assetRepository.AssetSeek(offset, AssetRepository::Begin)) {
                return 0;
            }

            if (count == m_assetRepository.GetConstData(&result, static_cast<SizeType>(1), count)) {
                return result;
            }
            return 0;
        }

        bool AssetSet::InitializeAssetVersionInfo(const AssetDataHandle& headerDataHandle, AssetData::AssetVersionInfo& versionInfo)
        {
            if (!headerDataHandle.IsValid()) {
                return false;
            }
            versionInfo.m_fileVersion = CFFReader::GetAssetFileVersion(headerDataHandle);
            versionInfo.m_fileSize = CFFReader::GetAssetFileSize(headerDataHandle);
            versionInfo.m_fileTimestamp = CFFReader::GetAssetFileTimeStamp(headerDataHandle);
            FeatStd::Internal::Memory::Copy(versionInfo.m_solutionGuid, CFFReader::GetAssetSolutionGUID(headerDataHandle), 16);
            versionInfo.m_customId = CFFReader::GetAssetCustomId(headerDataHandle);
            FeatStd::Internal::Memory::Copy(versionInfo.m_sceneComposerVersion, CFFReader::GetAssetSceneComposerVersion(headerDataHandle), 4 * CFF_INT_SIZE);
            FeatStd::Internal::Memory::Copy(versionInfo.m_canderaVersion, CFFReader::GetAssetCanderaVersion(headerDataHandle), 4 * CFF_INT_SIZE);
            versionInfo.m_widgetSetVersion = CFFReader::GetAssetWidgetSetVersion(headerDataHandle);
            versionInfo.m_widgetHash = CFFReader::GetAssetWidgetHash(headerDataHandle);
            return true;
        }

        ResourceDataHandle AssetSet::CreateAssetItemHeaderHandle(const AssetId& id)
        {
            AssetDataHandle elementHandle = GetAssetLibElementHandle(id);
            if (!elementHandle.IsValid()) {
                return ResourceDataHandle::InvalidHandle();
            }

            return CreateAssetItemHeaderHandle(static_cast<OffsetType>(m_dataOffset + CFFReader::GetLibElementDataOffset(elementHandle) + CFF_INT_SIZE));
        }

        ResourceDataHandle AssetSet::CreateAssetItemHeaderHandle(OffsetType offset)
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker criticalSectionLocker(&m_assetRepository.m_seekAndReadCriticalSection);
#endif

            //TODO read header size from assetlib element header.
            if (!m_assetRepository.AssetSeek(offset, AssetRepository::Begin)) {
                FEATSTD_LOG_ERROR("Asset seek error!");
                return ResourceDataHandle::InvalidHandle();
            }

            Int32 count[2] = {0, 0};
            if (2 != m_assetRepository.AssetRead(&count, CFF_INT_SIZE, SizeType(2))) {
                FEATSTD_LOG_ERROR("Asset read error!");
                return ResourceDataHandle::InvalidHandle();
            }

            if (count[0] != 0) {
                if (!m_assetRepository.AssetSeek(static_cast<OffsetType>(count[0] * static_cast<Int32>(CFFReader::CFF_REGION_SIZE) - static_cast<Int32>(CFF_INT_SIZE)), AssetRepository::Current)) {
                    FEATSTD_LOG_ERROR("Asset seek error!");
                    return ResourceDataHandle::InvalidHandle();
                }

                if (1 != m_assetRepository.AssetRead(&count[1], CFF_INT_SIZE, SizeType(1))) {
                    FEATSTD_LOG_ERROR("Asset read error!");
                    return ResourceDataHandle::InvalidHandle();
                }
            }
#if 1 //Extensive overflow handling as required by Covertiy. If an overflow is experienced here the reason is most likely a corrupted asset.
            if ((count[0] > (FeatStd::Internal::NativeTypeLimit<Int32>::Max() / static_cast<Int32>(CFFReader::CFF_REGION_SIZE))) || (count[0] < 0)) {
                FEATSTD_LOG_ERROR("Asset read error! Value out of range.");
                return ResourceDataHandle::InvalidHandle();
            }

            if ((count[1] > (FeatStd::Internal::NativeTypeLimit<Int32>::Max() / static_cast<Int32>(CFFReader::CFF_ITEM_SIZE))) || (count[1] < 0)){
                FEATSTD_LOG_ERROR("Asset read error! Value out of range.");
                return ResourceDataHandle::InvalidHandle();
            }

            Int32 a = count[0] * static_cast<Int32>(CFFReader::CFF_REGION_SIZE);
            Int32 b = count[1] * static_cast<Int32>(CFFReader::CFF_ITEM_SIZE);

            //assume a > 0 as CFFReader::CFF_REGION_SIZE is unsigned and count[0] >=0
            if ((a > FeatStd::Internal::NativeTypeLimit<Int32>::Max() - b)){
                FEATSTD_LOG_ERROR("Asset read error! Value out of range.");
                return ResourceDataHandle::InvalidHandle();
        }
            //Both a and b are >= 0. c therefore also needs overflow checking.
            Int32 c = a+b;

            if ((c > (FeatStd::Internal::NativeTypeLimit<Int32>::Max() - (2*static_cast<Int32>(CFF_INT_SIZE))))){
                FEATSTD_LOG_ERROR("Asset read error! Value out of range.");
                return ResourceDataHandle::InvalidHandle();
            }
            return CreateAssetHandle(offset, ( 2 * static_cast<Int32>(CFF_INT_SIZE)) + c);
#else //original code
            return CreateAssetHandle(offset, static_cast<Int32>(CFF_INT_SIZE) + count[0] * static_cast<Int32>(CFFReader::CFF_REGION_SIZE) + static_cast<Int32>(CFF_INT_SIZE) + count[1] * static_cast<Int32>(CFFReader::CFF_ITEM_SIZE));
#endif
        }
        ResourceDataHandle AssetSet::CreateAssetItemDataHandle(const AssetId& id)
        {
            AssetDataHandle elementHandle = GetAssetLibElementHandle(id);
            if (!elementHandle.IsValid()) {
                return ResourceDataHandle::InvalidHandle();
            }

            return CreateAssetHandle(static_cast<OffsetType>(m_dataOffset + CFFReader::GetLibElementDataOffset(elementHandle)), CFFReader::GetLibElementDataSize(elementHandle) - CFFReader::GetLibElementRegionDataSize(elementHandle));
        }

        AssetDataHandle AssetSet::GetAssetLibElementHandle(const AssetId& id) const
        {
            if (id.IsValid()) {
                AssetDataHandle libHeaderDataHandle(GetAssetLibHeaderHandle(static_cast<AssetLib>(id.m_libraryType)));
                Int32 leftIndex = 0;
                Int32 rightIndex = CFFReader::GetLibElementCount(libHeaderDataHandle) - 1;
                Id libraryId = AssetIdFunctions::GetLibraryId(id);
                while (leftIndex <= rightIndex) {
                    Int32 middleIndex = (leftIndex + rightIndex) / 2;
                    AssetDataHandle elementHandle = CFFReader::GetLibElement(libHeaderDataHandle, middleIndex);
                    Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetLibElementId(elementHandle));
                    if (!result.IsValid()) {
                        FEATSTD_LOG_DEBUG("GetAssetId is not valid");
                    }
                    Id currentLibraryId = AssetIdFunctions::GetLibraryId(result);
                    if (libraryId < currentLibraryId) {
                        rightIndex = middleIndex - 1;
                    }
                    else if (libraryId > currentLibraryId) {
                        leftIndex = middleIndex + 1;
                    }
                    else {
                        return elementHandle;
                    }
                }
            }

            return AssetDataHandle();
        }

        const Char* AssetSet::GetName(UInt32 offset) const
        {
            return CFFReader::CffGetPtr<Char>(m_nameTableHandle, offset + CFF_STRINGLENGTH_SIZE);
        }
    }
}
