//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if  !defined(CANDERA_ASSETDESCRIPTOR_H)
#define CANDERA_ASSETDESCRIPTOR_H

#include <Candera/Environment.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetIdIterator.h>

namespace Candera {

    class AssetRepository;

    /** @addtogroup AssetLoaderBase
     *  @{
     */

    /**
     * @brief Descriptor of the loaded asset.
     * 
     * AssetDescriptor provides asset names, version and other information 
     *  stored in the asset.
     */
    class AssetDescriptor {
        friend class DefaultAssetProvider;
        typedef Candera::Internal::InternalAssetIdIterator InternalAssetIdIterator;
        typedef Candera::Internal::InternalRenderTargetIdIterator InternalRenderTargetIdIterator;
    public:
        typedef ForwardIterator<InternalAssetIdIterator> AssetIdIterator;
        typedef ForwardIterator<InternalRenderTargetIdIterator> RenderTargetIdIterator;

        /**
         * @brief Get object count in asset.
         * 
         * @param assetType Type of the asset objects for which the count is requested
         * @param assetRepository Optional repository. If specified only the
         *  objects from the given AssetRepository will be counted.
         * @return object count.
         */
        Int32 GetAssetObjectCount(AssetLib assetType, const AssetRepository* assetRepository = 0) const;

        /**
         * @brief Get an iterator over object AssetIds.
         * 
         * Use the iterator for retrieving ids of objects of a given type from
         *  the asset. Use the AssetProvider interface to retrieve the objects 
         *  with the retrieved AssetId.
         * 
         * For a list of GraphicDeviceUnit objects sorted by their Upload dependency
         *  order use: GetRenderTargetIdIterator.
         * @param assetType Type of the asset objects for which the AssetId
         *  iterator is requested.
         * @param assetRepository Optional repository. If specified only the
         *  AssetIds from the given AssetRepository will be iterated.
         * @return AssetId ForwardIterator.
         */
        AssetIdIterator GetAssetIdIterator(AssetLib assetType, const AssetRepository* assetRepository = 0) const;

        /**
         * @brief Get an iterator over GraphicDeviceUnit AssetIds.
         * 
         * Use the iterator for retrieving ids of GraphicDeviceUnit objects from
         *  the asset. Use the AssetProvider interface to retrieve the objects 
         *  with the retrieved AssetId.
         * The GraphicDeviceUnit AssetIds are sorted by the Upload dependencies
         *  between them.
         * 
         * @return AssetId ForwardIterator.
         */
        RenderTargetIdIterator GetRenderTargetIdIterator() const;

        /**
         * Retrieve the master theme AssetId.
         * 
         * @return Master theme AssetId.
         */
        Id GetMasterThemeAssetId() const;

        /**
         * Retrieve the current culture name.
         *
         * @return Current culture name.
         */
        const Char* GetDefaultCultureName() const;

        /**
         * Retrieve the platform name.
         * 
         * @return Platform name.
         */
        const Char* GetPlatformName() const;

        /**
         * Retrieve the platform instance name.
         * 
         * @return Platform instance name.
         */
        const Char* GetPlatformInstanceName() const;

        /**
         * Retrieve whether the asset was generated for being loaded on target
         *  or not (host).
         * 
         * @return Bool specifying if asset was generated for target.
         */
        bool IsForTarget() const;

        /**
         * Initialize an AssetData::AssetVersionInfo structure with current 
         *  loaded asset version information.
         * 
         * @return Bool specifying if initialization was successful or not.
         */
        bool GetVersionInfo(AssetData::AssetVersionInfo& versionInfo) const;

        /**
         * Retrieve the size of the biggest glyph available in any of the BitmapFont files present in the asset.
         * @return Maximum bitmap glyph size or -1 if the asset does not contain any BitmapFont.
         */
        Int32 GetMaximumBitmapGlyphSize() const;

    private:
        Candera::Internal::AssetGroup::SharedPointer m_assetGroup;

        AssetDescriptor() {}

        FEATSTD_MAKE_CLASS_UNCOPYABLE(AssetDescriptor);
    };

    /**
     * @}
     */

}

#endif //CANDERA_ASSETDESCRIPTOR_H
