//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AssetDescriptor.h"

#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AssetHeaderCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/LibraryHeaderCffReader.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);
    using namespace FeatStd::MemoryManagement;
    using namespace Candera::Internal;

    Id AssetDescriptor::GetMasterThemeAssetId() const
    {
        AssetDataHandle assetHeaderHandle(m_assetGroup->GetAssetHeaderHandle());
        if (!assetHeaderHandle.IsValid()) {
            return AssetIdFunctions::GetLibraryId(AssetId::InvalidId());
        }

        UInt32 themeNameOffset = CFFReader::GetAssetMasterThemeName(assetHeaderHandle);
        for (AssetGroup::AssetSetIterator it = m_assetGroup->GetAssetSetIterator(); it.IsValid(); ++it) {
            AssetDataHandle headerHandle((*it)->GetAssetLibHeaderHandle(ThemeLib));
            if (headerHandle.IsValid()) {
                Int32 elementCount = CFFReader::GetLibElementCount(headerHandle);
                for (Int32 index = 0; index < elementCount; ++index) {
                    AssetDataHandle elementHandle = CFFReader::GetLibElement(headerHandle, index);
                    if (themeNameOffset == CFFReader::GetLibElementName(elementHandle)) {
                        Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetLibElementId(elementHandle));
                        if (!result.IsValid()) {
                            FEATSTD_LOG_DEBUG("Master Theme AssetId is not valid");
                        }
                        return AssetIdFunctions::GetLibraryId(result);
                    }
                }
            }
        }

        return AssetIdFunctions::GetLibraryId(AssetId::InvalidId());
    }

    const Char* AssetDescriptor::GetDefaultCultureName() const
    {
        AssetGroup::AssetSetIterator it = m_assetGroup->GetAssetSetIterator();
        if (!it.IsValid()) {
            return 0;
        }
        return (*it)->GetName(CFFReader::GetAssetDefaultCultureName(AssetDataHandle(m_assetGroup->GetAssetHeaderHandle())));
    }

    const Char* AssetDescriptor::GetPlatformName() const
    {
        AssetGroup::AssetSetIterator it = m_assetGroup->GetAssetSetIterator();
        if (!it.IsValid()) {
            return 0;
        }
        return (*it)->GetName(CFFReader::GetAssetPlatformName(AssetDataHandle(m_assetGroup->GetAssetHeaderHandle())));
    }

    const Char* AssetDescriptor::GetPlatformInstanceName() const
    {
        AssetGroup::AssetSetIterator it = m_assetGroup->GetAssetSetIterator();
        if (!it.IsValid()) {
            return 0;
        }
        return (*it)->GetName(CFFReader::GetAssetPlatformInstanceName(AssetDataHandle(m_assetGroup->GetAssetHeaderHandle())));
    }

    bool AssetDescriptor::IsForTarget() const
    {
        return CFFReader::GetAssetSimulationShaders(AssetDataHandle(m_assetGroup->GetAssetHeaderHandle()));
    }

    bool AssetDescriptor::GetVersionInfo(AssetData::AssetVersionInfo& versionInfo) const
    {
        return m_assetGroup->InitializeAssetVersionInfo(versionInfo);
    }

    Int32 AssetDescriptor::GetMaximumBitmapGlyphSize() const
    {
        return CFFReader::GetAssetMaximumBitmapGlyphSize(AssetDataHandle(m_assetGroup->GetAssetHeaderHandle()));
    }

    Int32 AssetDescriptor::GetAssetObjectCount(AssetLib assetType, const AssetRepository* assetRepository) const
    {
        Int32 count = 0;
        if (m_assetGroup != 0) {
            for (AssetGroup::AssetSetIterator setIterator = m_assetGroup->GetAssetSetIterator(); setIterator.IsValid(); ++setIterator) {
                if ((assetRepository == 0) || (*setIterator)->IsWrapping(assetRepository)) {
                    AssetDataHandle headerHandle((*setIterator)->GetAssetLibHeaderHandle(assetType));
                    if (headerHandle.IsValid()) {
                        count += CFFReader::GetLibElementCount(headerHandle);
                    }
                }
            }
        }

        return count;
    }

    AssetDescriptor::AssetIdIterator AssetDescriptor::GetAssetIdIterator(AssetLib assetType, const AssetRepository* assetRepository) const
    {
        if (m_assetGroup != 0) {
            if (assetRepository == 0) {
                return AssetIdIterator(
                    InternalAssetIdIterator(assetType, AssetGroup::AssetSetIterator(m_assetGroup->m_assetSetList.Begin(), m_assetGroup->m_assetSetList.End())),
                    InternalAssetIdIterator(assetType, AssetGroup::AssetSetIterator(m_assetGroup->m_assetSetList.End(), m_assetGroup->m_assetSetList.End())));
            }

            for (AssetGroup::AssetSetIterator setIterator = m_assetGroup->GetAssetSetIterator(); setIterator.IsValid(); ++setIterator) {
                if ((*setIterator)->IsWrapping(assetRepository)) {
                    AssetGroup::AssetSetIterator nextIterator = setIterator;
                    ++nextIterator;
                    return AssetIdIterator(InternalAssetIdIterator(assetType, setIterator), InternalAssetIdIterator(assetType, nextIterator));
                }
            }
        }

        return AssetIdIterator();        
    }

    AssetDescriptor::RenderTargetIdIterator AssetDescriptor::GetRenderTargetIdIterator() const
    {   
        if (m_assetGroup != 0) {
            AssetDataHandle devicePackageDescriptorHandle(m_assetGroup->GetDevicePackageMetaInfoHandle());
            if (devicePackageDescriptorHandle.IsValid()) {
                return RenderTargetIdIterator(InternalRenderTargetIdIterator(devicePackageDescriptorHandle), InternalRenderTargetIdIterator());
            }
        }

        return RenderTargetIdIterator();
    }

}
