//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AssetDecompression.h"

#include <FeatStd/Platform/Memory.h>
#include <FeatStd/Util/StaticObject.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Mathematics/Math.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLibHeadInformation.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLoaderMemoryPool.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetCache.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDataHandle.h>

#if defined(CANDERA_ASSET_DECOMPRESSION_ENABLED)
#include <FeatStd/Util/NumericUtil.h>
#include <zlib.h>
#endif

namespace Candera {
    namespace Internal {

#if defined(CANDERA_ASSET_DECOMPRESSION_ENABLED)

        FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);

        static void* DecompressAlloc(void* /*opaque*/, UInt items, UInt size)
        {
            return ASSETLOADER_TRANSIENT_ALLOC(static_cast<SizeType>(items * size));
        }

        static void DecompressFree(void* /*opaque*/, void* ptr)
        {
            ASSETLOADER_FREE(ptr);
        }

        UInt32 AssetDecompressor::UncompressData(const ResourceDataHandle& compressedDataHandle, UInt32 uncompressedSize, void* buffer)
        {
                        using namespace FeatStd::Internal;

            FEATSTD_LINT_SYMBOL(550, buffer, "False positive, buffer variable is assigned")
            FEATSTD_LINT_SYMBOL(818, buffer, "False positive, buffer variable is assigned")
            const void* data = ResourceData::GetAddress(compressedDataHandle);

            z_stream strm;
            strm.zalloc = &DecompressAlloc;
            strm.zfree = &DecompressFree;
            strm.opaque = Z_NULL;
            strm.avail_in = 0;
            strm.next_in = Z_NULL;

            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1924, "Violates MISRA C++ 2008 Required Rule 5-2-4. Third party (zlib) macro")
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(970, "Violates MISRA C++ 2008 Required Rule 3-9-2. Third party (zlib) macro")
            Int result = inflateInit(&strm);
                        if (result != Z_OK)
                        {
                                FEATSTD_LOG_ERROR("In Assetdecompressor, zlib inflateInif failed with RC: %d", result);
                                return 0;
                        }

            strm.next_out = FeatStd::Internal::PointerToPointer<Bytef*>(buffer);
            strm.avail_out = uncompressedSize;

            void* chunk = 0;
            UInt32 chunkSize =
#ifdef CANDERA_ASSET_DECOMPRESSION_CHUNK_SIZE
                CANDERA_ASSET_DECOMPRESSION_CHUNK_SIZE;
#else
                CANDERA_ASSET_CACHE_SIZE / 10;
#endif
            if (data == 0) {
                FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4309, "NOT_A_REGION fits into the 16 bit bitfield.")
                ResourceDataHandle::Accessor::Asset key = {0, NOT_A_REGION, 0};
                chunk = AssetCache::GetInstance().GetAndRetain(key);
                if (chunk == 0) {
                    chunk = AssetCache::GetInstance().Reserve(key, chunkSize);
                }
            }

            do {
                strm.avail_in = Math::Minimum<UInt32>(compressedDataHandle.m_size - strm.total_in, chunkSize);
                if (strm.avail_in == 0) {
                    break;
                }

                if (data == 0) {
                    strm.next_in = PointerToPointer<UInt8*>(chunk);
                    static_cast<void>(NumericConversion<UInt>(ResourceData::CopyData(strm.next_in, compressedDataHandle, strm.total_in, strm.avail_in))); //result is compared to UInt defined in 3rd party code (zLib)
                }
                else {
                    strm.next_in = const_cast<UInt8*>(PointerAdd<const UInt8*>((data), strm.total_in));
                }

                result = inflate(&strm, Z_NO_FLUSH);

                // Z_STREAM_ERROR marks an zlib usage error (the caller handles z_stream)
                FEATSTD_DEBUG_ASSERT(result != Z_STREAM_ERROR); // state not clobbered
                // Log error cases
                switch (result)
                {
                    case Z_STREAM_ERROR:
                        FEATSTD_LOG_ERROR("AssetDecompression failed with internal error (RC: %d)", result);
                        break;
                    case Z_NEED_DICT:
                    case Z_DATA_ERROR:
                    case Z_MEM_ERROR:
                        FEATSTD_LOG_ERROR("AssetDecompression zlib failed with RC: %d", result);
                        break;
                    default:
                        break;
                }

                if ((result != Z_OK) && (result != Z_STREAM_END))
                {
                    break;
                }
            } while (result != Z_STREAM_END);   // done when inflate() says it's done

            result = (result == Z_STREAM_END) ? Z_OK : Z_DATA_ERROR;

            static_cast<void>(inflateEnd(&strm));     // clean up

            if ((result == Z_OK) && (uncompressedSize != strm.total_out))
            {
                FEATSTD_LOG_WARN(
                        "AssetDecompression uncompressed data size mismatch with total output reported by zlib (%d != %d)",
                        uncompressedSize, strm.total_out
                        );
            }

            if (data == 0) {
                AssetCache::GetInstance().Release(chunk);
            }

            // Return uncompressed data size
            return strm.total_out;
        }
#else
        UInt32 AssetDecompressor::UncompressData(const ResourceDataHandle&, UInt32, void*)
        {
            FEATSTD_DEBUG_FAIL(); // CFF asset reader failed to load compressed item! CANDERA_ASSET_DECOMPRESSION_ENABLED is not set.
            return 0;
        }
#endif

    }
}
