//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "StyleAssetBuilder.h"
#include <Candera/TextEngine/Font.h>
#include <Candera/TextEngine/Types.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/TextStyleCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/TextStyleEntryCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/TextStyleCultureCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/FontInfoCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLoaderBaseDataTypes.h>

namespace Candera {

    using namespace TextRendering;
    using namespace MemoryManagement;
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);

    namespace Internal {

        bool AssetReaderBase<Candera::TextRendering::SharedStyle>::ReadFirstPass(Candera::TextRendering::SharedStyle& style, LoaderContext& context)
        {
            const AssetDataHandle& defaultFontHandle = CFFReader::GetTextStyleDefaultFont(context.handle);
            bool result = true;
            if (defaultFontHandle.IsValid()) {
                Candera::TextRendering::Font font = ReadFont(defaultFontHandle, context);
                style.SetDefaultFont(font);
            }

            Int32 styleEntryCount = CFFReader::GetTextStyleChildrenCount(context.handle);
            for (Int32 styleEntryIndex = 0; styleEntryIndex < styleEntryCount; ++styleEntryIndex) {
                const AssetDataHandle& styleEntryHandle = CFFReader::GetTextStyleChildrenElementAt(context.handle, styleEntryIndex);
                if (!styleEntryHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read style entry of Style " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }

                const AssetDataHandle& fontHandle = CFFReader::GetTextStyleEntryFont(styleEntryHandle);
                if (!fontHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read font of Style " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }

                Candera::TextRendering::Font font = ReadFont(fontHandle, context);
                if (font.IsValid()) {
                    result = style.AddStyleEntry(0, styleEntryIndex, font, static_cast<Utf32>(CFFReader::GetTextStyleEntryLowerBound(styleEntryHandle)), static_cast<Utf32>(CFFReader::GetTextStyleEntryUpperBound(styleEntryHandle)));
                    if (!result) {
                        FEATSTD_LOG_WARN("Failed to add style entry in Style " AssetIdLogStr, AssetIdLogArgs(context.id));
                    }
                }
            }


            Int32 cultureEntryCount = CFFReader::GetTextStyleCulturesCount(context.handle);
            for (Int32 cultureEntryIndex = 0; cultureEntryIndex < cultureEntryCount; ++cultureEntryIndex) {
                const AssetDataHandle& cultureEntryHandle = CFFReader::GetTextStyleCulturesElementAt(context.handle, cultureEntryIndex);
                if (!cultureEntryHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read culture entry of Style " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }
                const Char* cultureName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetTextStyleCultureCultureLocaleName(cultureEntryHandle));
                if (cultureName == 0) {
                    FEATSTD_LOG_ERROR("Failed to read culture name of Style " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }
                Int32 cultureChildrenCount = CFFReader::GetTextStyleCultureChildrenCount(cultureEntryHandle);
                for (Int32 cultureChildIndex = 0; cultureChildIndex < cultureChildrenCount; ++cultureChildIndex) {
                    const AssetDataHandle& cultureChildHandle = CFFReader::GetTextStyleCultureChildrenElementAt(cultureEntryHandle, cultureChildIndex);
                    if (!cultureChildHandle.IsValid()) {
                        FEATSTD_LOG_ERROR("Failed to read culture child entry of Style " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }

                    const AssetDataHandle& cultureFontHandle = CFFReader::GetTextStyleEntryFont(cultureChildHandle);
                    if (!cultureFontHandle.IsValid()) {
                        FEATSTD_LOG_ERROR("Failed to read culture font of Style " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }

                    Candera::TextRendering::Font font = ReadFont(cultureFontHandle, context);
                    if (font.IsValid()) {
                        result = style.AddStyleEntry(cultureName, cultureChildIndex, font, static_cast<Utf32>(CFFReader::GetTextStyleEntryLowerBound(cultureChildHandle)), static_cast<Utf32>(CFFReader::GetTextStyleEntryUpperBound(cultureChildHandle)));
                        if (!result) {
                            FEATSTD_LOG_WARN("Failed to add culture style entry in Style " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }
                    }
                }
            }

            if (!result) {
                FEATSTD_LOG_WARN("Failed to intialize Style " AssetIdLogStr, AssetIdLogArgs(context.id));
            }

            return result;
        }

        bool AssetReaderBase<Candera::TextRendering::SharedStyle>::ReadSecondPass(Candera::TextRendering::SharedStyle& style, LoaderContext& context)
        {
            bool result = true;
            AssetId baseStyleAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetTextStyleBaseStyle(context.handle));
            if (baseStyleAssetId.IsValid()) {
                MemoryManagement::SharedPointer<TextRendering::SharedStyle> textStyle = context.provider->GetTextStyleByAssetId(baseStyleAssetId);
                if (textStyle.PointsToNull()){
                    FEATSTD_LOG_WARN("GetTextStyleByAssetId returns a null pointer.");
                }
                result = style.SetBaseStyle(textStyle);
            }
            else {
                FEATSTD_LOG_DEBUG("GetAssetId is not valid");
            }
            return result;
        }

        Candera::TextRendering::Font AssetReaderBase<Candera::TextRendering::SharedStyle>::ReadFont(const AssetDataHandle& fontHandle, LoaderContext& context)
        {
            Candera::Internal::AssetId resultAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetFontInfoFont(fontHandle));
            if (!resultAssetId.IsValid()) {
                FEATSTD_LOG_DEBUG("GetAssetId is not valid");
            }
            const Char* fontName = context.provider->GetNameByAssetId(resultAssetId);
            if (fontName == 0) {
                FEATSTD_LOG_DEBUG("GetNameByAssetId returns 0.");
            }
            Float x;
            Float y;
            CFFReader::GetFontInfoScale(fontHandle, x, y);
            FontSize fontSize(CFFReader::GetFontInfoSize(fontHandle), x, y);
            UInt8 fontFace = CFFReader::GetFontInfoFaceIndex(fontHandle);

            Candera::TextRendering::Font font;
            if (!font.Setup(fontName, fontSize, fontFace)) {
                FEATSTD_LOG_WARN("Failed Setup font %s of Style " AssetIdLogStr, fontName, AssetIdLogArgs(context.id));
            }
            return font;
        }

        MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> AssetBuilderBase<MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> >::Create(LoaderContext& /*context*/)
        {
            return Candera::TextRendering::SharedStyle::Create();
        }
    }
}
