//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "StateMachineAssetBuilder.h"

#include <CanderaAssetLoader/AssetLoaderBase/CffReader/StateMachineCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/BaseStateCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/TransitionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/WidgetBaseAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoaderBase/StateMachineData.h>

#include <CanderaBehavior/BehaviorBase/ConditionBehavior.h>
#include <CanderaBehavior/BehaviorBase/ActionBehavior.h>

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);

    namespace Internal {

        namespace StateType {
            enum Enum
            {
                State,
                StateMachine,
                HistoryStateMachine,

                Count
            };
        }
  
        static bool IsStateMachine(StateType::Enum stateType)
        {
            static const bool isStateMachine[StateType::Count] = {
                false,   //State,
                true,    //StateMachine,
                true     //HistoryStateMachine,
            };
            FEATSTD_COMPILETIME_ASSERT((sizeof(isStateMachine) / sizeof(bool)) == StateType::Count);
            FEATSTD_DEBUG_ASSERT(stateType < StateType::Count);

            return (stateType < StateType::Count) ? isStateMachine[stateType] : false;
        }

        bool AssetReaderBase<StateMachineBehaviorData>::ReadFirstPass(StateMachineBehaviorData& stateMachine, LoaderContext& context)
        {
            if (!context.handle.IsValid()) {
                FEATSTD_LOG_ERROR("Asset error, missing handle for StateMachineBehaviorData.");
                return false;
            }

            //Build ConditionBehavior list
            Int32 conditionCount = CFFReader::GetStateMachineConditionsCount(context.handle);
            if (!stateMachine.m_conditions.Reserve(conditionCount)) {
                FEATSTD_LOG_ERROR("Cannot allocate memory for conditions.");
                return false;
            }
            for (Int32 conditionIndex = 0; conditionIndex < conditionCount; ++conditionIndex) {
                const AssetDataHandle& conditionHandle = CFFReader::GetStateMachineConditionsElementAt(context.handle, conditionIndex);
                if (!conditionHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Asset error, missing handle for condition.");
                    return false;
                }

                CffLoaderContext conditionContext = context.Clone(conditionHandle);
                ConditionBehavior* condition = Dynamic_Cast<ConditionBehavior*>(AssetBuilder<WidgetBase*>::CreateAndBuildFirstPass(conditionContext));
                if (condition != 0) {
                    static_cast<void>(stateMachine.m_conditions.Add(condition));
                }
                else {
                    //FEATSTD_LOG_WARN("Widget failed to create for scene %s.", sceneContext.GetScene()->GetName());
                }
            }

            //Build ActionBehavior list
            SizeType actionCount = static_cast<SizeType>(CFFReader::GetStateMachineActionsCount(context.handle));
            if (!stateMachine.m_actions.Reserve(actionCount)) {
                FEATSTD_LOG_ERROR("Cannot allocate memory for actions.");
                return false;
            }
            for (SizeType actionIndex = 0; actionIndex < actionCount; ++actionIndex) {
                const AssetDataHandle& actionHandle = CFFReader::GetStateMachineActionsElementAt(context.handle, static_cast<Int32>(actionIndex));
                if (!actionHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Asset error, missing handle for action.");
                    return false;
                }

                CffLoaderContext actionContext = context.Clone(actionHandle);
                ActionBehavior* action = Dynamic_Cast<ActionBehavior*>(AssetBuilder<WidgetBase*>::CreateAndBuildFirstPass(actionContext));
                if (action != 0) {
                    static_cast<void>(stateMachine.m_actions.Add(action));
                }
                else {
                    //FEATSTD_LOG_WARN("Widget failed to create for scene %s.", sceneContext.GetScene()->GetName());
                }
            }

            //Build States
            Int32 stateCount = CFFReader::GetStateMachineStatesCount(context.handle);
            if (!stateMachine.m_states.Reserve(static_cast<SizeType>(stateCount+1))) {
                FEATSTD_LOG_ERROR("Cannot allocate memory for states.");
                return false;
            }

            {
                StateMachineBehaviorActionCall entryActionCall(stateMachine, 0, 0);
                StateMachineBehaviorActionCall exitActionCall(stateMachine, 0, 0);
                StateMachineBehaviorActionCall updateActionCall(stateMachine, 0, 0); 
                const Char* name = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetCanderaObjectCanderaName(context.handle));
                StateMachineBehaviorData::State* state = FEATSTD_NEW(StateMachineBehaviorData::StateMachine)(name, entryActionCall, exitActionCall, updateActionCall);
                if (0 != state) {
                    static_cast<void>(stateMachine.m_states.Add(state));
                }
            }

            SizeType currentActionIndex = 0;
            for (Int32 stateIndex = 0; stateIndex < stateCount; ++stateIndex) {
                StateMachineBehaviorData::State* state = 0;
                const AssetDataHandle& stateHandle = CFFReader::GetStateMachineStatesElementAt(context.handle, stateIndex);
                if (!stateHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Asset error, missing handle for state.");
                    return false;
                }

                const Char* name = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetBaseStateStateName(stateHandle));
                StateType::Enum type = static_cast<StateType::Enum>(CFFReader::GetBaseStateStateType(stateHandle));
                SizeType numberOfEntryActions = CFFReader::GetBaseStateNoEntryActions(stateHandle);
                SizeType numberOfExitActions = CFFReader::GetBaseStateNoExitActions(stateHandle);

                StateMachineBehaviorActionCall entryActionCall(stateMachine, currentActionIndex, currentActionIndex + numberOfEntryActions);
                currentActionIndex += numberOfEntryActions;
                StateMachineBehaviorActionCall exitActionCall(stateMachine, currentActionIndex, currentActionIndex + numberOfExitActions);
                currentActionIndex += numberOfExitActions;
                StateMachineBehaviorActionCall updateActionCall(stateMachine, 0, 0);

                if (currentActionIndex > actionCount) {
                    FEATSTD_LOG_ERROR("Asset error, cummulated state acction count is greater than the actual action count.");
                    return false;
                }
                switch (type) {
                    case StateType::State:
                        state = FEATSTD_NEW(StateMachineBehaviorData::State)(name, entryActionCall, exitActionCall, updateActionCall);
                        break;
                    case StateType::StateMachine:
                        state = FEATSTD_NEW(StateMachineBehaviorData::StateMachine)(name, entryActionCall, exitActionCall, updateActionCall);
                        break;
                    case StateType::HistoryStateMachine:
                        state = FEATSTD_NEW(StateMachineBehaviorData::HistoryStateMachine)(name, entryActionCall, exitActionCall, updateActionCall);
                        break;
                    default:
                        break;
                }

                if (0 != state) {
                    static_cast<void>(stateMachine.m_states.Add(state));
                }
            }

            Int32 numberOfSubStates = CFFReader::GetStateMachineNoParentStates(context.handle);
            if (numberOfSubStates > 0) {
                static_cast<StateMachineBehaviorData::StateMachine*>(stateMachine.m_states[0])->InitStates(*stateMachine.m_states[1], numberOfSubStates, &stateMachine.m_states[1]);
            }
            
            //Initialize States
            Int32 statePosition = 1 + numberOfSubStates;
            for (Int32 stateIndex = 0; stateIndex < stateCount; ++stateIndex) {
                if (statePosition > (stateCount + 1)) {
                    FEATSTD_LOG_ERROR("Asset error, cummulated substate count is greater than the actual state count.");
                    return false;
                }
                const AssetDataHandle& stateHandle = CFFReader::GetStateMachineStatesElementAt(context.handle, stateIndex);
                numberOfSubStates = CFFReader::GetBaseStateNoSubStates(stateHandle);
                StateType::Enum type = static_cast<StateType::Enum>(CFFReader::GetBaseStateStateType(stateHandle));

                if (numberOfSubStates != 0) {
                    if (IsStateMachine(type)) {
                        static_cast<StateMachineBehaviorData::StateMachine*>(stateMachine.m_states[static_cast<SizeType>(stateIndex + 1)])->InitStates(
                            *stateMachine.m_states[statePosition], static_cast<SizeType>(numberOfSubStates), &stateMachine.m_states[statePosition]);
                        statePosition += numberOfSubStates;
                    }
                    else {
                        FEATSTD_LOG_ERROR("Asset error, only StateMachines can have states.");
                    }
                }
            }

            //Build Transitions
            Int32 stateTransitionStartIndex = 0;
            Int32 numberOfTransitions = CFFReader::GetStateMachineStateTransitionsCount(context.handle);
            if (!stateMachine.m_transitions.Reserve(numberOfTransitions)) {
                FEATSTD_LOG_ERROR("Cannot allocate memory for transitions.");
                return false;
            }
            for (Int32 transitionIndex = 0; transitionIndex < numberOfTransitions; ++transitionIndex) {
                const AssetDataHandle& transitionHandle = CFFReader::GetStateMachineStateTransitionsElementAt(context.handle, transitionIndex);
                if (!transitionHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Asset error, missing handle for state.");
                    return false;
                }

                const Char* name = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetTransitionTransitionName(transitionHandle));
                SizeType numberOfActions = CFFReader::GetTransitionNoActions(transitionHandle);
                Int32 conditionIndex = CFFReader::GetTransitionConditionindex(transitionHandle);
                Int32 targetState = CFFReader::GetTransitionTargetStateIndex(transitionHandle) + 1;

                StateMachineBehaviorConditionEvaluation conditionEvaluation = StateMachineBehaviorConditionEvaluation();
                if (conditionIndex != -1) {
                    conditionEvaluation = StateMachineBehaviorConditionEvaluation(stateMachine.m_conditions[conditionIndex]);
                }

                StateMachineBehaviorActionCall actionCall(stateMachine, currentActionIndex, currentActionIndex + numberOfActions);
                currentActionIndex += numberOfActions;

                if (currentActionIndex > actionCount) {
                    FEATSTD_LOG_ERROR("Asset error, cummulated state acction count is greater than the actual action count.");
                    return false;
                }

                StateMachineBehaviorData::Transition* transition = FEATSTD_NEW(StateMachineBehaviorData::Transition)(
                    name, conditionEvaluation, stateMachine.m_states[targetState], actionCall);
                if (0 != transition) {
                    static_cast<void>(stateMachine.m_transitions.Add(transition));
                }
            }

            for (Int32 stateIndex = 0; stateIndex < stateCount; ++stateIndex) {
                const AssetDataHandle& stateHandle = CFFReader::GetStateMachineStatesElementAt(context.handle, stateIndex);
                Int32 numberOfStateTransitions = CFFReader::GetBaseStateNoTransitions(stateHandle);

                if ((stateTransitionStartIndex + numberOfStateTransitions) > numberOfTransitions) {
                    FEATSTD_LOG_ERROR("Asset error, cummulated state transition count is greater than the actual transition count.");
                    return false;
                }

                stateMachine.m_states[static_cast<SizeType>(stateIndex + 1)]->InitTransitions(numberOfStateTransitions, &stateMachine.m_transitions[stateTransitionStartIndex]);
                stateTransitionStartIndex += numberOfStateTransitions;
            }

            FEATSTD_DEBUG_ASSERT(stateTransitionStartIndex == numberOfTransitions);
            FEATSTD_DEBUG_ASSERT(currentActionIndex == currentActionIndex);
            FEATSTD_DEBUG_ASSERT(statePosition == (stateCount + 1));

            return true;
        }

        bool AssetReaderBase<StateMachineBehaviorData>::ReadSecondPass(StateMachineBehaviorData& stateMachine, LoaderContext& context)
        {
            if (!context.handle.IsValid()) {
                FEATSTD_LOG_ERROR("Asset error, missing handle for StateMachineBehaviorData.");
                return false;
            }

            //Build ConditionBehavior list
            Int32 conditionCount = CFFReader::GetStateMachineConditionsCount(context.handle);
            if (static_cast<SizeType>(conditionCount) != stateMachine.m_conditions.Size()) {
                FEATSTD_LOG_ERROR("Incorrect number of conditions in the StateMachine");
                return false;
            }

            for (Int32 conditionIndex = 0; conditionIndex < conditionCount; ++conditionIndex) {
                const AssetDataHandle& conditionHandle = CFFReader::GetStateMachineConditionsElementAt(context.handle, conditionIndex);
                if (!conditionHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Asset error, missing handle for condition.");
                    return false;
                }

                ConditionBehavior* conditionBehavior = stateMachine.m_conditions[conditionIndex];
                if (conditionBehavior != 0) {
                    CffLoaderContext conditionContext = context.Clone(conditionHandle);
                    if (!AssetBuilder<WidgetBase*>::BuildSecondPass(conditionBehavior, conditionContext)) {
                        FEATSTD_LOG_WARN("Widget %s failed to initialize.", conditionBehavior->GetName());
                    }
                }
            }

            //Build ActionBehavior list
            Int32 actionCount = CFFReader::GetStateMachineActionsCount(context.handle);
            if (static_cast<SizeType>(actionCount) != stateMachine.m_actions.Size()) {
                FEATSTD_LOG_ERROR("Incorrect number of actions in the StateMachine");
                return false;
            }

            for (Int32 actionIndex = 0; actionIndex < actionCount; ++actionIndex) {
                const AssetDataHandle& actionHandle = CFFReader::GetStateMachineActionsElementAt(context.handle, actionIndex);
                if (!actionHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Asset error, missing handle for action.");
                    return false;
                }

                ActionBehavior* actionBehavior = stateMachine.m_actions[actionIndex];
                if (actionBehavior != 0) {
                    CffLoaderContext actionContext = context.Clone(actionHandle);
                    if (!AssetBuilder<WidgetBase*>::BuildSecondPass(actionBehavior, actionContext)) {
                        FEATSTD_LOG_WARN("Widget %s failed to initialize.", actionBehavior->GetName());
                    }
                }
            }

            return true;
        }

        StateMachineBehaviorData* AssetBuilderBase<StateMachineBehaviorData*>::Create(LoaderContext& /*context*/)
        {
            return StateMachineBehaviorData::Create();
        }

        void AssetBuilderBase<StateMachineBehaviorData*>::Dispose(StateMachineBehaviorData* stateMachine)
        {
            if (stateMachine == 0) {
                return;
            }

            Vector<StateMachineBehaviorData::Transition*>& transitions = stateMachine->m_transitions;
            for (SizeType i = 0; i < transitions.Size(); ++i) {
                FEATSTD_DELETE(transitions[i]);
                transitions[i] = 0;
            }
            transitions.Clear();

            Vector<StateMachineBehaviorData::State*>& states = stateMachine->m_states;
            for (SizeType i = 0; i < states.Size(); ++i) {
                FEATSTD_DELETE(states[i]);
                states[i] = 0;
            }
            states.Clear();

            Vector<Candera::ConditionBehavior*>& conditions = stateMachine->m_conditions;
            for (FeatStd::SizeType i = 0; i < conditions.Size(); ++i) {
                AssetBuilder<WidgetBase*>::Dispose(conditions[i]);
                conditions[i] = 0;
            }
            conditions.Clear();

            Vector<Candera::ActionBehavior*>& actions = stateMachine->m_actions;
            for (FeatStd::SizeType i = 0; i < actions.Size(); ++i) {
                AssetBuilder<WidgetBase*>::Dispose(actions[i]);
                actions[i] = 0;
            }
            actions.Clear();
        }
    }
}
