//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AnimationBlendedPropertyListAssetReader.h"
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/EngineBase/Animation/AnimationPlayer.h>
#include <Candera/EngineBase/Animation/AnimationBlendedProperty.h>
#include <Candera/EngineBase/Animation/AnimationKeyframeSequence.h>
#include <Candera/EngineBase/Animation/BezierInterpolationStrategy.h>
#include <Candera/EngineBase/Animation/LinearInterpolationStrategy.h>
#include <Candera/EngineBase/Animation/SplineInterpolationStrategy.h>
#include <Candera/EngineBase/Animation/StepInterpolationStrategy.h>
#include <Candera/EngineBase/Animation/EaseInterpolationStrategy.h>
#include <Candera/EngineBase/Animation/BackEaseFunction.h>
#include <Candera/EngineBase/Animation/BounceEaseFunction.h>
#include <Candera/EngineBase/Animation/ElasticEaseFunction.h>
#include <Candera/EngineBase/Animation/ExponentialEaseFunction.h>
#include <Candera/EngineBase/Animation/PowerEaseFunction.h>
#include <CanderaAssetLoader/AssetLoaderBase/AnimationPropertySetterFactory.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AnimatedPropertyCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/BaseAnimationInterpolationCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AnimationKeyframeSequenceCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/BezierInterpolationCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/EaseInterpolationCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/BackFunctionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/BounceFunctionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/ElasticFunctionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/ExponentialFunctionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AnimationKeyframeSequenceCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/PowerFunctionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/BaseAnimationKeyframeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/FloatAnimationKeyframeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/BoolAnimationKeyframeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/IntAnimationKeyframeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/UIntAnimationKeyframeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/Int16AnimationKeyframeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/UInt16AnimationKeyframeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/SceneAnimationInfoCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/SceneContextBase.h>

namespace Candera { namespace Internal {

    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

    bool AnimationBlendedPropertyListAssetReader::Read(CffLoaderContext& context)
    {
        AnimationPropertySetterFactory::SetAssetProvider(static_cast<AssetProvider*>(&DefaultAssetProvider::GetInstance()));

        Int32 animatedPropertyIndex = 0;
        Int32 channelIndex = -1;

        while (animatedPropertyIndex < GetAnimationPropertyCount(context)) {
            const AssetDataHandle& apHandle = GetAnimationProperty(context, animatedPropertyIndex);
            if (!apHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read AnimatedProperty for Animation " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

            Int32 channelCount = CFFReader::GetAnimatedPropertyChannelsCount(apHandle);

            if (channelIndex < channelCount) {
                Vector<AnimatedPropertyChannels> apcList;
                if (channelIndex < 0) {
                    for (Int32 k = 0; k < channelCount; k++) {
                        static_cast<void>(apcList.Add(static_cast<AnimatedPropertyChannels>(CFFReader::GetAnimatedPropertyChannelsElementAt(apHandle, k))));
                    }
                } else {
                    static_cast<void>(apcList.Add(static_cast<AnimatedPropertyChannels>(CFFReader::GetAnimatedPropertyChannelsElementAt(apHandle, channelIndex))));
                }

                PropertySetterType propertySetterType = static_cast<PropertySetterType>(CFFReader::GetAnimatedPropertyAnimatedPropertyType(apHandle));
                Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetAnimatedPropertyTransformable(apHandle));
                if (!result.IsValid()) {
                    FEATSTD_LOG_DEBUG("Animation property asset id is not valid");
                }
                Animation::AnimationPropertySetter::SharedPointer propertySetter = AnimationPropertySetterFactory::CreateCorrespondingSetter(static_cast<AnimationPropertySetterId>(CFFReader::GetAnimatedPropertyHostItemPropertyId(apHandle)), AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetAnimatedPropertyPropertyName(apHandle)), apcList, result, propertySetterType);
                if (propertySetter == 0) {
                    if (channelIndex < 0) {
                        channelIndex = 0;
                        continue;
                    } else {
                        channelIndex++;
                        continue;
                    }
                } else {
                    channelCount = (channelIndex < 0)?channelCount:1;

                    Animation::AnimationBlendedProperty::SharedPointer animationProperty = Animation::AnimationBlendedProperty::Create();
                    if (animationProperty == 0) {
                        FEATSTD_LOG_ERROR("Failed to create AnimationBlendedProperty for Animation " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }
                    animationProperty->SetAnimationPropertySetter(MemoryManagement::SharedPointer<Animation::AnimationPropertySetter>(propertySetter));

                    //sc mixed up interpolation and key frame sequences...
                    const AssetDataHandle& interpolationHandle = CFFReader::GetAnimatedPropertyInterpolation(apHandle);
                    if (!interpolationHandle.IsValid()) {
                        continue;
                    }

                    Int32 kfSeqCount = CFFReader::GetBaseAnimationInterpolationKeyframesCount(interpolationHandle);
                    for (Int32 kfSeqIndex = 0; kfSeqIndex < kfSeqCount; kfSeqIndex++) {
                        const AssetDataHandle& kfsqHandle = CFFReader::GetBaseAnimationInterpolationKeyframesElementAt(interpolationHandle, kfSeqIndex);
                        if (!kfsqHandle.IsValid()) {
                            FEATSTD_LOG_ERROR("Asset error, missing handle for keyframe sequence for Animation " AssetIdLogStr, AssetIdLogArgs(context.id));
                            return false;
                        }

                        Animation::InterpolationStrategy::SharedPointer strategy;
                        InterpolationStrategyType strategyEnum = static_cast<InterpolationStrategyType>(CFFReader::GetAnimationKeyframeSequenceInterpolationStrategy(kfsqHandle));
                        switch (strategyEnum) {
                            case StepInterpolation: strategy = Animation::StepInterpolationStrategy::Create();
                                break;
                            case LinearInterpolation: strategy = Animation::LinearInterpolationStrategy::Create();
                                break;
                            case SplineInterpolation: strategy = Animation::SplineInterpolationStrategy::Create();
                                break;
                            case BezierInterpolation: {
                                Int32 cpKfsqCount = CFFReader::GetBezierInterpolationControlPointsCount(interpolationHandle);
                                if (kfSeqIndex >= cpKfsqCount) {
                                    break;
                                }

                                Animation::BezierInterpolationStrategy::SharedPointer bezierStrategy = Animation::BezierInterpolationStrategy::Create();
                                if (bezierStrategy != 0) {
                                    MemoryManagement::SharedPointer<Animation::KeyframeSequence> cpSequence = Animation::KeyframeSequence::Create();
                                    if (!cpSequence.PointsToNull()) {
                                        FEATSTD_LINT_CURRENT_SCOPE(446, "Violates MISRA C++ 2008 Required Rule 6-5-3: function does not alter context")
                                        CffLoaderContext cpSeqContext = context.Clone(CFFReader::GetBezierInterpolationControlPointsElementAt(interpolationHandle, kfSeqIndex));
                                        if (ReadKeyframeSequence(cpSequence, cpSeqContext, channelCount, channelIndex)) {
                                            bezierStrategy->SetControlKeyframeSequence(cpSequence);
                                        }
                                    }
                                    else {
                                        FEATSTD_LOG_ERROR("Failed to create KeyframeSequence for Bezier interpolation for Animation " AssetIdLogStr, AssetIdLogArgs(context.id));
                                    }
                                }

                                strategy = bezierStrategy;
                                break;
                                                                        }
                            case EaseInterpolation: {
                                Animation::EaseInterpolationStrategy::SharedPointer easeStrategy = Animation::EaseInterpolationStrategy::Create();

                                easeStrategy->SetEaseDirection(static_cast<Animation::EaseInterpolationStrategy::EaseDirection>(CFFReader::GetEaseInterpolationDirection(interpolationHandle)));

                                Animation::AbstractEasingFunction::SharedPointer easeFunction;

                                EaseFunctionType functionType = static_cast<EaseFunctionType>(CFFReader::GetEaseInterpolationFunctionType(interpolationHandle));
                                const AssetDataHandle& efHandle = CFFReader::GetEaseInterpolationInterpolationFunction(interpolationHandle);
                                if (!efHandle.IsValid()) {
                                    FEATSTD_LOG_ERROR("Asset error, missing handle for ease function for Animation " AssetIdLogStr, AssetIdLogArgs(context.id));
                                    return false;
                                }

                                switch (functionType) {
                                    case EaseFunctionBack: {
                                        Animation::BackEaseFunction::SharedPointer backFunction = Animation::BackEaseFunction::Create();
                                        if (backFunction != 0) {
                                            backFunction->SetAmplitude(CFFReader::GetBackFunctionAmplitude(efHandle));
                                        }
                                        easeFunction = backFunction;
                                        break;
                                    }
                                    case EaseFunctionBounce: {
                                        Animation::BounceEaseFunction::SharedPointer bounceFunction = Animation::BounceEaseFunction::Create();
                                        if (bounceFunction != 0) {
                                            bounceFunction->SetBounceCount(CFFReader::GetBounceFunctionBounceCount(efHandle));
                                            bounceFunction->SetRestitutionCoefficient(CFFReader::GetBounceFunctionRestitutionCoefficient(efHandle));
                                        }
                                        easeFunction = bounceFunction;
                                        break;
                                    }
                                    case EaseFunctionElastic: {
                                        Animation::ElasticEaseFunction::SharedPointer elasticFunction = Animation::ElasticEaseFunction::Create();
                                        if (elasticFunction != 0) {
                                            elasticFunction->SetOscillationCount(CFFReader::GetElasticFunctionOscillationCount(efHandle));
                                            elasticFunction->SetExponent(CFFReader::GetExponentialFunctionExponent(efHandle));
                                        }
                                        easeFunction = elasticFunction;
                                        break;
                                    }
                                    case EaseFunctionExponential: {
                                        Animation::ExponentialEaseFunction::SharedPointer exponentialFunction = Animation::ExponentialEaseFunction::Create();
                                        if (exponentialFunction != 0) {
                                            exponentialFunction->SetExponent(CFFReader::GetExponentialFunctionExponent(efHandle));
                                        }
                                        easeFunction = exponentialFunction;
                                        break;
                                    }
                                    case EaseFunctionPower: {
                                        Animation::PowerEaseFunction::SharedPointer powerFunction = Animation::PowerEaseFunction::Create();
                                        if (powerFunction != 0) {
                                            powerFunction->SetPower(CFFReader::GetPowerFunctionPower(efHandle));
                                        }
                                        easeFunction = powerFunction;
                                        break;
                                    }
                                    default: FEATSTD_LOG_ERROR("Invalid ease function type for Animation " AssetIdLogStr, AssetIdLogArgs(context.id));
                                }

                                easeStrategy->SetEaseFunction(easeFunction);
                                easeStrategy->SetEaseDirection(static_cast<Animation::EaseInterpolationStrategy::EaseDirection>(CFFReader::GetEaseInterpolationDirection(interpolationHandle)));
                                strategy = easeStrategy;
                                break;
                            }
                            default: FEATSTD_LOG_ERROR("Invalid interpolation strategy type for Animation " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }

                        if (strategy == 0) {
                            FEATSTD_LOG_WARN("Could not create interpolation strategy for Animation " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }

                        MemoryManagement::SharedPointer<Animation::AnimationKeyframeSequence> keyframeSequence = Animation::AnimationKeyframeSequence::Create();
                        if (!keyframeSequence.PointsToNull()) {
                            keyframeSequence->SetInterpolationStrategy(strategy);
                            keyframeSequence->SetPeriodic(CFFReader::GetAnimationKeyframeSequenceIsPeriodic(kfsqHandle));

                            CffLoaderContext kfSeqContext = context.Clone(kfsqHandle);
                            if (ReadKeyframeSequence(keyframeSequence, kfSeqContext, channelCount, channelIndex)) {
                                if (!animationProperty->AddKeyframeSequence(keyframeSequence)) {
                                    FEATSTD_LOG_WARN("AddKeyframeSequence failed for Animation " AssetIdLogStr, AssetIdLogArgs(context.id));
                                }
                            }
                        }
                        else {
                            FEATSTD_LOG_WARN("Could not create AnimationKeyframeSequence for Animation " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }
                    }

                    if (m_animation != 0) {
                        if (!m_animation->GetController()->AddProperty(animationProperty)) {
                            FEATSTD_LOG_WARN("Could not add animated property to Animation for Animation " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }
                    }
                    if (!AddAnimationBlendedProperty(animationProperty)) {
                        FEATSTD_LOG_WARN("AddAnimationBlendedProperty failed for Animation " AssetIdLogStr, AssetIdLogArgs(context.id));
                    }
                }

                if (channelIndex < 0) {
                    animatedPropertyIndex++;
                } else {
                    channelIndex++;
                }
            } else {
                animatedPropertyIndex++;
                channelIndex = -1;
            }
        }

        return true;
    }

    bool AnimationBlendedPropertyListAssetReader::ReadKeyframeSequence(Animation::KeyframeSequence::SharedPointer keyframeSequence, const CffLoaderContext& context, Int32 channelCount, Int32 channelIndex) const
    {
        Int32 keyframeCount = CFFReader::GetAnimationKeyframeSequenceKeyframesCount(context.handle);

        Animation::SequenceTimeType* sequenceTimes = FEATSTD_NEW_ARRAY(Animation::SequenceTimeType, static_cast<UInt>(keyframeCount));
        if (sequenceTimes == 0) {
            return false;
        }
        Float* values = FEATSTD_NEW_ARRAY(Float, static_cast<SizeType>(keyframeCount)* static_cast<SizeType>(channelCount));
        if (values == 0) {
            FEATSTD_DELETE_ARRAY(sequenceTimes);
            return false;
        }

        for (Int32 index = 0; index < keyframeCount; index++){
            const AssetDataHandle& keyframeHandle =  CFFReader::GetAnimationKeyframeSequenceKeyframesElementAt(context.handle, index);
            sequenceTimes[index] = CFFReader::GetBaseAnimationKeyframeSequenceTime(keyframeHandle);

            switch(CFFReader::GetBaseAnimationKeyframeValueType(keyframeHandle)) {
                case aktFloat:
                    if (channelIndex < 0) {
                        Int32 valueCount = CFFReader::GetFloatAnimationKeyframeValuesCount(keyframeHandle);
                        for (Int32 i = 0; i < valueCount; i++) {
                            values[(index * valueCount) + i] = CFFReader::GetFloatAnimationKeyframeValuesElementAt(keyframeHandle, i);
                        }
                    } else {
                        values[index] = CFFReader::GetFloatAnimationKeyframeValuesElementAt(keyframeHandle, channelIndex);
                    }
                    break;
                case aktInt:
                    if (channelIndex < 0) {
                        Int32 valueCount = CFFReader::GetIntAnimationKeyframeValuesCount(keyframeHandle);
                        for (Int32 i = 0; i < valueCount; i++) {
                            values[(index * valueCount) + i] = static_cast<Float>(CFFReader::GetIntAnimationKeyframeValuesElementAt(keyframeHandle, i));
                        }
                    } else {
                        values[index] = static_cast<Float>(CFFReader::GetIntAnimationKeyframeValuesElementAt(keyframeHandle, channelIndex));
                    }

                    break;
                case aktBool:
                    if (channelIndex < 0) {
                        Int32 valueCount = CFFReader::GetBoolAnimationKeyframeValuesCount(keyframeHandle);
                        for (Int32 i = 0; i < valueCount; i++) {
                            values[(index * valueCount) + i] = CFFReader::GetBoolAnimationKeyframeValuesElementAt(keyframeHandle, i) ? 1.0F : 0.0F;;
                        }
                    } else {
                        values[index] = CFFReader::GetBoolAnimationKeyframeValuesElementAt(keyframeHandle, channelIndex) ? 1.0F : 0.0F;;
                    }

                    break;
                case aktUInt:
                    if (channelIndex < 0) {
                        Int32 valueCount = CFFReader::GetUIntAnimationKeyframeValuesCount(keyframeHandle);
                        for (Int32 i = 0; i < valueCount; i++) {
                            values[(index * valueCount) + i] = static_cast<Float>(CFFReader::GetUIntAnimationKeyframeValuesElementAt(keyframeHandle, i));
                        }
                    } else {
                        values[index] = static_cast<Float>(CFFReader::GetIntAnimationKeyframeValuesElementAt(keyframeHandle, channelIndex));
                    }
                    break;
                case aktInt16:
                    if (channelIndex < 0) {
                        Int32 valueCount = CFFReader::GetInt16AnimationKeyframeValuesCount(keyframeHandle);
                        for (Int32 i = 0; i < valueCount; i++) {
                            values[(index * valueCount) + i] = static_cast<Float>(CFFReader::GetInt16AnimationKeyframeValuesElementAt(keyframeHandle, i));
                        }
                    }
                    else {
                        values[index] = static_cast<Float>(CFFReader::GetInt16AnimationKeyframeValuesElementAt(keyframeHandle, channelIndex));
                    }

                    break;
                case aktUInt16:
                    if (channelIndex < 0) {
                        Int32 valueCount = CFFReader::GetUInt16AnimationKeyframeValuesCount(keyframeHandle);
                        for (Int32 i = 0; i < valueCount; i++) {
                            values[(index * valueCount) + i] = static_cast<Float>(CFFReader::GetUInt16AnimationKeyframeValuesElementAt(keyframeHandle, i));
                        }
                    }
                    else {
                        values[index] = static_cast<Float>(CFFReader::GetUInt16AnimationKeyframeValuesElementAt(keyframeHandle, channelIndex));
                    }

                    break;
                default: {
                    FEATSTD_LOG_ERROR("Invalid animation values type");
                }
            }
        }

        keyframeSequence->SetKeyframes(channelCount, keyframeCount, sequenceTimes, Animation::KeyframeSequence::TimeTypeDisposer::Dispose, values, Animation::KeyframeSequence::ValuesDisposer::Dispose);

        return true;
    }

    AnimationBlendedPropertyListAssetReader::AnimationBlendedPropertyListAssetReader(AssetId animationId, Animation::AnimationPlayer::SharedPointer animation):
        m_animation(animation), m_animationId(animationId)
    {

    }

    Int32 SceneAnimationBlendedPropertyListAssetReader::GetAnimationPropertyCount(CffLoaderContext& context)
    {
        return CFFReader::GetSceneAnimationInfoAnimatedPropertiesCount(context.handle);
    }

    AssetDataHandle SceneAnimationBlendedPropertyListAssetReader::GetAnimationProperty(CffLoaderContext& context, Int32 index)
    {
        return CFFReader::GetSceneAnimationInfoAnimatedPropertiesElementAt(context.handle, index);
    }

    SceneAnimationBlendedPropertyListAssetReader::SceneAnimationBlendedPropertyListAssetReader(AssetId animationId, Animation::AnimationPlayer::SharedPointer animation):
        AnimationBlendedPropertyListAssetReader(animationId, animation)
    {
    }

    SceneContextAnimationBlendedPropertyListAssetReader::SceneContextAnimationBlendedPropertyListAssetReader(SceneContextBase& sceneContext, AssetId animationId, Animation::AnimationPlayer::SharedPointer animation) :
        SceneAnimationBlendedPropertyListAssetReader(animationId, animation),
        m_sceneContext(sceneContext)
    {
    }

    bool SceneContextAnimationBlendedPropertyListAssetReader::AddAnimationBlendedProperty(const MemoryManagement::SharedPointer<Animation::AnimationBlendedProperty>& property)
    {
        m_sceneContext.AddAnimatedPropertyAttachment(AssetIdFunctions::GetLibraryId(m_animationId), property);
        return true;
    }

    } // namespace Internal
} // namespace Candera
