//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ARRAYPROPERTY_H)
#define CANDERA_ARRAYPROPERTY_H

#include <FeatStd/Util/NumericUtil.h>
#include <FeatStd/Util/StaticObject.h>
#include <Candera/System/Container/Vector.h>

namespace Candera {

    /**
     *  @brief ArrayProperty for generic meta info array properties.
     */
    template <typename T>
    class ArrayProperty {
        public:
            /** Constructor. */
            ArrayProperty();

            /** Destructor */
            ~ArrayProperty();

            /**
             * Retrieve number of values.
             */
            FeatStd::SizeType GetCount() const;

            /**
             * Allocate memory for a number of values.
             */
            void Reserve(FeatStd::SizeType size);

            /**
             * Add a value.
             */
            void Add(const T& value);

            /**
             * Clear values.
             */
            void Clear();

            /**
             *  Gets element at certain position.
             *  @param index Position of requested element.
             *  @return The requested element.
             */
            const T& Get(FeatStd::SizeType index) const;

            /**
             *  Gets element at certain position.
             *  @param index Position of requested element.
             *  @return The requested element.
             */
            T& Get(FeatStd::SizeType index);

            /**
            *  Compares the element at certain position.
            *  @param array
            *  @return The requested element.
            */
            template<typename T1>
            bool operator==(const ArrayProperty<T1>& array) {
                if (GetCount() != array.GetCount()) {
                    return false;
                }
                for (FeatStd::SizeType i = 0; i < GetCount(); ++i) {
                    if (Get(i) != array.Get(i)) {
                        return false;
                    }
                }
                return true;
            }

            template<typename T1>
            bool operator!=(const ArrayProperty<T1>& array) {
                return !operator==(array);
            }

        private:
            Candera::Internal::Vector<T> m_values;
    };

    template<typename T>
    ArrayProperty<T>::ArrayProperty() 
    {
    }

    template<typename T>
    ArrayProperty<T>::~ArrayProperty() 
    {
    }

    template<typename T>
    FeatStd::SizeType ArrayProperty<T>::GetCount() const
    {
        return FeatStd::Internal::NumericConversion<FeatStd::SizeType>(m_values.Size());
    }

    template<typename T>
    void ArrayProperty<T>::Reserve(FeatStd::SizeType size)
    {
        //void cast for static code checker. 
        (void)m_values.Reserve(size);
    }

    template<typename T>
    void ArrayProperty<T>::Add(const T& value)
    {
        (void)m_values.Add(value);
    }

    template<typename T>
    void ArrayProperty<T>::Clear()
    {
        m_values.Clear();
    }

    template<typename T>
    const T& ArrayProperty<T>::Get(FeatStd::SizeType index) const
    {
        if (index >= GetCount()) {
            FEATSTD_UNSYNCED_STATIC_OBJECT(T, s_default);
            return s_default;
        }

        return m_values[index];
    }

    template<typename T>
    T& ArrayProperty<T>::Get(FeatStd::SizeType index)
    {
        if (index >= GetCount()) {
            FEATSTD_UNSYNCED_STATIC_OBJECT(T, s_default);
            return s_default;
        }

        return m_values[index];
    }

}
#endif // CANDERA_ARRAYPROPERTY_H
